# -*- coding: utf-8 -*-

from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__ = 'GPL 3'
__copyright__ = '2011, John Schember <john@nachtimwald.com>'
__docformat__ = 'restructuredtext en'

from calibre.customize import InterfaceActionBase
from calibre.devices.kindle.apnx import APNXBuilder

import argparse
import os

class ActionAPNX(InterfaceActionBase):

    name = 'APNX Generator'
    description = 'Generate Amazon APNX page number file'
    supported_platforms = ['windows', 'osx', 'linux']
    author = 'John Schember'
    version = (1, 3, 0)
    minimum_calibre_version = (7, 0, 0)
    
    actual_plugin = 'calibre_plugins.apnx_generator.apnxaction:APNXAction'
    
    def cli_main(self, args):
        """Handles plugin CLI interface"""
        args = args[1:] # args[0] = PLUGINNAME
        
      
        parser = argparse.ArgumentParser(
                    prog='calibre-debug -r "APNX Generator"',
                    formatter_class=argparse.RawDescriptionHelpFormatter,
                    description='APNX Generator for Kindle devices',
                    epilog='examples: \n   calibre-debug -r "APNX Generator" -- -f test.mobi -p 211 \n   calibre-debug -r "APNX Generator" -- -f test.mobi')
        #parser.add_argument('path to MOBI file for APNX generation')           # positional argument
        parser.add_argument('-f', "--file",
                                help="path to MOBI file for APNX generation",
                                action="store", type=str)
        parser.add_argument('-p', "--pages",
                                help="number of pages, optional",
                                action="store", type=int)
        
        if len(args) == 0:
            parser.print_help()
            print("\n")
            import sys
            return sys.exit()
        
        inputargs = parser.parse_args(args)
        if inputargs.file == None:
            print("Missing filename!")
            print("\n")
            import sys
            return sys.exit()
            
        #print( inputargs.file,"  ", inputargs.pages)
        
        if (inputargs.pages !=None):
            numberofpages = inputargs.pages
        else:
            numberofpages = int(0)
        
        method = "None"
        
        self.create_apnx_file(inputargs.file,self.replace_file_extension(inputargs.file,"apnx"),method, numberofpages)
        
        print("Generated APNX file: "+ self.replace_file_extension(inputargs.file,"apnx"))
        print("Done!")
        
    
    def create_apnx_file(self, mobi_file_path: str, apnx_path: str, method: str, page_count: int):
        self.apnx_builder = APNXBuilder()
        self.apnx_builder.write_apnx(mobi_file_path, apnx_path,page_count)
        
    def replace_file_extension(self, file_path, new_extension):
        base_name, old_extension = os.path.splitext(file_path)
        new_file_path = base_name + "." + new_extension
        return new_file_path