#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys, os, sigil_bs4 as bs4

def generate_toc_for_file(soup, file_name):
    """Generates the table of contents for the headings (h1, h2, h3) in a single file."""
    toc_list = []
    for heading in soup.find_all(['h1', 'h2', 'h3']):
        # Determine the class based on the heading tag
        if heading.name == 'h1':
            toc_class = 'sgc-toc-level-1'
        elif heading.name == 'h2':
            toc_class = 'sgc-toc-level-2'
        elif heading.name == 'h3':
            toc_class = 'sgc-toc-level-3'

        # Ensure each heading has a unique ID attribute
        if not heading.has_attr('id'):
            heading['id'] = f'{heading.name}_{file_name}_{len(toc_list)}'

        # Add the heading to the table of contents as a linked list item
        toc_list.append(f'<li class="{toc_class}"><a href="{file_name}#{heading["id"]}">{heading.text}</a></li>')
    
    return toc_list

def generate_toc_for_files(bk, start_file_index):
    """Generates the table of contents for all files whose names match the first 11 characters of the start file."""
    html_files = bk.getspine()
    toc_list = []

    # Get the filename of the current file and extract the first 11 characters as the prefix
    current_file = html_files[start_file_index]
    prefix = current_file[:11]

    # Iterate through all files starting from the current one
    for i in range(start_file_index, len(html_files)):
        next_file = html_files[i]

        # Stop including files if the first 11 characters do not match the current file's prefix
        if next_file[:11] != prefix:
            break

        # Read the contents of the file
        current_content = bk.readfile(next_file)

        # Parse the HTML content of the file
        soup = bs4.BeautifulSoup(current_content, 'html.parser')

        # Add the headings of the file to the table of contents
        toc_list.extend(generate_toc_for_file(soup, next_file))

    # Generate the HTML structure for the table of contents
    toc_html = '<ul>\n' + '\n'.join(toc_list) + '\n</ul>'
    return toc_html

def run(bk):
    # Get a list of all HTML files in the EPUB
    html_files = bk.getspine()

    # Iterate through each HTML file and look for the <nav class="toc-box"> section
    for current_file_index, current_file in enumerate(html_files):
        # Read the contents of the current file
        current_content = bk.readfile(current_file)
        
        # Parse the HTML content with BeautifulSoup
        soup = bs4.BeautifulSoup(current_content, 'html.parser')

        # Find the <nav class="toc-box"> section
        toc_nav = soup.find('nav', {'class': 'toc-box'})
        if toc_nav:
            # Clear the current content inside the nav tag
            toc_nav.clear()

            # Generate the table of contents for the current and subsequent files
            # that have the same first 11 characters in their filenames
            toc_html = generate_toc_for_files(bk, current_file_index)

            # Insert the generated table of contents into the nav section
            toc_nav.append(bs4.BeautifulSoup(toc_html, 'html.parser'))

            # Update the HTML content of the file
            new_content = str(soup)
            bk.writefile(current_file, new_content)

    # Return success status
    return 0