#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright (c) 2024John Crew

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, for commercial and non-commercial purposes are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor contributors may be used to endorse or promote products derived from this software.

4. The source code, and any derived source code, must be made available to users of this software and any software derived from this source code.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

"""
This plugin will add footnotes to the end of the text in each xhtml section or 
add footnotes to a section at the end of the epub.

It can also put the footnotes back in the body of the text if the footnotes are
saved into each xhtml section

If footnotes are placed in each xhtml section, then the plugin provides the
option for the footnotes to be renumbered for each section
"""

import tkinter as tk				# Essential for custom dialog box using tk. commands
import tkinter.ttk as ttk 	# Essential for ttk. commands
from tkinter import * 				# Essential for root = Tk()
import tkinter
from tkinter import messagebox	   #Essential for messagebox
from tkinter import filedialog	   #Essential for file dialog box
import sys #Needed to force exit
import re
from UpdateChecker import CheckForUpdates, SetUpdateTimes
from GenUtils import centerWindow

class cAdvancedOptionsDlg:
	dlgTwoTop=""
	def __init__(self, parent, prefs):
		self.top = tk.Toplevel(parent)
		self.top.title("Advanced options")	#Title for dialog box
		cAdvancedOptionsDlg.dlgTwoTop=self.top
		self.prefs=prefs
		self.top.bind('<Escape>', self.EscapeClicked)

	def showAdvancedOptionsDlg(self):
		self.DlgFrame = ttk.Frame(self.top, padding="15 15 12 12")
		self.DlgFrame.grid(column=0, row=0, sticky=(N, W, E, S))
		self.DlgFrame.columnconfigure(0, weight=1)
		self.DlgFrame.rowconfigure(0, weight=1)

## Insert widgets

		tk.Label(self.DlgFrame, text="ADVANCED OPTIONS", fg='Blue', font='bold').grid(row=0, column=0, sticky=W, pady=4) 

		self.label = ttk.Label(self.DlgFrame, text="Enter name for xhtml footnote section").grid(row=10, column=0, sticky=W, pady=4) 
		self.entryXhtlmSectionName = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryXhtlmSectionName.grid(row=10, column=1, sticky="W")
		self.entryXhtlmSectionName.delete(0,END)
		
		self.label = ttk.Label(self.DlgFrame, text="Enter heading for footnotes").grid(row=15, column=0, sticky=W, pady=4) 
		self.entryFootnoteHeading = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryFootnoteHeading.grid(row=15, column=1, sticky="W")
		self.entryFootnoteHeading.delete(0,END)	
		
		self.label = ttk.Label(self.DlgFrame, text="Enter class name for footnote heading").grid(row=20, column=0, sticky=W, pady=4) 
		self.entryFootnoteHeadingClass = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryFootnoteHeadingClass.grid(row=20, column=1, sticky="W")
		self.entryFootnoteHeadingClass.delete(0,END)

		self.label = ttk.Label(self.DlgFrame, text="Enter ID for footnote").grid(row=23, column=0, sticky=W, pady=4) 
		self.entryFootnoteID = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryFootnoteID.grid(row=23, column=1, sticky="W")
		self.entryFootnoteID.delete(0,END)
		
		self.label = ttk.Label(self.DlgFrame, text="Enter class name for footnote").grid(row=25, column=0, sticky=W, pady=4) 
		self.entryFootnoteClass = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryFootnoteClass.grid(row=25, column=1, sticky="W")
		self.entryFootnoteClass.delete(0,END)
		
		self.label = ttk.Label(self.DlgFrame, text="Enter ID for hyperlink").grid(row=30, column=0, sticky=W, pady=4) 
		self.entryHyperlinkID = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryHyperlinkID.grid(row=30, column=1, sticky="W")
		self.entryHyperlinkID.delete(0,END)
		
		self.label = ttk.Label(self.DlgFrame, text="Enter name for highlight class").grid(row=35, column=0, sticky=W, pady=4) 
		self.entryHighlightClass = ttk.Entry(self.DlgFrame, exportselection=0, width=20)
		self.entryHighlightClass.grid(row=35, column=1, sticky="W")
		self.entryHighlightClass.delete(0,END)
		
		self.isUseEPub3Types = IntVar()  #This determines whether to include an autonumber in the footnote
		self.chkEPub3Types = ttk.Checkbutton(self.DlgFrame, text="Include epub3 types?", variable=self.isUseEPub3Types)
		self.chkEPub3Types.grid(row=40, column=0,  sticky=W)   
				
		tk.Button(self.DlgFrame, text="Save", command=self.SaveSettings, width = 20).grid(row=70, column = 0, sticky=W)
		tk.Button(self.DlgFrame, text="Cancel", command=self.Cancel, width = 20).grid(row=70, column = 1, sticky=W)
		
		centerWindow(self.top) #This must be called after widgets have been set up

##Initialise the state of the widgets	 
		self.entryXhtlmSectionName.insert(0, self.prefs['xhtmlFootnoteSectionName'])
		self.entryFootnoteHeading.insert(0, self.prefs['FootnoteHeading'])
		self.entryFootnoteHeadingClass.insert(0, self.prefs['FootnoteHeadingClass'])
		self.entryFootnoteID.insert(0, self.prefs['FootnoteID'])
		self.entryFootnoteClass.insert(0, self.prefs['FootnoteClass'])
		self.entryHyperlinkID.insert(0, self.prefs['HyperlinkID'])
		self.entryHighlightClass.insert(0, self.prefs['HighlightClass'])
		self.isUseEPub3Types.set(self.prefs['epub3Types'])
		
	def EscapeClicked(self, event):
		self.Cancel()
		
	def Cancel(self):
		"""
		This method is called when the 'Cancel' button is clicked.
		It is also called via EscapeClicked() when the ESC button is pressed.
		"""
		self.top.destroy()
		
	def SaveSettings(self):
		"""
		This method is called when the 'Save' button is clicked.
		"""
		self.prefs['xhtmlFootnoteSectionName'] = self.entryXhtlmSectionName.get()
		self.prefs['FootnoteHeading'] = self.entryFootnoteHeading.get()
		self.prefs['FootnoteHeadingClass'] = self.entryFootnoteHeadingClass.get()
		self.prefs['FootnoteID'] = self.entryFootnoteID.get()
		self.prefs['FootnoteClass'] = self.entryFootnoteClass.get()
		self.prefs['HyperlinkID'] = self.entryHyperlinkID.get()
		self.prefs['HighlightClass']=self.entryHighlightClass.get()
		self.prefs['epub3Types'] = self.isUseEPub3Types.get()
		
		messagebox.showinfo('INFORMATION','Your options have been saved.')
		
		self.top.destroy()

class footNoteClass:
	dlgTop=""
	CssPath=""  #Path and filename of the css file to load
	def __init__(self, parent, bk, prefs):
		self.top = tk.Toplevel(parent)
		footNoteClass.dlgTop=self.top		 #Needed for calling code to destroy the dialog window
		self.bk1=bk					#initialise internal variables
		self.prefs=prefs
		self.top.bind('<Escape>', self.EscapePressed)
		
	def ShowDialogBox(self):
		"""
		Called by the method run().
		Parent is the window that owns this dialog.
		bk is the parameter from Sigil
		"""
		self.top.title("Footnote plugin for Sigil") #Title for dialog box
		self.dlgframe = ttk.Frame(self.top, padding="15 15 12 12")
		self.dlgframe.grid(column=0, row=0, sticky=(N, W, E, S))
		self.dlgframe.columnconfigure(0, weight=1)
		self.dlgframe.rowconfigure(0, weight=1)

		self.textFootNoteSection='Text/' + self.prefs['xhtmlFootnoteSectionName'] + '.xhtml'#The name of the section where footnotes will be inserted
		self.textCurrentSection=''
		self.allFootnotes='\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n' #The heading for the footnote section
		self.noteNumber=1 #Number for footnotes if placed in separate xhtml files
		self.textSectionXHTML=""
		self.html=""
		self.FNList=[]  #Stores a list of footnotes when restoring them to the body of the text
## Insert widgets

		self.label = tk.Label(self.dlgframe, text="Where should footnotes be placed?", fg='Blue', font='bold').grid(row=20, sticky=W, pady=4)
		
		#Frame for radio buttons and renumber checkbox
		#This allows the first radio button and an entry box to be close to each other
		self.radioFrame = ttk.Frame(self.dlgframe, padding="1 1 1 1")
		self.radioFrame.grid(column=0, row=30, sticky=(N, W, E, S))
		
		self.radioFootNote = StringVar()
		rb = Radiobutton(self.radioFrame, text="Current xhtml section		 ", variable=self.radioFootNote, value="C", command = lambda : self.toggleChkRenumbering() )
		rb.grid(column=0, row=0, sticky=W)
		rb = Radiobutton(self.radioFrame, text="New xhtml section", variable=self.radioFootNote, value="N", command = lambda : self.toggleChkRenumbering() )
		rb.grid(column=0, row=3, sticky=W)
		self.radioFootNote.set("C") # initialise -  essential to prevent both buttons being selected at the start

		self.isRenumberingChecked = IntVar() #This controls renumbering of footnotes if put into existing sections 
		self.chkRenumber = ttk.Checkbutton(self.radioFrame, text="Restart numbering for each xhtml section", variable=self.isRenumberingChecked)
		self.chkRenumber.grid(column=1, row=0, sticky=W)
		
		#### TO BE IMPLEMENTED ####
		# self.isFirstWordLink = IntVar() #This controls the first word of the footnote text to act as a link 
		# self.chkFirstWordLink = ttk.Checkbutton(self.dlgframe, text="Use first word as hyperlink", variable=self.isFirstWordLink)
		# self.chkFirstWordLink.grid(column=0, row=17, sticky=W)	 

		#Frame for entry boxes
		#This allows a label and an entry box to be close to each other
		self.editFrame = ttk.Frame(self.dlgframe, padding="1 1 1 1")
		self.editFrame.grid(column=0, row=70, sticky=(N, W, E, S))
		
		#### HYPERLINK SECTION ####
		tk.Label(self.editFrame, text="Hyperlinks in the main text", fg='Blue', font='bold').grid(row=10, column=0,  columnspan=2, sticky=W, pady=4)  
		
		
		tk.Label(self.editFrame, text="Backlinks in the footnote", fg='Blue', font='bold').grid(row=10, column=2,  columnspan=2, sticky=W, pady=4)
		
		self.isSuperscript = IntVar() #Determines whether the hyperlink is in a supercript 
		self.chkSuperscript = ttk.Checkbutton(self.editFrame, text="Make the hyperlink a superscript", variable=self.isSuperscript)
		self.chkSuperscript.grid(row=20, column=0, columnspan=2, sticky=W)
		
		self.isLinkAtEnd = IntVar() #Determines whether the link is put at the end of a footnote 
		self.chkLinkAtEnd = ttk.Checkbutton(self.editFrame, text="Put the link at the end of the footnote", variable=self.isLinkAtEnd)
		self.chkLinkAtEnd.grid(row=20, column=2, columnspan=2, sticky=W)
		
		self.label = ttk.Label(self.editFrame, text="").grid(row=20, column=0, sticky=W, pady=4)  
		
		self.isUseNumber = IntVar()  #Determines whether to include an autonumber in the footnote
		self.chkUseNumber = ttk.Checkbutton(self.editFrame, text="Include a number in the footnote", variable=self.isUseNumber)
		self.chkUseNumber.grid(row=40, column=2, columnspan=2, sticky=W)   
		
		ttk.Label(self.editFrame, text="Prefix for hyperlink").grid(row=50, column=0, sticky=W, pady=4)		
		self.entryPrefix = ttk.Entry(self.editFrame, exportselection=0, width=20)
		self.entryPrefix.grid(row=50, column=1, sticky="W")
		self.entryPrefix.delete(0,END)

		ttk.Label(self.editFrame, text="Prefix for backlink").grid(row=50, column=2, sticky=W, pady=4)		
		self.entryPrefixFooter = ttk.Entry(self.editFrame, exportselection=0, width=20)
		self.entryPrefixFooter.grid(row=50, column=3, sticky="W")
		self.entryPrefixFooter.delete(0,END)
		
		self.label = ttk.Label(self.editFrame, text="Suffix for hyperlink").grid(row=60, column=0, sticky=W, pady=4)		
		self.entrySuffix = ttk.Entry(self.editFrame, exportselection=0, width=20)
		self.entrySuffix.grid(row=60, column=1, sticky="W")
		self.entrySuffix.delete(0,END)

		self.label = ttk.Label(self.editFrame, text="Suffix for backlink").grid(row=60, column=2, sticky=W, pady=4)		
		self.entrySuffixFooter = ttk.Entry(self.editFrame, exportselection=0, width=20)
		self.entrySuffixFooter.grid(row=60, column=3, sticky="W")
		self.entrySuffixFooter.delete(0,END)

		#### CSS SECTION ####
		tk.Label(self.editFrame, text="CSS file", fg='Blue', font='bold').grid(row=70, column=0, sticky=W, pady=4) 

		self.isUseCssFile = IntVar() #Determines whether a css file to format footnotes is to be used
		self.chkUseCssFile = tk.Checkbutton(self.editFrame, text="Insert css file", variable=self.isUseCssFile, command = lambda : self.toggleCssWidgets() )
		self.chkUseCssFile.grid(column=0, row=80, sticky=W)  

		self.labelCSS = tk.Label(self.editFrame, text = "CSS file:")
		self.labelCSS.grid(column = 0, row=90, sticky = W)
		self.entryCSS = tk.Entry(self.editFrame, exportselection=0, width=70)
		self.entryCSS.grid(row=90, column=1, columnspan=3, sticky="W")
		self.entryCSS.delete(0,END)

		self.btnCSS = tk.Button(self.editFrame, text="...", command=self.getCSSFileName, width = 5)
		self.btnCSS.grid(column=5, row=90, sticky=W) #This must be separated from the above otherwise the button cannot be disabled

		#Create a frame for the buttons
		#Without this, if the buttons are placed in different columns they will be spaced too far apart.
		self.buttonFrame = ttk.Frame(self.dlgframe, padding="1 1 1 1")
		self.buttonFrame.grid(column=0, row=90, sticky=(N, W, E, S))

		self.label = ttk.Label(self.buttonFrame, text="").grid(row=0, sticky=W, pady=4)	 #BLANK LINE
		tk.Button(self.buttonFrame, text="Insert footnotes", command=self.Insert, width = 15).grid(column=0, row=90, sticky=E)
		tk.Button(self.buttonFrame, text="Reinstate footnotes", command=self.StartReinstateFootnotes, width = 15).grid(column=1, row=90, sticky=W)
		tk.Button(self.buttonFrame, text="Save settings", command=self.SaveSettings, width = 15).grid(column=5, row=90, sticky=W)
		tk.Button(self.buttonFrame, text="Advanced", command=self.Advanced, width = 15).grid(column=10, row=90, sticky=E)
		tk.Button(self.buttonFrame, text="Cancel", command=self.Cancel, width = 15).grid(column=15, row=90, sticky=E)
		
		centerWindow(self.top) #This must be called after widgets have been set up

##Initialise the state of the widgets	 
		self.radioFootNote.set(self.prefs['FNPosition'])
		self.isSuperscript.set(self.prefs['Superscript'])
		self.isRenumberingChecked.set(self.prefs['Renumber'])
		self.entryPrefix.insert(0, self.prefs['Prefix'])
		self.entrySuffix.insert(0, self.prefs['Suffix'])
		self.entryCSS.insert(0, self.prefs['CSSFile'])
		self.isUseCssFile.set(self.prefs['UseCssFile'])
		self.entryPrefixFooter.insert(0, self.prefs['PrefixFooter'])
		self.entrySuffixFooter.insert(0, self.prefs['SuffixFooter'])
		self.isUseNumber.set(self.prefs['UseNumber'])
		self.isLinkAtEnd.set(self.prefs['LinkAtEnd'])

		if self.isUseCssFile.get()==0:
			self.entryCSS.configure(state="disabled")
			self.labelCSS.configure(state="disabled")
			self.btnCSS.configure(state="disabled")
		if self.radioFootNote.get()=="N":
			self.chkRenumber.configure(state="disabled")
			
	def GetEpub3Settings(self):
		"""
		This method controls the addition of epub:type elements to the eBook
		epub:type elements added to ePub3 books are as follows:
		epub:type="noteref" is added to hyperlinks
		epub:type="footnote" is added to footnotes
		The choice for including these is set in the Advanced Options dialog.
		Called by: StartReinstateFootnotes() and Insert()
		"""
		self.eTypeFootnote=""
		self.eTypeNoteref=""
		epubver = self.bk1.epub_version()
		if (epubver.startswith('3')) and (self.prefs['epub3Types']==True):
			self.eTypeFootnote=' epub:type="footnote"'
			self.eTypeNoteref=' epub:type="noteref"'
						
	def Advanced(self):
		AdvancedOptionsDlg = cAdvancedOptionsDlg(self.top, self.prefs) #Initialise instance of class
		AdvancedOptionsDlg.showAdvancedOptionsDlg()
		
	def toggleChkRenumbering(self):
		"""
		This method is called when a radio button is clicked.
		It sets the state of the checkbox for renumbering
		"""
		if self.radioFootNote.get()=="N":
			self.chkRenumber.configure(state="disabled")
		else:
			self.chkRenumber.configure(state="normal")
			
	def toggleCssWidgets(self):
		"""
		This method is called when the check box for using a css file is ticked.
		It sets the state of the css label, entry box and button
		"""
		if self.isUseCssFile.get()==0:
			self.entryCSS.configure(state="disabled")
			self.labelCSS.configure(state="disabled")
			self.btnCSS.configure(state="disabled") 
		else:
			self.entryCSS.configure(state="normal")
			self.labelCSS.configure(state="normal")
			self.btnCSS.configure(state="normal") 
			
	def SaveSettings(self):
		"""
		This method is called when the 'Save' button is clicked.
		"""
		self.prefs['FNPosition'] = self.radioFootNote.get()
		self.prefs['Prefix'] = self.entryPrefix.get()
		self.prefs['Suffix'] = self.entrySuffix.get()
		self.prefs['CSSFile'] = self.entryCSS.get()
		self.prefs['Renumber']  = self.isRenumberingChecked.get()
		self.prefs['Superscript'] = self.isSuperscript.get()
		self.prefs['UseCssFile'] = self.isUseCssFile.get()
		self.prefs['SuffixFooter'] = self.entrySuffixFooter.get()
		self.prefs['PrefixFooter'] = self.entryPrefixFooter.get()
		self.prefs['LinkAtEnd'] = self.isLinkAtEnd.get()
		self.prefs['UseNumber'] = self.isUseNumber.get()
		
		messagebox.showinfo('INFORMATION','Your options have been saved.')
		
	def TidyExit(self):
		"""
		This method is called before exiting the plugin.
		It can be used to ensure any tidying up is done before the plugin exits
		"""
		self.bk1.savePrefs(self.prefs)
		
	def getCSSSectionName(self):
		"""
		Puts either the css secion name (eg Footnote.css) or an empty string if
		the entry box for the css file is empty into self.entryCSS
		Called by getCSSFileName(), LoadCSSFile(), InitClass()
		"""
		if self.entryCSS.get()  == '':	#If no file specified
			self.cssName=""				#then set cssName to an empty string
		else:
			path=self.entryCSS.get().rpartition('/') 	#Break path into three tuples at the slash
			self.cssName=path[2]

	def getCSSFileName(self):
		"""
		This method is called when the button for obtaining a CSS filename is clicked
		It displays a file dialog box so that the user can select a css file for the footnotes
		"""
		fileOptions = dict(defaultextension='.css', filetypes=[('CSS file','*.css')], title='Open CSS file')
		filePath = filedialog.askopenfilename(**fileOptions)
		if filePath == "": return
		footNoteClass.CssPath = filePath
		self.entryCSS.delete(0,END)
		self.entryCSS.insert(0,footNoteClass.CssPath)
		self.prefs['CSSFile'] = self.entryCSS.get()
		self.getCSSSectionName() #Update the object global self.cssName
		
	def LoadCSSFile(self):
		"""
		This method writes the CSS file into the ePub.
		It is called by the method insert()
		"""
		if self.entryCSS.get()  == '':	#Get name of css file if not previously selected
			messagebox.showwarning('INFORMATION',"You have not selected a css stylesheet for the footers.\n\nYou need to install a css file if you want to format the footers automatically.\n\nThe footnotes will be inserted without including a css stylesheet.\n\nFor instructions on how to set up a file for this please see the manual.")
			return

		self.getCSSSectionName()
		basename=self.cssName				#Name of css section eg Footnotes.css
		uid=basename						#Set uid to basename
		mime = "text/css"
		
		CSSFileAbsent=True
		for (idCSS, href) in self.bk1.css_iter():		#Check whether the css file or ID is already present
			href=href.rpartition('/') 			#Break href into three tuples at the slash
			href=href[2]						#Get the *.css part only
			if idCSS==uid or href==basename:	#If uid or the css style name is in use then clear flag 
				CSSFileAbsent=False				#If so, clear flag 
		
		if CSSFileAbsent:	#If css file is not present in the ePub
	
			try:
					f=open(self.entryCSS.get(), 'r', encoding="utf-8")
			except IOError:
					print('Error opening the css file for the style sheet')
					messagebox.showwarning('WARNING',"The CSS file could not be loaded. Please check the name and the path of the css file that you want to include.")
			else: 
				try:
					cssString = f.read()
				except IOError:
					print('Error reading the css file for the style sheet')
					messagebox.showwarning('WARNING',"There was an error reading the css file for the style sheet.")
				else:
					self.bk1.addfile(uid, basename, cssString, mime)
	
	def MakeFootNoteList(self):
		"""
		This method creates a list containing information about footnotes
		It includes the xhtml section that links to the footnote,
		the ID for the footnote, and the text for the footnote
		It is called by the method ReinstateFootnotes()
		"""
		self.FNList=[]
		#Is the backlink at the end of each footnote?
		m1 = re.search(r'<p class="'+self.prefs['FootnoteClass']+'">(.*?)</a></p>', self.html)
		if m1: #The backlink is at the end of the footnote
			searchTerm=r'<p class="'+self.prefs['FootnoteClass']+'"><a id="(.*?)"></a>(.*?)<a href="(.*?)#'+ self.prefs['HyperlinkID']+'(.*?)"(.*?)</a></p>(\s+)?'

			foundIt = re.search(searchTerm, self.html)
			
			if foundIt is not None:
				FNInnerList=[]
				for m in re.finditer(searchTerm, self.html):
					#Groups 4, 2 and 3 contain the ID, text and xhtml section of each footnote
					#Order must be xhtml section, ID and text ie  3  4   2
					FNInnerList.extend([m.group(3),  m.group(4),   m.group(2) ])
					self.FNList.append(FNInnerList)
					FNInnerList=[]
					self.html= self.html.replace(m.group(0), '') #Deletes current footnote
		else:  #The backlink is at the start of the footnote
			searchTerm='<p class="'+self.prefs['FootnoteClass']+'">(.*?)<a href="(.*?)#'+self.prefs['HyperlinkID']+'(.*?)">(.*?)>(.*?)</p>(\s+)?'
			
			foundIt = re.search(searchTerm, self.html)
			
			if foundIt is not None:
				FNInnerList=[]
				for m in re.finditer(searchTerm, self.html):
					#Groups 2, 3 and 5 contain the xhtml section, ID and text of each footnote
					FNInnerList.extend([m.group(2),  m.group(3),   m.group(5) ])
					self.FNList.append(FNInnerList)
					FNInnerList=[]
					self.html= self.html.replace(m.group(0), '') #Delete current footnote

	def ReinstateFootnotes(self):
		"""
		This method restores the footnotes to their place in the body of the text
		This method is used when the footnotes are included in each xhtml
		section of the epub
		It is called by RestoreFromFootnoteSection() and StartReinstateFootnotes()
		"""
		for (id, href) in self.bk1.text_iter():  # For each xhtml section in the epub
			self.html = self.bk1.readfile(id)    # Read the section into html
			if not isinstance(self.html, str):   # If the section is not type str
				self.html = text_type(self.html, 'utf-8')  # then sets its type to 'utf-8'
			html_orig = self.html  # Copy the result to html_orig

			#Create a list of data for each footnote in the current xhtml section
			self.MakeFootNoteList()

			for j in range(len(self.FNList)): #For each footnote in the list...
				FNSection=self.FNList[j][0]  #Get the xhtml section that contains the footnote eg sec_01.html
				FNID=self.FNList[j][1]   #Get the ID for the footnote eg ID4
				FNtext=self.FNList[j][2].lstrip()  #Get the text for the footnote
				
				#Insert the footnote in the body of the xhtml section
				self.html=re.sub(r'<a id="'+self.prefs['HyperlinkID']+ FNID+'"'+self.eTypeNoteref+'></a>(.*?)</a>', '<span class="'+self.prefs['HighlightClass']+'">'+FNtext+'</span>', self.html)
				
				#Delete the heading for the footnotes and all empty lines following it:
				if self.prefs["FootnoteHeading"] != "": #Footnotes have a heading
					self.html= re.sub(r'\s+<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\s+', '\n', self.html)
				else:  #Footnotes do not have a heading
					self.html= re.sub(r'</p>(\s)+</body>', '</p>\n</body>', self.html)

			if not self.html == html_orig:self.bk1.writefile(id, self.html)	#If the text has changed then write the amended text to the book

	def processSectionList(self):
		"""
		This method reinstates footnotes from where they have been placed in one
		xhtml section to the body of the text in the relevant xhtml section.
		It iterates through each xhtml section and finds data in the list for
		footnotes in each section in turn. It uses this data to reinstate the
		footnote in the relevant xhtml section
		Called by RestoreFromFootnoteSection()
		"""
		for (id, href) in self.bk1.text_iter():  # For each xhtml section in the epub
			self.html = self.bk1.readfile(id)    # Read the section into html
			if not isinstance(self.html, str):   # If the section is not type str
				self.html = text_type(self.html, 'utf-8')  # then sets its type to 'utf-8'
			html_orig = self.html  # Copy the result to html_orig

			h = href.replace('Text/', '')

			#Code to reinstate footnotes in the body of the text:
			#Iterate through list, looking for an xhtml section match
			#Reinstate matched footnotes details from the list
			
			for j in range(len(self.FNList)): #For each footnote in the list...
				if (h in self.FNList[j][0]): #self.FNList[j][0] has the xhtml section that contains the footnote eg sec_01.html
					FNID=self.FNList[j][1]  #Get the ID for the footnote eg ID4
					FNtext=self.FNList[j][2].lstrip()  #Get the text for the footnote
					
					#Insert the footnote in the body of the xhtml section
					self.html=re.sub(r'<a id="'+self.prefs['HyperlinkID']+ FNID+'"'+self.eTypeNoteref+'></a>(.*?)</a>', '<span class="'+self.prefs['HighlightClass']+'">'+FNtext+'</span>', self.html)
					
					#Delete the heading for the footnotes and all empty lines following it
					self.html= re.sub(r'\s+<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\s+', '\n', self.html)
	
			if not self.html == html_orig:self.bk1.writefile(id, self.html)	#If the text has changed then write the amended text to the ePub

	def RestoreFromFootnoteSection(self, ePubFootNoteID):
		"""
		The parameter ePubFootNoteID is the ID of the xhtml section of the 
		ePub that contains the footnotes
		This method reads the footnote section into memory.
		It calls MakeFootNoteList() to store the xhtml section, ID and
		self.processSectionList() text for each footnote. Then it calls
		processSectionList() to move footnotes back into the relevant xhtml
		section.
		It is called by StartReinstateFootnotes()
		"""
		self.html = self.bk1.readfile(ePubFootNoteID)  # Read the footnote section into html
		if not isinstance(self.html, str):  # If the section is not type str
			self.html = text_type(self.html, 'utf-8')  # then sets its type to 'utf-8'
		html_orig = self.html  # Copy the result to html_orig
		
		self.MakeFootNoteList()

		self.processSectionList()

	def StartReinstateFootnotes(self):
		"""
		This method restores the footnotes to their place in the body of the text
		It is called when the button 'Reinstate Footnotes' is clicked
		"""
		self.GetEpub3Settings()
		#Determine whether there is an xhtml section that holds all footnotes
		FootnoteSectionExists=False
		for (id, href) in self.bk1.text_iter():
			if href=='Text/' + self.prefs['xhtmlFootnoteSectionName'] + '.xhtml':
				FootnoteSectionExists=True
				ePubFootNoteID=id
				
		if FootnoteSectionExists: #If the footnote is placed in one xhtml section:
			#Reinstate footnotes from the footnote section
			self.RestoreFromFootnoteSection(ePubFootNoteID)
			#and then delete the footnote section
			self.bk1.deletefile(ePubFootNoteID)
		else:
			#Reinstate footnotes in each xhtml section
			self.ReinstateFootnotes()
		
		self.TidyExit()
		self.top.destroy()
		
	def InitClass(self):
		"""
		This method determines whether the css file for formatting footers is
		present in the ePub
		If the css file is absent then self.CSSFileAbsent is set to True
		If the css file is present then self.CSSFileAbsent is set to False and
		the name of the css section is stored in self.cssName
		If the name of the css file is blank then this method sets
		self.cssName to "" and self.CSSFileAbsent=False
		Called by Insert()
		"""
		if self.entryCSS.get() != '':
			self.getCSSSectionName()
			#Extract the name of css file including extension but without the path
			
			self.CSSFileAbsent=True
			for (idCSS, href) in self.bk1.css_iter():	#Check whether the css file is already present
				href=href.rpartition('/') 			#Break href into three tuples at the slash, eg   ('Styles', '/', 'Footnotes.css')
				href=href[2]						#Get the *.css part only

				if href==self.cssName:	#If uid or the css style name is in use then clear flag 
					self.CSSFileAbsent=False		#If so, clear flag
		else: #Safety measures...
			self.CSSFileAbsent=False
			self.cssName=""

	def LinkStyle(self):
		"""
		This method will add a link to the css sheet in each xhtml section of the epub
		IMPORTANT: This must be called before the css file is loaded because 
		the absence of the css file is used to indicate that links have not been
		inserted.
		Called by Insert()
		"""
		#Insert a link in the xhtml file if it is not already present 
		if (self.cssName not in self.html):
			self.html=re.sub(r'</head>', '\n<link href="../Styles/' + self.cssName + '" type="text/css" rel="stylesheet"/>\n</head>', self.html)

	def InsertFooterSection(self):
		"""
		This method inserts all footnotes in one xhtml section
		It is called by the method insert()
		"""
		for (id, href) in self.bk1.text_iter():
			if href==self.textFootNoteSection:	#If the section already exists
				messagebox.showerror(title='WARNING', message='Section exists\n', detail='The section named '+self.textFootNoteSection+ 'is already present in your eBook', icon='warning')
				sys.exit("")

		epubver = self.bk1.epub_version()
		
		xml  = '<?xml version="1.0" encoding="UTF-8" standalone="no" ?>\n'
		
		if epubver.startswith('2'):
			xml += '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n'
			xml += '<html xmlns="http://www.w3.org/1999/xhtml">\n'
		else:
			xml += '<!DOCTYPE html>\n'
			xml += '<html xmlns="http://www.w3.org/1999/xhtml" xmlns:epub="http://www.idpf.org/2007/ops">\n'
			
		xml += '<link href="../Styles/' + self.cssName + '" type="text/css" rel="stylesheet"/>'
		xml += '<head>\n'
		xml += '  <title></title>\n'
		xml += '<style type="text/css">\n'
		xml += '@page {padding: 0; margin:0}\n'
		xml += '</style>\n'  
		xml += '</head>\n'
		xml += '\n'
		xml += '<body>\n'
		xml += self.allFootnotes
		xml += '</body>\n'
		xml += '</html>\n'		   
	
		uid	   = self.prefs['xhtmlFootnoteSectionName'] #'Footnotes'
		basename  = self.prefs['xhtmlFootnoteSectionName'] + '.xhtml'  #Footnotes.xhtml
		mime = 'application/xhtml+xml'
	
		if epubver.startswith('2'):
			self.bk1.addfile(uid, basename, xml, mime)	#Add new xhtml section to the epub2 eBook
		else:
			self.bk1.addfile(uid, basename, xml, mime, self.prefs['xhtmlFootnoteSectionName'])	#Add new xhtml section to the epub3 eBook
	
		self.bk1.spine_insert_before(-1, uid, None, None)

	def repl(self, m):
		"""
		This method replaces the marked text corresponding to the footnote in
		the current text section.
		It also constructs the current footnote, including tags
		It is called by the method Insert()
		"""
		# print("GroupNone: ", m.group())
		# print("Group0: ", m.group(0))
		
		Ref = str(self.noteNumber)
		self.noteNumber = self.noteNumber +1
			
		FN=m.group(2) #Get the text for the footnote

		#Now construct the string to replace the text marked for the footnote in the current xhtml section
		if (self.isSuperscript.get() == 1): #If the user wants superscripts...
			backLinkStr = '<a id="'+self.prefs['HyperlinkID']+Ref+'"'+self.eTypeNoteref+'></a><a href="../' + self.textSectionXHTML +'#' +self.prefs['FootnoteID'] + Ref+'"><sup>'+self.entryPrefix.get() + Ref +self.entrySuffix.get()+'</sup></a>'
			
		else: #The user does not want superscripts
			backLinkStr = '<a id="'+self.prefs['HyperlinkID']+Ref+'"'+self.eTypeNoteref+'></a><a href="../' + self.textSectionXHTML +'#' +self.prefs['FootnoteID'] + Ref+'">'+self.entryPrefix.get() + Ref +self.entrySuffix.get()+'</a>'	
		
		insertRef=""
		if self.isUseNumber.get():	#Check whether a number is required in the footnote
			insertRef=Ref
		
		#Construct the footnote 
		
		if self.isLinkAtEnd.get(): #Put link at end of footnote
			FootNote='<p class="'+self.prefs['FootnoteClass']+'"><a id="'+self.prefs['FootnoteID'] + Ref +'"'+self.eTypeFootnote+'></a>'+FN+'<a href="../' + self.textCurrentSection +'#'+self.prefs['HyperlinkID']+Ref+'">'+self.entryPrefixFooter.get()+insertRef+self.entrySuffixFooter.get()+'</a></p>'
## Amendments			
		else: #Put link at start of footnote
			
			if (self.prefs['epub3Types']==True): #Implements an epub3 eBook with popup footnotes
 #Implements an epub3 eBook with popup footnotes
				FootNote='<aside id="'+self.prefs['FootnoteID'] + Ref +'"'+self.eTypeFootnote+'><p class="'+self.prefs['FootnoteClass']+'"><a href="../' + self.textCurrentSection +'#'+self.prefs['HyperlinkID']+Ref+'">'+self.entryPrefixFooter.get() + insertRef+self.entrySuffixFooter.get()+'</a>' + " " +FN+'</p></aside>'
				
			else: 
				FootNote='<p class="'+self.prefs['FootnoteClass']+'"><a id="'+self.prefs['FootnoteID'] + Ref +'"'+self.eTypeFootnote+'></a><a href="../' + self.textCurrentSection +'#'+self.prefs['HyperlinkID']+Ref+'">'+self.entryPrefixFooter.get() + insertRef+self.entrySuffixFooter.get()+'</a>' + " " +FN+'</p>'

			
		#Add the footnote to the string that contains all the footnotes for the section
		self.allFootnotes=self.allFootnotes+"\n\n" + FootNote
		return backLinkStr
	
	def Insert(self):
		"""
		This method inserts footnotes in the ePub
		Called when the user clicks the "Insert footnote" button
		"""
		self.InitClass() #Set the value of self.CSSFileAbsent (True if the css file for the footer is absent) and sets self.cssName
		self.GetEpub3Settings()
		if self.prefs["FootnoteHeading"] !="":
			self.allFootnotes='\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n'
		else:
			self.allFootnotes=""
		self.textFootNoteSection='Text/' + self.prefs['xhtmlFootnoteSectionName'] + '.xhtml'#The name of the section where footnotes will be inserted
		for (id, href) in self.bk1.text_iter():  # For each xhtml section in the epub
			self.html = self.bk1.readfile(id)  # Read the section into html
			if not isinstance(self.html, str):  # If the section is not type str
				self.html = text_type(self.html, 'utf-8')  # then sets its type to 'utf-8'
			html_orig = self.html  # Copy the result to html_orig

			#Precaution: If footnotes are already in this xhtml section then exit this loop
			#Prevents putting more footnotes into the section that holds only footnotes
			#Prevents writing footnotes to a section that already has footnotes, hence,
			#can add new xhtml section after putting footnotes in previous sections.
			if ('<p class="' + self.prefs['FootnoteHeadingClass'] + '">' in self.html) : break

			#Put the location for the footnotes in textSectionXHTML and update noteNumber if relevant
			if (self.radioFootNote.get() == "N"):	   #Put footnotes in o(N)e xhtml section...
				self.textSectionXHTML = self.textFootNoteSection
			else:									   #Put all footnotes in the (C)urrent xhtml section
				self.textSectionXHTML = href			#href contains the text section eg Text/Footnotes.html
				if (self.isRenumberingChecked.get() == 1): #Reset counter if renumbering is required for each xhtml section
					self.noteNumber = 1
				
			self.textCurrentSection = href
			
			#Replace text marked as a footnote in the xhtml file with the link to the footnote
			#Also, the repl method will construct each footnote for the current xhtml section and then add
			#it to the string that contains all the footnotes for the current xhtml section
			self.html=re.sub(r'(<span class="'+self.prefs['HighlightClass']+'">(.*?)</span>)', self.repl, self.html)
	
			if (self.radioFootNote.get() == "C"):	#If footnotes are to go in (C)urrent xhtml section...
			
				if (self.allFootnotes !='\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n') and (self.allFootnotes !=""): #If there are footnotes to add to this xhtm section...
				
					index = self.html.find("</body>")	#then insert them before the </body> tag of the current xhtml section
					self.html= self.html[: index ] + self.allFootnotes  + self.html[ index :]
					
					#Reset the content of allFootnotes
					if self.prefs["FootnoteHeading"] !="":
						self.allFootnotes='\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n'
					else:
						self.allFootnotes=""
		
					self.LinkStyle() #and insert a style link in this xhtml file
	
			if not self.html == html_orig:self.bk1.writefile(id, self.html)	#If the text has changed then write the amended text to the book
	
		#Insert a footnote section IF footnotes are to go in a New section AND footnotes have been found - if none
		#then allFootnotes will only contain '\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n' or an empty string
		
		if ((self.radioFootNote.get() == "N") and (self.allFootnotes!='\n\n<p class="' + self.prefs['FootnoteHeadingClass'] + '">'+self.prefs["FootnoteHeading"]+'</p>\n\n')): 
			self.InsertFooterSection()
			
		if  self.isUseCssFile.get()==1:
			self.LoadCSSFile()
		self.TidyExit()
		self.top.destroy()

	def EscapePressed(self, event):
		self.Cancel()
		
	def Cancel(self):
		"""
		Called when the user clicks the "Cancel" button
		"""
		#print("The operation was cancelled")
		self.top.destroy()

def run(bk):
	"""
	This is the entry point for the plugin
	It checks the Sigil version and shows the main dialog box
	"""
	
	print('Python Laucher Version:', bk.launcher_version())
	if bk.launcher_version() < 20151024:	#Check Sigil version is greater than 0.9.0....
		#print("You need to use Sigil 0.9.1 or greater for this plugin")
		#print('\n\nPlease click OK to close the Plugin Runner window.')
		messagebox.showwarning('WARNING',"You need to use Sigil 0.9.1 or greater for this plugin.\n\nPlease click OK to close the Plugin Runner window.")
		return 0		#....and return if Sigil  is not greater than 0.9.0)

	prefs = bk.getPrefs()		# Get preferences from json file
  ##Init prefs  
	if not 'Prefix' in prefs: prefs['Prefix']  = '['
	if not 'Suffix' in prefs: prefs['Suffix']  = ']'
	if not 'Superscript' in prefs: prefs['Superscript']  = 1
	if not 'Renumber' in prefs: prefs['Renumber']  = 0
	if not 'CSSFile' in prefs: prefs['CSSFile']  = ""
	if not 'FNPosition' in prefs: prefs['FNPosition']  = "C"
	if not 'UseCssFile'in prefs: prefs['UseCssFile']  = 0
	if not 'PrefixFooter'in prefs: prefs['PrefixFooter']  = '('
	if not 'SuffixFooter'in prefs: prefs['SuffixFooter']  = ')'
	if not 'UseNumber' in prefs: prefs['UseNumber']  = 1
	if not 'LinkAtEnd' in prefs: prefs['LinkAtEnd']  = 0
	# Prefs for the Advanced settings
	if not 'xhtmlFootnoteSectionName' in prefs: prefs['xhtmlFootnoteSectionName']  = 'Footnotes'
	if not 'FootnoteHeading' in prefs: prefs['FootnoteHeading']  = 'FOOTNOTES'
	if not 'FootnoteHeadingClass' in prefs: prefs['FootnoteHeadingClass']  = 'Footer'
	if not 'FootnoteID' in prefs: prefs['FootnoteID']  = "FNID"
	if not 'FootnoteClass' in prefs: prefs['FootnoteClass']  = 'Footnote'
	if not 'HyperlinkID' in prefs: prefs['HyperlinkID']  = 'FOOTNOTE'
	if not 'HighlightClass' in prefs: prefs['HighlightClass']  = 'FNOTE'
	if not 'epub3Types' in prefs: prefs['epub3Types']  = 0 	#Governs whether epub:types are used
	
	root = tk.Tk()
	root.withdraw()	#Hide main window
	
	footnoteDialog = footNoteClass(root, bk, prefs)	#Create an instance of the footNoteClass class
	footnoteDialog.ShowDialogBox()   #Show the main dialog	
	
	#Check for updates after opening the main window so that the dialog for checking updates is on top of the main window
	url = "https://www.mobileread.com/forums/showthread.php?t=324637"
	CheckForUpdates(root, prefs, url)
	
	root.wait_window(footNoteClass.dlgTop)	#Wait until the main dialog is destroyed
	
	bk.savePrefs(prefs)					#Save user preferences to a json file
	root.destroy()						#Destroy the root
	
	print('\n\nPlease click OK to close the Plugin Runner window.')	
	
	root.mainloop()
	return (0)