﻿#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2019, Daniel Prazak <kret33n@gmail.com>'
__docformat__ = 'restructuredtext cs'

from threading import Thread
from calibre.ebooks.metadata.book.base import Metadata
from datetime import datetime
from calibre.utils.date import utc_tz
import calibre_plugins.databazeknih.config as cfg
from calibre_plugins.databazeknih import load_url

def get_config_option(config_option):
    return cfg.plugin_prefs[cfg.STORE_NAME].get(config_option, cfg.DEFAULT_STORE_VALUES[config_option])

class Worker(Thread): # Get details
    isbn = None
    '''
    Get book details from databazeknih.cz book page in a separate thread
    '''

    def __init__(self, url, result_queue, browser, log, relevance, plugin, extra_metadata, timeout=20):
        Thread.__init__(self)
        self.daemon = True
        self.url, self.result_queue = url, result_queue
        self.log, self.timeout = log, timeout
        self.relevance, self.plugin = relevance, plugin
        self.browser = browser.clone_browser()
        self.cover_url = self.databazeknih_id = self.dbknih_id = self.dbk_id = self.orig_databazeknih_id = None #self.isbn = None
        self.cover_urls = None

        self.is_tale = True if '/povidky/' in url else False
        self.pubyear = extra_metadata.get('pubdate', None)
        self.publisher_meta = extra_metadata.get('publisher', None)
        self.book_id = self.url.split('?')[0].split('-')[-1]
        self.urlCall = 'https://www.databazeknih.cz/book-detail-more-info/%s'%self.book_id

    def run(self):
        self.log.info('worker jede')
        try:
            self.get_details()
        except:
            self.log.exception('get_details failed for url: %r'%self.url)

    def get_details(self):
        """
        this function loads all pages from which plugin parses data
        """
        if self.is_tale:
            try:
                self.log.debug('Running load_url for: %s ' % self.url )
                root, _ = load_url(self.log, self.url, self.browser)
            except Exception as e:
                self.log.error('Load url problem: %r - %s' % (self.url, e))
                return
            self.parse_tale(root)
        else:
            try:
                self.log.info('Get main parsing page:%s'%self.url)
                root, _ = load_url(self.log, self.url, self.browser)

                self.log.info('Get additional details:%s'%self.urlCall)
                additional, _ = load_url(self.log, self.urlCall, self.browser)

                old_url = self.url
                self.get_book_issues(root, additional)
                if old_url != self.url:
                    root, _ = load_url(self.log, self.url, self.browser)
            except Exception as e:
                self.log.error('Load url problem: %r - %s' % (self.url, e))
                additional = None
                return
            
            append_to_comments = get_config_option(cfg.APPEND_TO_COMMENTS)
            append_to_tags = get_config_option(cfg.APPEND_TO_TAG)

            root_tales = None
            if dict([[tupl[0], tupl[1]] for tupl in append_to_tags]).get(cfg.TALES_IN_BOOK, False) or cfg.TALES_IN_BOOK in [tupl[0] for tupl in append_to_comments]:
                try:
                    self.log.info('Get containing tales page:%s'%self.url.replace('/zajimavosti-knihy/', '/povidky-z-knihy/'))
                    root_tales, _ = load_url(self.log, self.url.replace('/zajimavosti-knihy/', '/povidky-z-knihy/'), self.browser)
                except Exception as e:
                    self.log.exception('Getting containing tales details failed. %s - %s'%(self.urlCall, e))

            root_whole_rating = None
            if get_config_option(cfg.CALCULATE_RATING):
                try:
                    self.log.info('Get detailed rating page :%s'%self.url.replace('/zajimavosti-knihy/', '/hodnoceni-knihy/'))
                    root_whole_rating, _ = load_url(self.log, self.url.replace('/zajimavosti-knihy/', '/hodnoceni-knihy/'), self.browser)
                except Exception as e:
                    self.log.exception('Getting detailed rating page failed. %s - %s'%(self.url.replace('/zajimavosti-knihy/', '/hodnoceni-knihy/'), e))
            self.parse_details(root, additional, root_tales, root_whole_rating)

    def get_book_issues(self, root, additional):
        # 0:default - 1:cz_new - 2:cz_old - 3:sk_new - 4:sk_old
        issue_pref = get_config_option(cfg.ISSUE_PREFERENCE)
        max_covers = get_config_option(cfg.MAX_COVERS)
        if issue_pref == 0 and not self.pubyear and not self.publisher_meta and max_covers == 1:
            self.log.info("No 'pubdate' or 'publisher' identifier specified in Title and default book issue specified in plugin preferences ... Parsing default issue...")
            return
        
        url_cz = '%s?lang=cz'%self.url
        url_cz = url_cz.replace('/zajimavosti-knihy/', '/dalsi-vydani/')
        url_sk = '%s?lang=sk'%self.url
        url_sk = url_sk.replace('/zajimavosti-knihy/', '/dalsi-vydani/')
        found_languages = self.parse_languages(additional)
        found_language = found_languages[0] if found_languages else None

        old_url = self.url
        if self.pubyear or self.publisher_meta:
            self.log.info("Parsing another book issue specified in Title 'pubdate' or 'publisher' identifier ...")
            if ((issue_pref == 0 and found_language != 'sk') or issue_pref == 1 or issue_pref == 2): #cz first
                root_book_issues_cz, _ = load_url(self.log, url_cz, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_cz)
                if new_url:
                    root, _ = load_url(self.log, new_url, self.browser)
                    self.url = new_url
                    self.cover_urls = cover_urls
                    return
                else: # try sk editions
                    root_book_issues_sk, _ = load_url(self.log, url_sk, self.browser)
                    new_url, cover_urls = self.parse_book_issues(root_book_issues_sk)
                    if new_url:
                        root, _ = load_url(self.log, new_url, self.browser)
                        self.url = new_url
                        self.cover_urls = cover_urls
                        return
            if ((issue_pref == 0 and found_language == 'sk') or issue_pref == 3 or issue_pref == 4): # sk first
                root_book_issues_sk, _ = load_url(self.log, url_sk, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_sk)
                if new_url:
                    root, _ = load_url(self.log, new_url, self.browser)
                    self.url = new_url
                    self.cover_urls = cover_urls
                    return
                else: # then cz editions
                    root_book_issues_cz, _ = load_url(self.log, url_cz, self.browser)
                    new_url, cover_urls = self.parse_book_issues(root_book_issues_cz)
                    if new_url:
                        root, _ = load_url(self.log, new_url, self.browser)
                        self.url = new_url
                        self.cover_urls = cover_urls
                        return

        if self.url == old_url: # without Title identifier checking
            self.pubyear = None
            self.publisher_meta = None
            self.log.info("Parsing another book issue specified in plugin preferences (not Default issue or max covers > 1) ...")
            if (issue_pref == 0 and found_language != 'sk') or issue_pref == 1:
                root_book_issues_cz, _ = load_url(self.log, url_cz, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_cz)
                if new_url:
                    root, _ = load_url(self.log, new_url[0], self.browser)
                    self.url = new_url[0]
                    self.cover_urls = cover_urls
                    return
            if issue_pref == 2:
                root_book_issues_cz, _ = load_url(self.log, url_cz, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_cz)
                if new_url:
                    root, _ = load_url(self.log, new_url[-1], self.browser)
                    self.url = new_url[-1]
                    self.cover_urls = cover_urls[::-1] if cover_urls else None
                    return
            if (issue_pref == 0 and found_language == 'sk') or issue_pref == 3:
                root_book_issues_sk, _ = load_url(self.log, url_sk, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_sk)
                if new_url:
                    root, _ = load_url(self.log, new_url[0], self.browser)
                    self.url = new_url[0]
                    self.cover_urls = cover_urls
                    return
            if issue_pref == 4:
                root_book_issues_sk, _ = load_url(self.log, url_sk, self.browser)
                new_url, cover_urls = self.parse_book_issues(root_book_issues_sk)
                if new_url:
                    root, _ = load_url(self.log, new_url[-1], self.browser)
                    self.url = new_url[-1]
                    self.cover_urls = cover_urls[::-1] if cover_urls else None
                    return

    def parse_book_issues(self, root):
        year_node = self.parse_all(root, '//p[@class="new odtopm"]/text()[normalize-space()]', 'parsed_issues_year', convert=lambda x: [node.strip() for node in x])
        url_node = self.parse_all(root, '//a[contains(@href, "/dalsi-vydani/") and  .//img]/@href', 'parsed_issues_url', convert=lambda x: ['https://www.databazeknih.cz%s'%node.replace('/dalsi-vydani/', '/zajimavosti-knihy/').replace('?lang=sk', '').replace('?lang=cz', '') for node in x])
        publisher_node = self.parse_all(root, '//p[@class="new odtopm"]/a[contains(@href, "/nakladatelstvi/") and position() = 1]/text()', 'parsed_issues_publisher', convert=lambda x: [node.strip() for node in x])
        img_urls = self.parse_all(root, '//a[contains(@href, "/dalsi-vydani/") and  .//img]//img/@src', 'parsed_issues_img_url')
        book_type_node = self.parse_all(root, '//p[@class="new odtopm"]/img/@title', 'parsed_issues_book_type')

        # Removing all audiobook issue records
        if book_type_node:
            indexes_audio = [index for index, value in enumerate(book_type_node) if value == 'audiokniha']
            if indexes_audio:
                for i in indexes_audio:
                    for node_list in [year_node, url_node, publisher_node, img_urls, book_type_node]:
                        node_list.pop(i)

        def find_closest_list_index(num_list, target):
            num_list = [int(num) for num in num_list]
            indexed_list = list(enumerate(num_list))
            indexed_list.sort(key=lambda x: abs(x[1] - target))
            sorted_indexes = [index for index, _ in indexed_list]
            return sorted_indexes

        issue_pref = get_config_option(cfg.ISSUE_PREFERENCE)
        if not self.pubyear and not self.publisher_meta:
            return url_node, img_urls # all urls

        if img_urls and year_node and url_node and publisher_node and len(img_urls) == len(year_node) == len(url_node) == len(publisher_node):
            if self.publisher_meta:
                meta_publisher = self._convert_to_calibre(self.publisher_meta.replace('_', ' '), get_config_option(cfg.KEY_PUBLISHER_MAPPINGS))
                if self.pubyear:
                    sorted_indexes = find_closest_list_index(year_node, int(self.pubyear))
                else:
                    if (issue_pref == 0 or issue_pref == 1 or issue_pref == 3):
                        sorted_indexes = find_closest_list_index(year_node, int(year_node[0]))
                    else:
                        sorted_indexes = find_closest_list_index(year_node, int(year_node[-1]))
                if sorted_indexes:
                    for index in sorted_indexes:
                        found_publi = self._convert_to_calibre(publisher_node[index], get_config_option(cfg.KEY_PUBLISHER_MAPPINGS))
                        if found_publi and meta_publisher:
                            if strip_accents(found_publi.replace('_', ' ').lower()) == strip_accents(meta_publisher.replace('_', ' ').lower()):
                                self.log(index, publisher_node[index], year_node[index])
                                return url_node[index], img_urls[index]
                    return url_node[sorted_indexes[0]], img_urls[sorted_indexes[0]]
            else:
                if self.pubyear:
                    sorted_indexes = find_closest_list_index(year_node, int(self.pubyear))
                    if sorted_indexes:
                        return url_node[sorted_indexes[0]], img_urls[sorted_indexes[0]]
        return (None, None)

    def parse_recalculated_rating(self, root):
        decimal_accuracy = get_config_option(cfg.CALCULATE_RATING_ACCURACY)
        ratings = self.parse_all(root, '//table[@class="odtop morePadding"]/tr/td[not(descendant::img or div or contains(descendant::text(), "odpad"))]/text()', 'ratings_stars_separated', convert=lambda x: [int(node.replace(' ', '').strip()) for node in x][::-1])
        rating5 = (sum([i*r for i, r in enumerate(ratings)])/sum(ratings)) if ratings else 0
        self.log.debug('debug ratings: %s sum ratings: %s'%(rating5, sum(ratings) if ratings else 0))
        rating100 = rating5/5 * 100
        rating10 = rating5/5 * 10
        rating_count = sum(ratings) if ratings else 0
        return round(rating5, decimal_accuracy), round(rating10, decimal_accuracy), round(rating100, decimal_accuracy), rating_count

    def get_parse_authors_real_name(self, root):
        real_names = []
        authors_urls = self.parse_all(root, '//span[@itemprop="author"]/a[contains(@href, "/autori/")]/@href', 'authors_urls', convert=lambda x: ['https://www.databazeknih.cz%s'%node.strip() for node in x])
        if authors_urls:
            for url in authors_urls:
                try:
                    root, _ = load_url(self.log, url, self.browser)
                    real_names.append(self.parse_author_real_name(root))
                except:
                    self.log.error('Error when loading authors page url:%s'%url)
        return real_names

    def get_parse_obalkyknih_cover(self, isbn):
        url_obalky = 'https://www.obalkyknih.cz/view?isbn=%s'%isbn

        try:
            self.log.info('Get obalkyknih page:%s'%url_obalky)
            root, _ = load_url(self.log, url_obalky, self.browser)
        except Exception as e:
            self.log.error('Load url problem: %r - %s' % (url_obalky, e))
            return
        cover_url = self.parse_first(root, '//a[@data-lightbox="book-cover"]/@href', 'obalkyknih_cover')
        if cover_url:
            return cover_url
        else:
            return None

    def parse_details(self, root, additional = None, root_tales = None, root_whole_rating = None):
        max_covers = get_config_option(cfg.MAX_COVERS)
        obalkyknih_cover = get_config_option(cfg.OBALKYKNIH_COVER)
        append_to_comments = get_config_option(cfg.APPEND_TO_COMMENTS)
        series_editions = get_config_option(cfg.SERIES_EDITIONS)
        publication_date = get_config_option(cfg.PUBLICATION_DATE)
        parse_isbn = get_config_option(cfg.PARSE_ISBN)
        parse_databazeknih = get_config_option(cfg.PARSE_DATABAZEKNIH)
        swap_authors = get_config_option(cfg.SWAP_AUTHORS)
        only_one_author = get_config_option(cfg.ONE_AUTHOR)
  
        title_line = get_config_option(cfg.TITLE_LINE)
        publisher_line = get_config_option(cfg.PUBLISHER_LINE)

        append_to_tags = get_config_option(cfg.APPEND_TO_TAG)
        append_to_identifiers = get_config_option(cfg.APPEND_TO_IDENTIFIERS)
        authors_include = get_config_option(cfg.AUTHORS_INCLUDE)
        real_authors_include = get_config_option(cfg.REAL_AUTHS_INCLUDE)
        translators_include = get_config_option(cfg.TRANSLATORS_INCLUDE)
        illustrators_include = get_config_option(cfg.ILLUSTRATORS_INCLUDE)
        cover_authors_include = get_config_option(cfg.COVER_AUTHORS_INCLUDE)

        try:
            self.log.info('Parse details:%s'%self.url)
            databazeknih_id, databazeknih_id_short = self.parse_databazeknih_id(root)
            self.log.info('Parsed DK identifier:%s'%databazeknih_id)

        except:
            self.log.exception('Error parsing databazeknih id for url: %r'%self.url)
            databazeknih_id = databazeknih_id_short = None

        try:
            title = self.parse_title(root)
            self.log.info('Parsed title:%s'%title)
        except:
            self.log.exception('Error parsing title for url: %r'%self.url)
            title = None
        
        try:
            authors = self.parse_authors(root)
            self.log.info('Parsed authors:%s'%authors)
        except:
            self.log.exception('Error parsing authors for url: %r'%self.url)
            authors = []

        try:
            if real_authors_include or\
                dict([[tupl[0], tupl[1]] for tupl in append_to_tags]).get(cfg.REAL_AUTHORS, False) or\
                    cfg.REAL_AUTHORS in [tupl[0] for tupl in append_to_comments] or\
                    cfg.REAL_AUTHORS in [tupl[0] for tupl in title_line] or\
                    cfg.REAL_AUTHORS in [tupl[0] for tupl in publisher_line]:
                self.log.info('Real names needed, attempt to get them ...')
                real_authors = self.get_parse_authors_real_name(root)
            else:
                real_authors = []
        except Exception as e:
            self.log.exception('Getting author pages for real author names failed. %s - %s'%(self.url, e))
            real_authors = []
            
        try:
            illustrators = self.parse_illustrators(additional)
            self.log.info('Parsed illustrators:%s'%illustrators)
        except:
            self.log.exception('Error parsing illustrators for url: %r'%self.urlCall)
            illustrators = []
            
        try:
            cover_authors = self.parse_cover_authors(additional)
            self.log.info('Parsed cover authors:%s'%cover_authors)
        except:
            self.log.exception('Error parsing cover authors for url: %r'%self.urlCall)
            cover_authors = []
            
        try:
            translators = self.parse_translators(additional)
            self.log.info('Parsed translators:%s'%translators)
        except:
            self.log.exception('Error parsing translators for url: %r'%self.urlCall)
            translators = []

        if not title or not authors or not databazeknih_id:
            self.log.error('Could not find title/authors/databazeknih id for %r'%self.url)
            self.log.error('DK id: %r Title: %r Authors: %r'%(databazeknih_id, title, authors))
            return

        authors_to_add = []
        if authors_include:
            authors_to_add.extend(authors)
        if real_authors_include:
            authors_to_add.extend(real_authors)
        if translators_include and translators:
            authors_to_add.extend(translators)
        if illustrators_include and illustrators:
            authors_to_add.extend(illustrators)
        if cover_authors_include and cover_authors:
            authors_to_add.extend(cover_authors)
        
        if only_one_author == 1:
            authors_to_add = authors_to_add[:1]

        lookup = set()  # a temporary lookup set
        authors_to_add = [a for a in authors_to_add if a not in lookup and lookup.add(a) is None]
        if swap_authors:
            swapped = []
            for a in authors_to_add:
                auth_parts = a.split()
                swapped.append('%s %s' %(auth_parts[-1], ' '.join(auth_parts[:-1])))
            authors_to_add = swapped
        
        mi = Metadata(title, authors_to_add)
        if databazeknih_id and dict([[tupl[0], tupl[1]] for tupl in append_to_identifiers]).get(cfg.ORIG_DK, True):
            mi.set_identifier('databazeknih', databazeknih_id)
            self.databazeknih_id = databazeknih_id

        #serie edition
        try:
            series, series_index = self.parse_series(root)

            self.log.info('Parsed series:%s'%series)
            self.log.info('Parsed series index:%s'%series_index)
        except:
            self.log.exception('Error parsing series for url: %r'%self.url)
            series = series_index = None

        try:
            edition, edition_index = self.parse_editions(additional)
            self.log.info('Edition %s'%edition)
            self.log.info('Edition index %s'%edition_index)
        except:
            self.log.exception('Error parsing editions for url: %r'%self.url)
            edition = edition_index = None
        try:
            # 0: only series, 1: only editions, 2: prefer series, 3: prefer editions
            if series_editions == 0 and series:
                mi.series, mi.series_index = series, series_index
            elif series_editions == 1 and edition:
                mi.series, mi.series_index = edition, edition_index
            elif series_editions == 2:
                if series:
                    mi.series, mi.series_index = series, series_index
                elif edition:
                    mi.series, mi.series_index = edition, edition_index
            elif series_editions == 3:
                if edition:
                    mi.series, mi.series_index = edition, edition_index
                elif series: 
                    mi.series, mi.series_index = series, series_index            
        except:
            self.log.exception('Error adding series or editions into metadata: %r'%self.url)
            
        try:
            pages = self.parse_pages(additional)
        except:
            self.log.exception('Error parsing pages number for url: %r'%self.urlCall)
            pages = None

        try:
            cover_type = self.parse_cover_type(additional)
        except:
            self.log.exception('Error parsing cover type for url: %r'%self.urlCall)
            cover_type = None

        try:
            print_run = self.parse_print_run(additional)
        except:
            self.log.exception('Error parsing print run for url: %r'%self.urlCall)
            print_run = None
            
        try:
            original_title, original_year = self.parse_origTitleYear(root, additional)
        except:
            self.log.exception('Error parsing original title and year for url: %r'%self.url)
            original_title = original_year = None
            
        try:
            publisher = self.parse_publisher(root)
            self.log.info('Parsed publisher: %s'%publisher)
        except:
            publisher = None
            self.log.exception('Error parsing publisher for url: %r'%self.url)
            
        try:
            pubdate = self.parse_pubdate(root)
            self.log.info('Parsed pubdate:%s'%pubdate)
        except:
            pubdate = None
            self.log.exception('Error parsing pubdate for url: %r'%self.url)           
        
        try:
            # 0: found book pubdate, 1: first book pubdate
            if publication_date == 0 and pubdate:
                mi.pubdate = datetime(int(pubdate), 1, 1, tzinfo=utc_tz)
            elif publication_date == 1 and original_year:
                mi.pubdate = datetime(int(original_year), 1, 1, tzinfo=utc_tz)
        except:
            self.log.exception('Error adding published date.')
            
        try:
            if root_whole_rating and get_config_option(cfg.CALCULATE_RATING):
                rating_star, rating10, rating, rating_count = self.parse_recalculated_rating(root_whole_rating)
            else:
                rating_star, rating10, rating, rating_count = self.parse_rating(root)
            
            mi.rating = rating_star if rating_star else 0
            self.log.info('Parsed rating:%s'%mi.rating)
        except:
            rating_star = rating10 = rating = rating_count = None
            self.log.exception('Error parsing rating for url: %r'%self.url)
   
        try:
            isbn = self.parse_isbn(additional)
            #if isbn and parse_isbn:
            #    self.isbn = mi.isbn = isbn
        except:
            self.log.exception('Error parsing ISBN for url: %r'%self.urlCall)
            isbn = None

        if self.databazeknih_id:
            self.plugin.cache_isbn_to_identifier(isbn, self.databazeknih_id)
            
        try:
            genres = self.parse_genre(root)
            tags = self.parse_tags(root)
        except:
            self.log.exception('Error parsing tags or genres for url: %r'%self.url)
            tags = []
            genres = []
            
        try:
            awards = self.parse_awards(root)
        except:
            self.log.exception('Error parsing awards for url: %r'%self.url)
            awards = []

        try:
            trivia = self.parse_trivia(root)
        except:
            self.log.exception('Error parsing trivia for url: %r'%self.url)
            trivia = []

        try:
            if root_tales:
                tales_in_book = self.parse_tales_list(root_tales)
            else:
                tales_in_book = []
        except:
            self.log.exception('Error parsing tales_in_book.')
            tales_in_book = []
   
        try:
            mi.comments = ''
            comments = self.parse_comments(root)
        except:
            self.log.exception('Error parsing comments for url: %r'%self.url)            
        try:
            # append_to_comments - list of tuples
            # (id_string, check_bool, visible_desc)
            def get_comment_variable(argument):
                if argument == cfg.DESCRIPTION and comments:
                    return comments
                elif argument == cfg.HR:
                    return '<hr>'
                elif argument == cfg.PAGES and pages:
                    return '<p id="numberOfPages">Počet stran: %s</p>' %pages
                elif argument == cfg.COVER_TYPE and cover_type:
                    return '<p id="bookCoverType">Vazba knihy: %s</p>' %cover_type
                elif argument == cfg.PRINT_RUN and print_run:
                    return '<p id="printRun">Náklad (ks): %s</p>' %print_run
                elif argument == cfg.TITLE and title:
                    return '<p id="title">Název: %s</p>' %title
                elif argument == cfg.ORIGINAL_TITLE and original_title:
                    return '<p id="origTitle">Původní název: %s</p>' %original_title
                elif argument == cfg.ORIGINAL_YEAR and original_year:
                    return '<p id="origYear">Rok prvního vydání: %s</p>' %original_year
                elif argument == cfg.PUB_YEAR and pubdate:
                    return '<p id="pubYear">Rok vydání: %s</p>' %pubdate
                elif argument == cfg.PUBLISHER and publisher:
                    return '<p id="publisher">Vydavatel: %s</p>' %publisher
                elif argument == cfg.RATING and rating:
                    return '<p id="rating100">Hodnocení (%%): %s</p>' %rating
                elif argument == cfg.RATING10 and rating:
                    return '<p id="rating10">Hodnocení (0-10): %s</p>' %rating10
                elif argument == cfg.RATING5 and rating:
                    return '<p id="rating5">Hodnocení (0-5): %s</p>' %rating_star
                elif argument == cfg.RATING_COUNT and rating_count:
                    return '<p id="ratingCount">Počet hodnocení: %s</p>' %rating_count
                elif argument == cfg.ISBN and isbn:
                    return '<p id="isbn">ISBN: %s </p>' %isbn
                elif argument == cfg.ORIG_DK and self.databazeknih_id:
                    return '<p id="databazeknih">databazeknih: %s</p>' %self.databazeknih_id
                elif argument == cfg.SERIES and series:
                    if series_index:
                        return '<p id="series">Série: %s [%s]</p>' %(series, series_index)
                    else:
                        return '<p id="series">Série: %s</p>' %series
                elif argument == cfg.EDITION and edition:
                    if edition_index:
                        return '<p id="edition">Edice: %s [%s]</p>' %(edition, edition_index)
                    else:
                        return '<p id="edition">Edice: %s</p>' %edition
                elif argument == cfg.AUTHORS and authors:
                    return '<p id="authors">Autoři: %s</p>'%(' & '.join(authors))
                elif argument == cfg.REAL_AUTHORS and real_authors:
                    return '<p id="authorsRealName">Autoři: %s</p>'%(' & '.join(real_authors))
                elif argument == cfg.TRANSLATION and translators:
                    return '<p id="translator">Překlad: %s</p>'%(' & '.join(translators))
                elif argument == cfg.ILLUSTRATION and illustrators:
                    return '<p id="illustrators">Ilustrace: %s</p>'%(' & '.join(illustrators))
                elif argument == cfg.COVER_AUTHORS and cover_authors:
                    return '<p id="coverAuthors">Autoři obálky: %s</p>'%(' & '.join(cover_authors))
                elif argument == cfg.TAGS and tags:
                    return '<p id="tags">Štítky: %s</p>' %', '.join(tags)
                elif argument == cfg.GENRES and genres:
                    return '<p id="genres">Žánr: %s</p>' %', '.join(genres)
                elif argument == cfg.AWARDS and awards:
                    return '<p id="awards">Ocenění: <br>%s</p>' %'<br>'.join(awards)
                elif argument == cfg.TALES_IN_BOOK and tales_in_book:
                    return '<p id="bookParts">Části díla: <br>%s</p>' %'<br>'.join(tales_in_book)
                elif argument == cfg.TRIVIA and trivia:
                    return '<p id="trivia">Zajímavosti: <br>%s</p>' %'<br><br>'.join(trivia)
                else:
                    return ''
            
            self.log.info('append to comments:', append_to_comments)
            for item in append_to_comments:
                if item[1]:
                    argument = item[0]
                    # converts parsed information into html paragraph string                  
                    mi.comments = '%s %s'%(mi.comments, get_comment_variable(item[0]))
        except:
            self.log.exception('Error adding more info to comments for url: %r'%self.url)
            mi.comments = comments
                    
        try:
            def get_tag_variable(argument):
                switch_check = {
                    cfg.GENRES: genres,
                    cfg.TAGS: tags,
                    cfg.PAGES: pages,
                    cfg.COVER_TYPE: cover_type,
                    cfg.PRINT_RUN: print_run,
                    cfg.AUTHORS: authors,
                    cfg.REAL_AUTHORS: real_authors,
                    cfg.TRANSLATION: translators,
                    cfg.ILLUSTRATION: illustrators,
                    cfg.COVER_AUTHORS: cover_authors,
                    cfg.TITLE: title,
                    cfg.ORIGINAL_TITLE: original_title,
                    cfg.ORIGINAL_YEAR: original_year,
                    cfg.PUB_YEAR: pubdate,
                    cfg.PUBLISHER: publisher,
                    cfg.SERIES: (series, series_index),
                    cfg.EDITION: (edition, edition_index),
                    cfg.RATING: rating,
                    cfg.RATING5: rating_star,
                    cfg.RATING10: rating10,
                    cfg.RATING_COUNT: rating_count,
                    cfg.ISBN: isbn,
                    cfg.ORIG_DK: self.databazeknih_id,
                    cfg.AWARDS: awards,
                    cfg.TALES_IN_BOOK: tales_in_book
                }
                switch = {
                    cfg.GENRES: genres,
                    cfg.TAGS: tags,
                    cfg.PAGES: ['Počet stran: %s' %pages],
                    cfg.COVER_TYPE: ['Vazba knihy: %s' %cover_type],
                    cfg.PRINT_RUN: ['Náklad (ks): %s' %print_run],
                    cfg.ORIGINAL_YEAR: ['Rok prvního vydání: %s' %original_year],
                    cfg.PUB_YEAR: ['Rok vydání: %s' %pubdate],
                    cfg.RATING: ['Hodnocení (%%): %s' %rating],
                    cfg.RATING5: ['Hodnocení (0-5): %s' %rating_star],
                    cfg.RATING10: ['Hodnocení (0-10): %s' %rating10],
                    cfg.RATING_COUNT: ['Počet hodnocení: %s' %rating_count],
                    cfg.ISBN: ['ISBN: %s' %isbn],
                    cfg.ORIG_DK: ['databazeknih: %s' %self.databazeknih_id]
                }
                if switch_check.get(argument, None):
                    if argument == cfg.TITLE and title:
                        return ['Název: %s' %title.replace(',', ';')]
                    if argument == cfg.PUBLISHER and publisher:
                        return ['Vydavatel: %s' %publisher.replace(',', ';')]
                    if argument == cfg.ORIGINAL_TITLE and original_title:
                        return ['Původní název: %s' %original_title.replace(',', ';')]
                    elif argument == cfg.EDITION and edition:
                        if edition_index:
                            return ['Edice: %s [%s]' %(edition, edition_index)]
                        else:
                            return ['Edice: %s' %edition]
                    elif argument == cfg.SERIES and series:
                        if series_index:
                            return ['Série: %s [%s]' %(series, series_index)]
                        else:
                            return ['Série: %s' %series]
                    elif argument == cfg.AUTHORS and authors:
                        return ['Autor: %s'%a.replace(',', ';') for a in authors]
                    elif argument == cfg.REAL_AUTHORS and real_authors:
                        return ['Autor (r): %s'%a.replace(',', ';') for a in real_authors]
                    elif argument == cfg.TRANSLATION and translators:
                        return ['Překlad: %s'%t.replace(',', ';') for t in translators]
                    elif argument == cfg.ILLUSTRATION and illustrators:
                        return ['Ilustrace: %s'%i.replace(',', ';') for i in illustrators]
                    elif argument == cfg.COVER_AUTHORS and cover_authors:
                        return ['Autor obálky: %s'%a.replace(',', ';') for a in cover_authors]
                    elif argument == cfg.AWARDS and awards:
                        return ['Ocenění: %s'%a.replace(',', ';') for a in awards]
                    elif argument == cfg.TALES_IN_BOOK and tales_in_book:
                        return ['Část díla: %s'%t.replace(',', ';') for t in tales_in_book]
                    else:
                        return switch.get(argument, '')
                return ''

            for item in append_to_tags:
                if item[1]:
                    mi.tags.extend(get_tag_variable(item[0]))
        except:
            self.log.exception('Error adding additional tags for url: %r'%self.url)

        
        try:
            def get_identifier_variable(argument):
                switch_check = {
                    cfg.GENRES: genres,
                    cfg.TAGS: tags,
                    cfg.PAGES: pages,
                    cfg.COVER_TYPE: cover_type,
                    cfg.PRINT_RUN: print_run,
                    cfg.AUTHORS: authors,
                    cfg.REAL_AUTHORS: real_authors,
                    cfg.TRANSLATION: translators,
                    cfg.ILLUSTRATION: illustrators,
                    cfg.COVER_AUTHORS: cover_authors,
                    cfg.TITLE: title,
                    cfg.ORIGINAL_TITLE: original_title,
                    cfg.ORIGINAL_YEAR: original_year,
                    cfg.PUB_YEAR: pubdate,
                    cfg.PUBLISHER: publisher,
                    cfg.SERIES: (series, series_index),
                    cfg.EDITION: (edition, edition_index),
                    cfg.RATING: rating,
                    cfg.RATING5: rating_star,
                    cfg.RATING10: rating10,
                    cfg.RATING_COUNT: rating_count,
                    cfg.ISBN: isbn,
                    cfg.ORIG_DK: self.databazeknih_id,
                    cfg.AWARDS: awards,
                    cfg.TALES_IN_BOOK: tales_in_book
                }
                return switch_check.get(argument, '')
            for item in append_to_identifiers:
                appending = get_identifier_variable(item[0])
                if item[0] in [cfg.GENRES, cfg.TAGS, cfg.AWARDS, cfg.TALES_IN_BOOK] and appending:
                    appending = '| '.join([a.replace(',', ';') for a in appending])
                elif item[0] in [cfg.AUTHORS, cfg.REAL_AUTHORS, cfg.TRANSLATION, cfg.ILLUSTRATION, cfg.COVER_AUTHORS] and appending:
                    appending = '| '.join([a.replace(',', ';') for a in appending])
                elif item[0] in [cfg.SERIES, cfg.EDITION] and appending:
                    if appending[0] and appending[1]:
                        appending = ' '.join([appending[0].replace(',', '|'), str(appending[1])])
                    elif appending[0]:
                        appending = appending[0].replace(',', '|')
                    else:
                        appending = None
                elif isinstance(appending, str) and appending:
                    appending = appending.replace(',', '|')
                if item[1] and appending:
                    mi.identifiers[item[3]] = str(appending)
        except:
            self.log.exception('Error adding extra identifiers for url: %r'%self.url)

        def get_metadata_variable(argument):
            switch_check = {
                cfg.GENRES: ', '.join(genres),
                cfg.TAGS: ', '.join(tags),
                cfg.PAGES: pages,
                cfg.COVER_TYPE: cover_type,
                cfg.PRINT_RUN: print_run,
                cfg.AUTHORS: ' & '.join(authors),
                cfg.REAL_AUTHORS: ' & '.join(real_authors),
                cfg.ILLUSTRATION: ' & '.join(illustrators),
                cfg.TRANSLATION: ' & '.join(translators),
                cfg.COVER_AUTHORS: ' & '.join(cover_authors),
                cfg.TITLE: title,
                cfg.ORIGINAL_TITLE: original_title,
                cfg.ORIGINAL_YEAR: original_year,
                cfg.PUB_YEAR: pubdate,
                cfg.PUBLISHER: publisher,
                cfg.SERIES: series,
                cfg.SERIES_INDEX: series_index,
                cfg.EDITION: edition,
                cfg.EDITION_INDEX: edition_index,
                cfg.RATING: rating,
                cfg.RATING5: rating_star,
                cfg.RATING10: rating10,
                cfg.RATING_COUNT: rating_count,
                cfg.ISBN: isbn,
                cfg.ORIG_DK: self.databazeknih_id,
                cfg.AWARDS: awards,
                cfg.TALES_IN_BOOK: tales_in_book,
                cfg.CUSTOM_TEXT: argument
            }
            return switch_check.get(argument, None)
            
        try:
            mi.publisher = ''
            for item in publisher_line:
                appending = get_metadata_variable(item[0])

                if appending == cfg.CUSTOM_TEXT and item[1]:
                    mi.publisher += str(item[2])
                elif item[1] and appending:
                    mi.publisher += str(appending)
        except:
            mi.publisher = publisher if publisher else None
            self.log.exception('Error parsing publisher for url: %r'%self.url)

        try:
            mi.title = ''
            for item in title_line:
                appending = get_metadata_variable(item[0])

                if appending == cfg.CUSTOM_TEXT and item[1]:
                    mi.title += str(item[2])
                elif item[1] and appending:
                    mi.title += str(appending)
        except:
            mi.title = title if title else None
            self.log.exception('Error parsing title for url: %r'%self.url)

    
        try:
            if isbn and obalkyknih_cover:
                obalky_cover_url = self.get_parse_obalkyknih_cover(isbn)
            else:
                obalky_cover_url = None

            self.cover_url = self.parse_cover(root)
            if max_covers == 0:
                mi.has_cover = False
            elif max_covers > 1 and self.cover_urls:
                if obalky_cover_url:
                    self.cover_urls.append(obalky_cover_url)
                self.plugin.cache_identifier_to_cover_url(self.databazeknih_id, self.cover_urls)
                if isbn:
                    self.plugin.cache_isbn_to_identifier(isbn, self.databazeknih_id)
                mi.has_cover = True if self.cover_urls else False
            else:
                self.log.info('Parsed URL for cover:%r'%self.cover_url)
                if obalky_cover_url:
                    self.plugin.cache_identifier_to_cover_url(self.databazeknih_id, [self.cover_url, obalky_cover_url])
                else:
                    self.plugin.cache_identifier_to_cover_url(self.databazeknih_id, [self.cover_url])
                if isbn:
                    self.plugin.cache_isbn_to_identifier(isbn, self.databazeknih_id)
                mi.cover_url = self.cover_url
                mi.has_cover = bool(self.cover_url)
        except:
            self.log.exception('Error parsing cover for url: %r'%self.url)
             
        try:
            mi.languages = self.parse_languages(additional)
            self.log.info('Parsed language:%s'%mi.languages)
        except:
            self.log.exception('Error parsing language for url: %r'%self.urlCall)
            
        mi.source_relevance = self.relevance
        # self.plugin.clean_downloaded_metadata(mi)
        self.log.info(mi)
        self.result_queue.put(mi)        

    def parse_first(self, root, xpath, loginfo, convert=lambda x: x[0].strip()):
        try:
            nodes = root.xpath(xpath)
            self.log.info('Found %s: %s' % (loginfo,nodes))
            return convert(nodes) if nodes else None
        except Exception as e:
            self.log.exception('Error parsing for %s with xpath: %s' % (loginfo, xpath))

    def parse_all(self, root, xpath, loginfo, convert=lambda x: [node.strip() for node in x]):
        try:
            if isinstance(xpath, list):
                self.log.info('Multiple paths.. %s'%xpath)
                all_nodes = []
                for path in xpath:
                    nodes = root.xpath(path)
                    self.log.info('Found %s: %s' % (loginfo,','.join(nodes)))
                    if nodes:
                        all_nodes.extend(nodes)
                return convert(all_nodes) if all_nodes else []
            else:
                nodes = root.xpath(xpath)
                self.log.info('Found %s: %s' % (loginfo,','.join(nodes)))
                return convert(nodes) if nodes else []
        except Exception:
            self.log.exception('Error parsing for %s with xpath: %s' % (loginfo, xpath))

    def parse_databazeknih_id(self, root):
        original_id = self.parse_first(root, '//meta[@property="og:url"]/@content', 'databazeknih_id_original', convert=lambda x: x[0].strip().split('/')[-1])
        #short_id = self.parse_first(root, '//*[@bid]/@bid', 'databazeknih_id_short')
        short_id = original_id.split('-')[-1] if original_id else None # universal, can be used for tales too
        return (original_id, short_id) if original_id else (None, None)

    def parse_isbn(self, additional):
        return self.parse_first(additional, '//span[@itemprop="isbn"]/text()', 'ISBN', lambda x: x[0].split(',')[0].strip())

    def parse_title(self, root):
        return self.parse_first(root,'//h1[@itemprop="name"]/text()','title', lambda x: x[0].replace('&nbsp;','').strip())

    def parse_pages(self, additional):
        return self.parse_first(additional, '//span[@itemprop="numberOfPages"]/text()', 'numberOfPages')

    def parse_pubdate(self, root):
        return self.parse_first(root, '//span[@itemprop="datePublished"]/text()', 'pubdate', lambda x: x[0].strip())

    def parse_publisher(self, root):
        publisher = self.parse_first(root, '//span[@itemprop="publisher"]/a/text()', 'publisher')
        filter_check = get_config_option(cfg.PUBLISHER_FILTER)
        if publisher:
            publisher = self._convert_to_calibre(publisher, get_config_option(cfg.KEY_PUBLISHER_MAPPINGS), filter_check)
        return publisher
    
    def parse_series(self, root):
        series_node = self.parse_first(root,'//h3/a[contains(@href, "serie")]/text()','series', lambda x: x[0].replace('&nbsp;','&').strip())
        series_localization = get_config_option(cfg.SERIES_LOCALIZATION)
        filter_check = get_config_option(cfg.SERIES_FILTER)
        if series_node:
            series_split = series_node.split(' / ')
            if len(series_split) == 2:
                if series_localization == 1:
                    series_node = series_split[0]
                elif series_localization == 2:
                    series_node = series_split[1]
            else:
                series_split = series_node.split('/')
                if len(series_split) == 2:
                    if series_localization == 1:
                        series_node = series_split[0]
                    elif series_localization == 2:
                        series_node = series_split[1]
            series_node = self._convert_to_calibre(series_node, get_config_option(cfg.KEY_SERIES_MAPPINGS), filter_check)
        index = self.parse_first(root, '//span[contains(@class, "nowrap")]/span[contains(@class, "odright_pet") or contains(@class, "odleft_pet")]/text()', 'series_index', lambda x: float(x[0].split('.')[0]))
        return series_node, index
        
    def parse_editions(self, additional):
        series_node = self.parse_first(additional,'//a[@itemprop="bookEdition"]/text()','edition', lambda x: x[0].replace('&nbsp;','&').strip())
        filter_check = get_config_option(cfg.SERIES_FILTER)
        if series_node:
            series_node = self._convert_to_calibre(series_node, get_config_option(cfg.KEY_SERIES_MAPPINGS), filter_check)
        index = self.parse_first(additional, '//a[@itemprop="bookEdition"]/following-sibling::em/text()', 'edition_index', lambda x: float(x[0][1:-2]))
        return series_node, index
    
    def parse_origTitleYear(self, root, additional):
        origTitle = self.parse_first(root, '//span[contains(text(), "riginál")]/following-sibling::h4/text()', 'orig_title')
        if origTitle:
            origYear = self.parse_first(root, '//span[contains(text(), "riginál")]/following-sibling::h4/text()[position() = 2]', 'orig_year', lambda x: x[0].strip())
        else:
            origYear = self.parse_first(additional, '//span[contains(text(), "vydání originálu")]/following-sibling::text()', 'orig_year', lambda x: x[0].strip())
        return origTitle, origYear

    def parse_rating(self, root):
        rating_node = self.parse_first(root, '//a[@class="bpoints"]/div/text()', 'rating_percent', lambda x: int(x[0].strip()))
        star_rating = self.parse_first(root, '//span[@itemprop="ratingValue"]/text()', 'rating_star', lambda x: float(x[0].strip()))
        rating_count = self.parse_first(root, '//a[@class="ratingDetail"]/text()', 'rating_count', lambda x: int(x[0].strip().replace(' hodnocení', '').replace(' ', '')))
        return star_rating, star_rating*2 if star_rating else None, rating_node, rating_count

    def parse_languages(self, additional):
        language_nodes = self.parse_all(additional, '//span[@itemprop="language"]/text()', 'languages')
        lang_mapping = {'český': 'cs',
                        'slovenský': 'sk',
                        'německý': 'de',
                        'polský': 'pl',
                        'anglický': 'en',
                        'francouzský': 'fr',
                        'španělský': 'es',
                        'italský': 'it'}
        if language_nodes:
            return [lang_mapping[k] for k in language_nodes if k in lang_mapping]
        
    def parse_authors(self, root):
        return self.parse_all(root, '//h2[@class="jmenaautoru"]/span[@itemprop="author"]/a/text()', 'authors', lambda x: list({node.strip() for node in x}))

    def parse_author_real_name(self, author_root):
        return self.parse_first(author_root, '//div[@id="left_less"]/h1/text()', 'real_name_author')

    def parse_authors_nationality(self, root):
        nationality_node = self.parse_all(root, '//div/span[@class="pozn_light"][last()]/text()[1]', 'nationality_authors', lambda x: list({node.strip() for node in x}))
        opt_one_author = get_config_option(cfg.ONE_AUTHOR)
        
        if opt_one_author == 1 and nationality_node:
            nationality_node = nationality_node[:1]
        return nationality_node

    def parse_translators(self, additional):
        return self.parse_all(additional, '//a[contains(@href, "prekladatele/")]/text()', 'translators', lambda x: list({node.strip() for node in x}))

    def parse_illustrators(self, root):
        return self.parse_all(root, '//span[@itemprop="ilustrator"]/a/text()', 'illustrators', lambda x: list({node.strip() for node in x}))

    def parse_cover_authors(self, root):
        return self.parse_all(root, '//a[contains(@href, "autori-obalek")]/text()', 'cover_authors', lambda x: list({node.strip() for node in x}))

    def parse_genre(self, root):
        tags = self.parse_all(root, '//h5[@itemprop="genre"]/a/text()', 'genre')
        filter_check = get_config_option(cfg.GENRE_FILTER)
        if tags:
            calibre_tags = self._convert_genres_to_calibre_tags(tags, filter_check)
            if len(calibre_tags) > 0:
                tags = calibre_tags
        return tags

    def parse_tags(self, root):
        tags = self.parse_all(root, '//p[@class="new2"]/a[contains(@href, "stitky/")]/text()', 'tags')
        filter_check = get_config_option(cfg.GENRE_FILTER)
        if tags:
            calibre_tags = self._convert_genres_to_calibre_tags(tags, filter_check)
            if len(calibre_tags) > 0:
                tags = calibre_tags
        return tags

    def _convert_genres_to_calibre_tags(self, genre_tags, filter_check=False):
        # for each tag, add if we have a dictionary lookup
        calibre_tag_lookup = cfg.plugin_prefs[cfg.STORE_NAME][cfg.KEY_GENRE_MAPPINGS]
        calibre_tag_map = dict((k.lower(),v) for (k,v) in calibre_tag_lookup.items())
        tags_to_add = list()
        for genre_tag in genre_tags:
            if genre_tag.lower() in calibre_tag_map:
                tags = calibre_tag_map.get(genre_tag.lower(), None)
                if tags:
                    for tag in tags:
                        if tag not in tags_to_add:
                            tags_to_add.append(tag.replace(',', ';'))
            elif filter_check:
                continue
            else: 
                tag = genre_tag
                if tag not in tags_to_add:
                            tags_to_add.append(tag.replace(',', ';'))
            
        return list(tags_to_add)

    def	_convert_to_calibre(self, remap_item, prefs, filter_check=False):
        # for each tag, add if we have a dictionary lookup
        calibre_map = dict((k.lower(),v) for (k,v) in prefs.items())
        self.log.info('calibre_map: %s'%calibre_map)
        calibre_remap_item = calibre_map.get(remap_item.lower(), None)
        if calibre_remap_item:
            return calibre_remap_item[0]
        elif filter_check:
            return None
        else:
            return remap_item

    def parse_comments(self, root):
        return self.parse_all(root, ['//p[contains(@class, "justify new2")]/text()', '//span[contains(@class, "end_text")]/text()'], 'description', lambda x: '<p id="description">{}</p>'.format(''.join(x)).replace('\n', '<br>'))

    def parse_awards(self, root):
        award_node = root.xpath('//a[contains(@href, "literarni-ceny/")]/text()')
        category_node = root.xpath('//a[contains(@href, "literarni-ceny/")]/following-sibling::text()[1]')
        year_node = root.xpath('//a[contains(@href, "literarni-ceny/")]/preceding-sibling::strong/text()')
        awards = []
        if award_node and category_node and year_node and len(award_node) == len(category_node):
            for i, k in enumerate(award_node):
                awards.append('%s %s [%s]'%(award_node[i], category_node[i].strip(), year_node[i]))
        elif len(award_node) != len(category_node):
            for i, k in enumerate(award_node):
                awards.append('%s [%s]'%(award_node[i], year_node[i]))
        self.log.info('Parsed awards: %s'%awards)
        return awards

    def parse_cover(self, root):
        return self.parse_first(root, '//img[@class="kniha_img"]/@src', 'cover', lambda x: x[0].strip().replace('bmid', 'big').replace('mid', 'big'))

    def parse_tales_list(self, root):
        tale_title = self.parse_all(root, '//a[contains(@href, "/povidky/")]/text()', 'tales_in_book_title')
        tale_origYear = self.parse_all(root, '//span[@class="pozn odleft_pet"]/text()', 'tales_in_book_year')
        tales = []
        if tale_title and tale_origYear and len(tale_title) == len(tale_origYear):
            for i, tt in enumerate(tale_title):
                tales.append('%s [%s]'%(tale_title[i], tale_origYear[i]))
        elif tale_title and tale_origYear and len(tale_title) != len(tale_origYear):
            tales = tale_title
        self.log.info('Parsed tales in book: %s'%tales)
        return tales

    def parse_cover_type(self, root):
        return self.parse_first(root, '//span[contains(text(), "Vazba knihy")]/following-sibling::text()', 'book_cover_type', convert=lambda x: x[0].strip().replace(',', ';'))

    def parse_print_run(self, root):
        return self.parse_first(root, '//span[contains(text(), "Náklad")]/following-sibling::text()', 'book_print_run', convert=lambda x: x[0].strip().replace(' ks', '').replace(' ', ''))

    def parse_trivia(self, root):
        trivia_node = root.xpath('//h2[contains(text(), "Zajímavosti")]/following-sibling::p[@class="new2"]')
        trivia_final = []
        if trivia_node:
            for one in trivia_node:
                one_parsed = self.parse_all(one, './/text()[not(ancestor::a[@class="gray smaller"]) and not(ancestor::p/a[contains(@href, "/literarni-ceny/")])]', 'trivia', convert=lambda x: ' '.join([node.strip() for node in x if node.strip()]))
                if one_parsed:
                    trivia_final.append(one_parsed)
        return trivia_final

    def parse_tale(self, root):
        '''
        When parsing into Calibre as Book record:
        databazeknih info           > Calibre data field
        ---------------------------------------------
        title                       > Title
        author(s)                   > Authors
        books containing this tale  > Tags
        percent rating              > 5 star Rating
        description                 > Comments
        original publication year   > Published date
        original title              > Publisher     
        '''
        try:
            self.log.info('Parse tale/poem details:%s'%self.url)
            databazeknih_id, databazeknih_id_short = self.parse_databazeknih_id(root)
            self.log.info('Parsed DK tale/poem identifier:%s'%databazeknih_id)
            self.log.info('Parsed DK tale/poem short:%s'%databazeknih_id_short)
        except:
            self.log.exception('Error parsing databazeknih tale/poem id for url: %r'%self.url)
            databazeknih_id = databazeknih_id_short = None

        try:
            title = self.parse_first(root, '//h2[@class="jmenaautoru"]/preceding-sibling::h1/text()', 'tale_title')
            self.log.info('Parsed title:%s'%title)
        except:
            self.log.exception('Error parsing title for url: %r'%self.url)
            title = None

        try:
            authors = self.parse_all(root, '//h2[@class="jmenaautoru"]/a[contains(@href, "/autori/")]/text()', 'tale_authors')
            self.log.info('Parsed authors:%s'%authors)
        
            opt_one_author = get_config_option(cfg.ONE_AUTHOR)
            if opt_one_author == 1 and authors:
                authors = authors[:1]

            swap_authors = get_config_option(cfg.SWAP_AUTHORS)
            if swap_authors and authors:
                swapped = []
                for a in authors:
                    auth_parts = a.split()
                    swapped.append('%s %s' %(auth_parts[-1], ' '.join(auth_parts[:-1])))
                authors = swapped
        except:
            self.log.exception('Error parsing authors for url: %r'%self.url)
            authors = []

        mi = Metadata(title, authors)
        mi.set_identifier('databazeknih_povidka', databazeknih_id)
        self.databazeknih_id = databazeknih_id

        try:
            books = self.parse_all(root, '//h2[@class="jmenaautoru"]/a[contains(@href, "/knihy/")]/text()', 'tale_books')
            self.log.info('Parsed books:%s'%books)
            if books:
                mi.tags = [b.replace(',', ';') for b in books]
        except:
            self.log.exception('Error parsing tale books for url: %r'%self.url)
            books = []

        try:
            percent_rating = self.parse_first(root, '//div[contains(@class, "hodnoceni")]/text()', 'tale_rating_percent', convert=lambda x: int(x[0].strip()))
            star_rating = 5*(percent_rating/100) if percent_rating else None
            self.log.info('Parsed star_rating:%s'%star_rating)

            mi.rating = star_rating
        except:
            self.log.exception('Error parsing tale rating for url: %r'%self.url)
            percent_rating = star_rating = None
        
        try:
            description = self.parse_first(root, '//p[@class="justify new2"]/text()', 'tale_description', convert=lambda x: '<p id="description">%s</p>'%x[0].strip())
            self.log.info('Parsed tale description:%s'%description)

            mi.comments = description
        except:
            self.log.exception('Error parsing tale description for url: %r'%self.url)
            description = None

        try:
            origYear = self.parse_first(root, '//div[@class="detail_description"]/span[contains(text(), "Rok vydání")]/following-sibling::text()', 'tale_origYear')
            self.log.info('Parsed tale origYear:%s'%origYear)

            if origYear:
                mi.pubdate = datetime(int(origYear), 1, 1, tzinfo=utc_tz)
        except:
            self.log.exception('Error parsing tale origYear for url: %r'%self.url)
            origYear = None

        try:
            origTitle = self.parse_first(root, '//div[@class="detail_description"]/span[contains(text(), "Originální název")]/following-sibling::h3/text()', 'tale_origTitle')
            self.log.info('Parsed tale origTitle:%s'%origTitle)
            if origTitle:
                mi.publisher = origTitle
        except:
            self.log.exception('Error parsing tale origTitle for url: %r'%self.url)
            origTitle = None

        mi.source_relevance = self.relevance
        self.log.info(mi)
        self.result_queue.put(mi)

def strip_accents(s):
        if s is None: # or s is not str:
            return s
        else:
            symbols = (u"öÖüÜóÓőŐúÚůŮéÉěĚáÁűŰíÍýÝąĄćĆčČęĘłŁńŃóÓśŚšŠźŹżŻžŽřŘďĎťŤňŇ\t @#$?%ˇ´˝¸~^˘°|/*()[]{}:<>.,;¨˛`·'_\"\\",
                       u"oOuUoOoOuUuUeEeEaAuUiIyYaAcCcCeElLnNoOsSsSzZzZzZrRdDtTnN--------------------------------------")
            tr = dict( [ (ord(a), ord(b)) for (a, b) in zip(*symbols) ] )
            return s.translate(tr)