#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2019, Daniel Prazak <kret33n@gmail.com>'
__docformat__ = 'restructuredtext cs'

import copy
from functools import partial

try:
    from qt.core import  Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidgetItem, QTableWidget, \
                        QAbstractItemView, QIcon, QInputDialog, QTabWidget, QWidget
except ImportError:
    try:
        from PyQt5.Qt import Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidgetItem, QTableWidget, \
                        QAbstractItemView, QIcon, QInputDialog, QTabWidget, QWidget
    except ImportError:
        from PyQt4.Qt import Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidgetItem, QTableWidget, \
                        QAbstractItemView, QIcon, QInputDialog, QTabWidget, QWidget

from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
from calibre.utils.config import JSONConfig

# python/pyqt backwards compability
from calibre import as_unicode
try:
    from calibre.gui2 import QVariant
    del QVariant
except ImportError:
    is_qt4 = False
    convert_qvariant = lambda x: x
else:
    is_qt4 = True
    def convert_qvariant(x):
        vt = x.type()
        if vt == x.String:
            return as_unicode(x.toString())
        if vt == x.List:
            return [convert_qvariant(i) for i in x.toList()]
        return x.toPyObject()

STORE_NAME               = 'Options'
KEY_MAX_DOWNLOADS        = 'maxDownloads'
MAX_COVERS               = 'maxCoverDownloads'
OBALKYKNIH_COVER         = 'obalkyKnihCover'
#CUSTOM_ID                = 'customID'
IDENTIFIER_SEARCH        = 'identifierSearch'
ISBN_SEARCH              = 'isbnSearch'
TALES_SEARCH             = 'talesSearch'
ISSUE_PREFERENCE         = 'issuePreference'
GOOGLE_SEARCH            = 'googleSearchEngine'

AUTHORS_INCLUDE          = 'authorsInclude'
REAL_AUTHS_INCLUDE       = 'realAuthorsNameInclude'
TRANSLATORS_INCLUDE      = 'translatorsInclude'
ILLUSTRATORS_INCLUDE     = 'illustratorsInclude'
COVER_AUTHORS_INCLUDE    = 'coverAuthorsInclude'
ONE_AUTHOR               = 'oneAuthor'

APPEND_TO_COMMENTS       = 'appendToComments'
APPEND_TO_TAGS           = 'appendToTags'
APPEND_TO_TAG            = 'appendToTag'
APPEND_TO_IDENTIFIERS    = 'appendToIdentifiers'
DESCRIPTION              = 'description'
HR                       = 'hr'
APPEND_PLACE             = 'appendPlace'
PAGES                    = 'appendPages'
COVER_TYPE               = 'appendCoverType'
PRINT_RUN                = 'appendPrintRun'
ORIGINAL_TITLE           = 'appendOriginalTitle'
TITLE                    = 'appendTitle'
ORIGINAL_YEAR            = 'appendOriginalYear'
PUB_YEAR                 = 'appendPubYear'
PUBLISHER                = 'appendPublisher'
AUTHORS                  = 'appendAuthors'
REAL_AUTHORS             = 'appendRealAuthors'
TRANSLATION              = 'appendTranslation'
ILLUSTRATION             = 'appendIllustration'
COVER_AUTHORS            = 'appendCoverAuthors'
SERIES                   = 'appendSeries'
SERIES_INDEX             = 'appendSeriesIndex'
EDITION                  = 'appendEdition'
EDITION_INDEX            = 'appendEditionIndex'
GENRES                   = 'appendGenres'
TAGS                     = 'appendTags'
RATING                   = 'appendRating'
RATING5                  = 'appendRating5'
RATING10                 = 'appendRating10'
RATING_COUNT             = 'appendRatingCount'
ISBN                     = 'appendIsbn'
ORIG_DK                  = 'appendOrigDK'
AWARDS                   = 'awards'
TALES_IN_BOOK            = 'appendTalesInBook'
TRIVIA                   = 'appendTrivia'
IDENTIFIER_STYLE         = 'identifierStyle'

SERIES_EDITIONS          = 'seriesEditions'
SERIES_LOCALIZATION      = 'seriesLocalization'
PUBLICATION_DATE         = 'publicationDate'
PARSE_ISBN               = 'parseIsbn'
PARSE_DATABAZEKNIH       = 'parseDatabazeknih'
SWAP_AUTHORS             = 'swapAuthors'
CALCULATE_RATING         = 'calculatedRating'
CALCULATE_RATING_ACCURACY = 'calculatedRatingAccuracy'

KEY_GENRE_MAPPINGS       = 'genreMappings'
KEY_SERIES_MAPPINGS      = 'seriesMappings'
KEY_PUBLISHER_MAPPINGS   = 'publisherMappings'
GENRE_FILTER             = 'genreFilter'
SERIES_FILTER            = 'seriesFilter'
PUBLISHER_FILTER         = 'publisherFilter'
TITLE_LINE               = 'customTitle'
PUBLISHER_LINE           = 'customPublisher'
CUSTOM_TEXT              = 'customText'

DEFAULT_GENRE_MAPPINGS = {
                'Architektura': ['Architektura'],
                'Biografie a memoáry': ['Biografie a memoáry'],
                'Cestopisy a místopisy': ['Cestopisy a místopisy'],
                'Citáty a přísloví': ['Citáty a přísloví'],
                'Detektivky, krimi': ['Detektivky; krimi'],
                'Divadelní hry': ['Divadelní hry'],
                'Dívčí romány': ['Dívčí romány'],
                'Dobrodružné': ['Dobrodružné'],
                'Doprava': ['Doprava'],
                'Duchovní literatura': ['Duchovní literatura'],
                'Dům a byt': ['Dům a byt'],
                'Ekologie, živ. prostředí': ['Ekologie; živ. prostředí'],
                'Ekonomie a obchod': ['Ekonomie a obchod'],
                'Encyklopedie': ['Encyklopedie'],
                'Erotika': ['Erotika'],
                'Esoterika, astrologie, okultismus': ['Esoterika; astrologie; okultismus'],
                'Fantasy': ['Fantasy'],
                'Fejetony, eseje': ['Fejetony; eseje'],
                'Filozofie': ['Filozofie'],
                'Fotopublikace': ['Fotopublikace'],
                'Gamebook': ['Gamebook'],
                'Historické romány': ['Historické romány'],
                'Historie': ['Historie'],
                'Hobby': ['Hobby'],
                'Horory': ['Horory'],
                'Hudebniny': ['Hudebniny'],
                'Humor': ['Humor'],
                'Jazyky, lingvistika': ['Jazyky; lingvistika'],
                'Komiksy': ['Komiksy'],
                'Kuchařky': ['Kuchařky'],
                'Literatura česká': ['Literatura česká'],
                'Literatura faktu': ['Literatura faktu'],
                'Literatura naučná': ['Literatura naučná'],
                'Literatura slovenská': ['Literatura slovenská'],
                'Literatura světová': ['Literatura světová'],
                'Mapy a atlasy': ['Mapy a atlasy'],
                'Matematika a logika': ['Matematika a logika'],
                'Mytologie': ['Mytologie'],
                'Náboženství': ['Náboženství'],
                'New Age': ['New Age'],
                'Novely': ['Novely'],
                'O literatuře': ['O literatuře'],
                'Omalovánky': ['Omalovánky'],
                'Osobní rozvoj a styl': ['Osobní rozvoj a styl'],
                'PC literatura': ['PC literatura'],
                'Poezie': ['Poezie'],
                'Pohádky': ['Pohádky'],
                'Politologie, mezinárodní vztahy': ['Politologie; mezinárodní vztahy'],
                'Pověsti': ['Pověsti'],
                'Povídky': ['Povídky'],
                'Pragensie': ['Pragensie'],
                'Právo': ['Právo'],
                'Pro děti a mládež': ['Pro děti a mládež'],
                'Pro nejmenší': ['Pro nejmenší'],
                'Pro ženy': ['Pro ženy'],
                'Průmysl': ['Průmysl'],
                'Příběhy': ['Příběhy'],
                'Příroda, zvířata': ['Příroda; zvířata'],
                'Přírodní vědy': ['Přírodní vědy'],
                'Psychologie a pedagogika': ['Psychologie a pedagogika'],
                'Rodina': ['Rodina'],
                'Romány': ['Romány'],
                'Sci-fi': ['Sci-fi'],
                'Sociologie': ['Sociologie'],
                'Sport': ['Sport'],
                'Šikovné děti - tvoření, hry, styl': ['Šikovné děti - tvoření; hry; styl'],
                'Technika a elektro': ['Technika a elektro'],
                'Thrillery': ['Thrillery'],
                'Turistický průvodce': ['Turistický průvodce'],
                'Učebnice a slovníky': ['Učebnice a slovníky'],
                'Umění': ['Umění'],
                'Válečné': ['Válečné'],
                'Věda': ['Věda'],
                'Vesmír': ['Vesmír'],
                'Vojenství': ['Vojenství'],
                'Vzdělávání': ['Vzdělávání'],
                'Záhady': ['Záhady'],
                'Zahrada': ['Zahrada'],
                'Zdraví': ['Zdraví'],
                'Zdravotnictví': ['Zdravotnictví'],
                'Zemědělství': ['Zemědělství'],
                'Žurnalistika, publicistika': ['Žurnalistika; publicistika']
                }

DEFAULT_STORE_VALUES = {
    KEY_MAX_DOWNLOADS: 10,
    MAX_COVERS: 1,
    OBALKYKNIH_COVER: False,
    #CUSTOM_ID: '',
    IDENTIFIER_SEARCH: True,
    ISBN_SEARCH: True,
    TALES_SEARCH: False,
    GOOGLE_SEARCH: False,
    ISSUE_PREFERENCE: 0,
    AUTHORS_INCLUDE: True,
    REAL_AUTHS_INCLUDE: False,
    TRANSLATORS_INCLUDE: False,
    ILLUSTRATORS_INCLUDE: False,
    COVER_AUTHORS_INCLUDE: False,
    APPEND_TO_COMMENTS: [
        (DESCRIPTION, True, _('Description')),
        #IDENTIFIER_STYLE: 0,
    ],
    APPEND_TO_TAG: [
        (GENRES, True, _('Genre'))
    ],
    APPEND_TO_IDENTIFIERS: [
        (ORIG_DK, True, _('Identifier databazeknih'), 'databazeknih'),
        (ISBN, True, _('ISBN'), 'isbn')
    ],
    SERIES_EDITIONS: 0,
    SERIES_LOCALIZATION: 0,
    PUBLICATION_DATE: 0,
    PARSE_ISBN: True, # UNUSED
    PARSE_DATABAZEKNIH: True, # UNUSED
    SWAP_AUTHORS: False,
    ONE_AUTHOR: False,
    CALCULATE_RATING: False,
    CALCULATE_RATING_ACCURACY: 3,
    GENRE_FILTER: False,
    SERIES_FILTER: False,
    PUBLISHER_FILTER: False,
    PUBLISHER_LINE: [
        (PUBLISHER, True, _('Publisher')),
    ],
    TITLE_LINE: [
        (TITLE, True, _('Title')),
    ],
    KEY_SERIES_MAPPINGS: {},
    KEY_GENRE_MAPPINGS: copy.deepcopy(DEFAULT_GENRE_MAPPINGS),
    KEY_PUBLISHER_MAPPINGS: {}

}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/databazeknih')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES

def get_plugin_pref(store_name, option):
    c = plugin_prefs[store_name]
    default_value = plugin_prefs.defaults[store_name][option]
    return c.get(option, default_value)

def get_plugin_prefs(store_name, fill_defaults=False):
    if fill_defaults:
        c = get_prefs(plugin_prefs, store_name)
    else:
        c = plugin_prefs[store_name]
    return c

def get_prefs(prefs_store, store_name):
    store = {}
    if prefs_store and prefs_store[store_name]:
        for key in plugin_prefs.defaults[store_name].keys():
            store[key] = prefs_store[store_name].get(key, plugin_prefs.defaults[store_name][key])
    else:
        store = plugin_prefs.defaults[store_name]
    return store

try:
    load_translations()
except NameError:
    pass # load_translations() added in calibre 1.9


def add_checklist_option(parent, name, help_desc, checkbox_list):
    pass

def add_spin_option(parent, name, help_desc, pref, min_val=1, max_val=100):
    spinbox_layout = QHBoxLayout()
    parent.addLayout(spinbox_layout)
    
    label = QLabel(name)
    label.setToolTip(help_desc)
    spinbox_layout.addWidget(label)
    
    spinbox = QSpinBox()
    spinbox.setMinimum(min_val)
    spinbox.setMaximum(max_val)
    spinbox.setMaximumWidth(60)
    spinbox.setProperty('value', get_plugin_pref(STORE_NAME, pref))
    spinbox.setProperty('option_name', pref)
    spinbox_layout.addWidget(spinbox)
    
    spinbox_layout.addStretch(1)
    return spinbox

def add_combobox_option(parent, name, help_desc, pref, choices):
    combobox_layout = QHBoxLayout()
    parent.addLayout(combobox_layout)
    
    label = QLabel(name)
    label.setToolTip(help_desc)
    combobox_layout.addWidget(label)

    combobox = QComboBox()
    combobox.setProperty('option_name', pref)
    [combobox.addItem(c) for c in choices]
    combobox.setCurrentIndex(get_plugin_pref(STORE_NAME, pref))
    combobox.setMaximumWidth(180)
    combobox_layout.addWidget(combobox)

    return combobox

def add_check_option(parent, name, help_desc=None, pref=None):
    checkbox = QCheckBox(name)
    if help_desc:
        checkbox.setToolTip(help_desc)
    if pref is not None:
        checkbox.setProperty('option_name', pref)
        checkbox.setChecked(get_plugin_pref(STORE_NAME, pref))
    else:
        checkbox.setChecked(True)
        checkbox.setEnabled(False)
    parent.addWidget(checkbox)
    
    return checkbox

def get_widget_value(widget):
    if isinstance(widget, QComboBox):
        return int(widget.currentIndex())
    elif isinstance(widget, QCheckBox):
        return widget.isChecked()
    elif isinstance(widget, QSpinBox):
        return int(as_unicode(widget.value()))
    elif isinstance(widget, QLineEdit):
        return str(widget.text())
    elif isinstance(widget, QTableWidget):
        return widget.get_data()
    
def set_widget_value(widget, value):
    if isinstance(widget, QComboBox):
        widget.setCurrentIndex(value)
    elif isinstance(widget, QCheckBox):
        widget.setChecked(value)
    elif isinstance(widget, QSpinBox):
        widget.setValue(int(value))
    elif isinstance(widget, QLineEdit):
        widget.setText(str(value))
    elif isinstance(widget, QTableWidget):
        widget.populate_table(value)
        
class ConfigWidget(DefaultConfigWidget):

    def set_default_prefs(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to reset plugin to default settings?'),
                show_copy_button=False):
            return
        plugin_prefs[STORE_NAME] = DEFAULT_STORE_VALUES
        set_widget_value(self.search_tab.max_downloads_spin, plugin_prefs[STORE_NAME][KEY_MAX_DOWNLOADS])
        set_widget_value(self.search_tab.max_covers_spin, plugin_prefs[STORE_NAME][MAX_COVERS])
        set_widget_value(self.search_tab.obalkyknih_cover_check, plugin_prefs[STORE_NAME][OBALKYKNIH_COVER])
        set_widget_value(self.tag_tab.table_widget, plugin_prefs[STORE_NAME][KEY_GENRE_MAPPINGS])
        set_widget_value(self.series_tab.table_widget, plugin_prefs[STORE_NAME][KEY_SERIES_MAPPINGS])
        set_widget_value(self.publisher_tab.table_widget, plugin_prefs[STORE_NAME][KEY_PUBLISHER_MAPPINGS])
        set_widget_value(self.tag_tab.genre_filter_check, plugin_prefs[STORE_NAME][GENRE_FILTER])
        set_widget_value(self.series_tab.series_filter_check, plugin_prefs[STORE_NAME][SERIES_FILTER])
        set_widget_value(self.publisher_tab.publishers_filter_check, plugin_prefs[STORE_NAME][PUBLISHER_FILTER])
        set_widget_value(self.authors_tab.one_author_check, plugin_prefs[STORE_NAME][ONE_AUTHOR])
        set_widget_value(self.search_tab.identifier_search_check, plugin_prefs[STORE_NAME][IDENTIFIER_SEARCH])
        set_widget_value(self.search_tab.isbn_search_check, plugin_prefs[STORE_NAME][ISBN_SEARCH])
        set_widget_value(self.search_tab.tales_search_check, plugin_prefs[STORE_NAME][TALES_SEARCH])
        set_widget_value(self.search_tab.google_engine_check, plugin_prefs[STORE_NAME][GOOGLE_SEARCH])
        set_widget_value(self.search_tab.issue_pref_combo, plugin_prefs[STORE_NAME][ISSUE_PREFERENCE])
        set_widget_value(self.authors_tab.authors_field_check, plugin_prefs[STORE_NAME][AUTHORS_INCLUDE])
        set_widget_value(self.authors_tab.real_authors_field_check, plugin_prefs[STORE_NAME][REAL_AUTHS_INCLUDE])
        set_widget_value(self.authors_tab.translators_field_check, plugin_prefs[STORE_NAME][TRANSLATORS_INCLUDE])
        set_widget_value(self.authors_tab.illustrators_field_check, plugin_prefs[STORE_NAME][ILLUSTRATORS_INCLUDE])
        set_widget_value(self.authors_tab.cover_authors_field_check, plugin_prefs[STORE_NAME][COVER_AUTHORS_INCLUDE])
        set_widget_value(self.comments_tab.comments_arrange_table, plugin_prefs[STORE_NAME][APPEND_TO_COMMENTS])
        self.tag_tab.tag_items_table.populate_table(self.tag_tab.DEFAULT_TAG_ITEMS)
        self.identifiers_tab.identifier_items_table.populate_table(self.identifiers_tab.DEFAULT_IDENTIFIERS_ITEMS)
        self.authors_tab.title_line_table.populate_table(self.authors_tab.DEFAULT_TITLE_ITEMS)
        self.publisher_tab.publisher_line_table.populate_table(self.publisher_tab.DEFAULT_PUBLISHER_ITEMS)
        set_widget_value(self.identifiers_tab.recalculate_rating_check, plugin_prefs[STORE_NAME][CALCULATE_RATING])
        set_widget_value(self.identifiers_tab.recalculate_rating_spin, plugin_prefs[STORE_NAME][CALCULATE_RATING_ACCURACY])
        set_widget_value(self.series_tab.series_or_editions, plugin_prefs[STORE_NAME][SERIES_EDITIONS])
        set_widget_value(self.series_tab.series_localization, plugin_prefs[STORE_NAME][SERIES_LOCALIZATION])
        set_widget_value(self.publisher_tab.publication_date, plugin_prefs[STORE_NAME][PUBLICATION_DATE])
        #set_widget_value(self.identifiers_tab.parse_isbn_check, plugin_prefs[STORE_NAME][PARSE_ISBN])
        #set_widget_value(self.identifiers_tab.parse_databazeknih_check, plugin_prefs[STORE_NAME][PARSE_DATABAZEKNIH])
        set_widget_value(self.authors_tab.swap_authors_check, plugin_prefs[STORE_NAME][SWAP_AUTHORS])

    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)

        top_box = QHBoxLayout(self)
        if plugin.config_message:
            config_message = QLabel(plugin.config_message)
            config_message.setWordWrap(True)
            config_message.setOpenExternalLinks(True)
            top_box.addWidget(config_message)

        reset_plugin_btn = QToolButton()
        reset_plugin_btn.setToolTip(_('Reset plugin to defaults (Alt+Shift+R)'))
        reset_plugin_btn.setIcon(QIcon(I('restart.png')))
        reset_plugin_btn.clicked.connect(self.set_default_prefs)
        reset_plugin_btn.setShortcut(_('Alt+Shift+R'))
        reset_plugin_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
        reset_plugin_btn.setText(_('Restart plugin to defaults'))
        top_box.addWidget(reset_plugin_btn)

        try:
            self.overl.insertLayout(0, top_box)
        except AttributeError:
            self.l.addLayout(top_box, 0, 0, 1, 1)
            
        self.gb.setMaximumHeight(70)

        tab = QTabWidget(self)
        self.l.addWidget(tab, self.l.rowCount(), 0, 1, 1)

        self.search_tab = SearchTab()
        self.authors_tab = AuthorsTab()
        self.tag_tab = TagTab()
        self.series_tab = SeriesTab()
        self.publisher_tab = PublisherTab()
        self.comments_tab = CommentsTab()
        self.identifiers_tab = IdentifiersTab()

        tab.addTab(self.search_tab, _('Search'))
        tab.addTab(self.authors_tab, _('Title and Authors'))
        tab.addTab(self.tag_tab, _('Tags'))
        tab.addTab(self.series_tab, _('Series'))
        tab.addTab(self.publisher_tab, _('Publishers and Date'))
        tab.addTab(self.comments_tab, _('Comments'))
        tab.addTab(self.identifiers_tab, _('Identifiers and Rating'))

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[KEY_MAX_DOWNLOADS] = get_widget_value(self.search_tab.max_downloads_spin)
        new_prefs[MAX_COVERS] = get_widget_value(self.search_tab.max_covers_spin)
        new_prefs[OBALKYKNIH_COVER] = get_widget_value(self.search_tab.obalkyknih_cover_check)
        new_prefs[KEY_GENRE_MAPPINGS] = get_widget_value(self.tag_tab.table_widget)
        new_prefs[KEY_SERIES_MAPPINGS] = get_widget_value(self.series_tab.table_widget)
        new_prefs[KEY_PUBLISHER_MAPPINGS] = get_widget_value(self.publisher_tab.table_widget)
        new_prefs[GENRE_FILTER] = get_widget_value(self.tag_tab.genre_filter_check)
        new_prefs[SERIES_FILTER] = get_widget_value(self.series_tab.series_filter_check)
        new_prefs[PUBLISHER_FILTER] = get_widget_value(self.publisher_tab.publishers_filter_check)
        new_prefs[ONE_AUTHOR] = get_widget_value(self.authors_tab.one_author_check)
        #new_prefs[CUSTOM_ID] = get_widget_value(self.identifiers_tab.custom_id_edit)
        new_prefs[IDENTIFIER_SEARCH] = get_widget_value(self.search_tab.identifier_search_check)
        new_prefs[ISBN_SEARCH] = get_widget_value(self.search_tab.isbn_search_check)
        new_prefs[TALES_SEARCH] = get_widget_value(self.search_tab.tales_search_check)
        new_prefs[GOOGLE_SEARCH] = get_widget_value(self.search_tab.google_engine_check)
        new_prefs[ISSUE_PREFERENCE] = get_widget_value(self.search_tab.issue_pref_combo)
        new_prefs[AUTHORS_INCLUDE] = get_widget_value(self.authors_tab.authors_field_check)
        new_prefs[REAL_AUTHS_INCLUDE] = get_widget_value(self.authors_tab.real_authors_field_check)
        new_prefs[TRANSLATORS_INCLUDE] = get_widget_value(self.authors_tab.translators_field_check)
        new_prefs[ILLUSTRATORS_INCLUDE] = get_widget_value(self.authors_tab.illustrators_field_check)
        new_prefs[COVER_AUTHORS_INCLUDE] = get_widget_value(self.authors_tab.cover_authors_field_check)

        #comments[IDENTIFIER_STYLE] = self.identifier_style.currentIndex()
        new_prefs[APPEND_TO_COMMENTS] = get_widget_value(self.comments_tab.comments_arrange_table)
        new_prefs[APPEND_TO_TAG] = get_widget_value(self.tag_tab.tag_items_table)
        new_prefs[APPEND_TO_IDENTIFIERS] = get_widget_value(self.identifiers_tab.identifier_items_table)
        new_prefs[TITLE_LINE] = get_widget_value(self.authors_tab.title_line_table)
        new_prefs[PUBLISHER_LINE] = get_widget_value(self.publisher_tab.publisher_line_table)

        new_prefs[SERIES_EDITIONS] = get_widget_value(self.series_tab.series_or_editions)
        new_prefs[SERIES_LOCALIZATION] = get_widget_value(self.series_tab.series_localization)
        new_prefs[PUBLICATION_DATE] = get_widget_value(self.publisher_tab.publication_date)
        #new_prefs[PARSE_ISBN] = get_widget_value(self.identifiers_tab.parse_isbn_check)
        #new_prefs[PARSE_DATABAZEKNIH] = get_widget_value(self.identifiers_tab.parse_databazeknih_check)
        new_prefs[SWAP_AUTHORS] = get_widget_value(self.authors_tab.swap_authors_check)
        new_prefs[CALCULATE_RATING] = get_widget_value(self.identifiers_tab.recalculate_rating_check)
        new_prefs[CALCULATE_RATING_ACCURACY] = get_widget_value(self.identifiers_tab.recalculate_rating_spin)
        plugin_prefs[STORE_NAME] = new_prefs

class SearchTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)

        #Search option groupbox
        #other_group_box = QGroupBox(_('Search'), self)
        other_group_box_layout = QVBoxLayout()
        self.setLayout(other_group_box_layout)      
        
        self.max_downloads_spin = add_spin_option(other_group_box_layout, _('Maximum title/author search matches to evaluate (1 = fastest):'), 
                                                  _('Number of maximum search results which you can get from DK.'),
                                                  KEY_MAX_DOWNLOADS)

        search_group_box = QGroupBox(_('Searching priority'), self)
        search_group_box_layout = QHBoxLayout()
        search_group_box.setLayout(search_group_box_layout)


        title_identifiers_search_check = add_check_option(search_group_box_layout, 
                                                        _('Title ids'),
                                                        _("You can add identifiers into 'Title' field (in format identifier:value e.g. databazeknih:123 or publisher:new_publisher)\nbefore you start metadata download and they will be used prior built-in one."\
                                                          "Identifiers can be combined in title\n\n"\
                                                          "Identifiers available:\n"\
                                                          "'databazeknih', 'dbknih' and 'dbk' for books\n"\
                                                          "'databazeknih_povidka', 'dbknih_povidka', 'dbk_povidka', 'dbk_p' and 'dbkp' for tales/poems\n"\
                                                          "'isbn' and 'ean' for searching by isbn number (databazeknih can search by ISBN without this prefixes too)\n"\
                                                          "'publisher' for book issue publisher specification (use underscore '_' for spacing multiple words)\n"\
                                                          "'pubdate' and 'pubyear' published edition year specification\n(e.g. pubdate:2024 will force plugin to prior search for issues from year 2024)"))
        
        search_group_box_layout.addWidget(QLabel('>', self))

        self.identifier_search_check = add_check_option(search_group_box_layout, 
                                                        _('Id databazeknih'),
                                                        _('If identifier found in book metadata, plugin will use it for faster book search.\n'\
                                                          'databazeknih, dbknih or dbk for books\n'\
                                                          'databazeknih_povidka for poems/tales'),
                                                        IDENTIFIER_SEARCH)
        search_group_box_layout.addWidget(QLabel('>', self))
        self.isbn_search_check = add_check_option(search_group_box_layout, 
                                                        _('ISBN'),
                                                        _('If isbn identifier found in book metadata, plugin will use it for faster book search.'),
                                                        ISBN_SEARCH)
        search_group_box_layout.addWidget(QLabel('>', self))

        self.tales_search_check = add_check_option(search_group_box_layout, 
                                                        _('Tales'),
                                                        _("Check this if you want to search in tales before searching in books.\nTales on databazeknih contain less metadata information than regular books.\n"\
                                                        "Search for tales/poems attempts:\ntitle + all authors > only whole title > only one word from title (longest first)\n\n"\
                                                        "This plugin uses default Calibre metadata fields differently than intended for books\n"\
                                                        "and DON'T support some features like tags mapping and comment builder.\n"\
                                                        "Here is how this plugin use fields for tales:\n"\
                                                        "--------------------------------------------------\n"\
                                                        "databazeknih info > Calibre data field\n"\
                                                        "---------------------------------------------------\n"\
                                                        "title > Title\n"\
                                                        "author(s) > Authors\n"\
                                                        "!!books containing this tale > Tags\n"\
                                                        "percent rating > 5 star Rating\n"\
                                                        "description > Comments\n"\
                                                        "!original publication year > Published date\n"\
                                                        "!!original title > Publisher"),
                                                        TALES_SEARCH)
        search_group_box_layout.addWidget(QLabel('>', self))

        authors_search_check = add_check_option(search_group_box_layout,
                                                        _('Books'),
                                                        _('Search for books\n\nusing databazeknih search engine:\n whole title + last name of first author > only title + all authors > whole title > only one word from title (longest first) >\n'\
                                                          '> all authors > only one author > only one part of authors name\n\nusing Google search engine: title + all authors')) 
        search_group_box_layout.addStretch(1)
        other_group_box_layout.addWidget(search_group_box)

        # Search engines
        search_engine_group_box = QGroupBox(_('Search engine'), self)
        search_engine_group_box_layout = QHBoxLayout()
        search_engine_group_box.setLayout(search_engine_group_box_layout)

        self.google_engine_check = add_check_option(search_engine_group_box_layout,
                                                        _('Google search'),
                                                        _('Tries to search for book or tale/poem via google search engine prior to databazeknih search.'),
                                                        GOOGLE_SEARCH)
        search_engine_group_box_layout.addWidget(QLabel('>', self))
        databazeknih_engine_check = add_check_option(search_engine_group_box_layout,
                                                        _('Databazeknih'),
                                                        _('Tries to search for book or tale/poem via Databazeknih search engine.')) 
        
        search_engine_group_box_layout.addStretch(1)
        other_group_box_layout.addWidget(search_engine_group_box)

        # Issue choice
        book_issues_group_box = QGroupBox(_('Book issues'), self)
        book_issues_group_box_layout = QHBoxLayout()
        book_issues_group_box.setLayout(book_issues_group_box_layout)
        self.issue_pref_combo = add_combobox_option(book_issues_group_box_layout, _('Choose preferred book issue'), _('Set which issue of the book will be preferred. You can choose between czech/slovak and oldest/newest.\nDefault will be chosen if specified issue not found.'), 
                            ISSUE_PREFERENCE, choices=[_('Default'), _('Czech newest'), _('Czech oldest'), _('Slovak newest'), _('Slovak oldest')])
        
        other_group_box_layout.addWidget(book_issues_group_box)

        # Covers download
        covers_group_box = QGroupBox(_('Covers'), self)
        covers_group_box_layout = QVBoxLayout()
        covers_group_box.setLayout(covers_group_box_layout)
        self.max_covers_spin = add_spin_option(covers_group_box_layout, _('Maximum covers from other issues to evaluate (1 = fastest):'), _('Set max covers to download. Faster evaluating if set to 1 with Default preferred issue set\n\n'\
                                                                                                                                            'When set to 0 - no covers will be downloaded.\n'\
                                                                                                                                            'When set to 1 and preferred book issue option set to default - only default found issue cover will be downloaded.\n'\
                                                                                                                                            'When set to >1 all covers from preferred issue language are downloaded\n'\
                                                                                                                                            '- When preferred czech only all czech cover gets downloaded and vice-versa (default option recognizes language of default book)\n'\
                                                                                                                                            '- oldest/newest order does matter if max cover option < available covers'),
                            MAX_COVERS, min_val=0)
        
        self.obalkyknih_cover_check = add_check_option(covers_group_box_layout,
                                                        _('Search for additional cover on Obalky knih'),
                                                        _('Tries to search for additional cover using ISBN number on obalkyknih.cz site. Does not count into max cover limit option.'),
                                                        OBALKYKNIH_COVER)
        
        
        other_group_box_layout.addWidget(covers_group_box)

        other_group_box_layout.addStretch(1)

class AuthorsTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Title field
        title_builder_group_box = QGroupBox(_("Set what will be parsed into Calibre 'Title' column"))
        title_builder_group_box.setMaximumHeight(100)
        layout.addWidget(title_builder_group_box)

        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)
        tablebox_layout = QVBoxLayout()
        title_builder_group_box.setLayout(tablebox_layout)
        combobox_layout = QHBoxLayout()
        
        self.DEFAULT_TITLE_ITEMS = [
            (TITLE, True, _('Title')),
        ]
        title_items = []
        for i, tupl in enumerate(c.get(TITLE_LINE, self.DEFAULT_TITLE_ITEMS)):
            title_items.append((tupl[0], tupl[1], tupl[2]))

        self.title_line_table = LineBuilderTableWidget(self, title_items, header_labels=[_('Title: ')])
        self.title_line_table.populate_table(title_items)
        self.title_line_string = QLabel(self.title_line_table.get_string())
        title_line_layout = QHBoxLayout()
        title = QLabel(_('Title: '))
        font = title.font()
        font.setBold(True)
        title.setFont(font)
        title_line_layout.addWidget(title)
        title_line_layout.addWidget(self.title_line_string, 1)
        tablebox_layout.addLayout(title_line_layout)
        tablebox_layout.addLayout(combobox_layout)
        combobox_layout.addWidget(self.title_line_table)

        button_layout = QHBoxLayout()
        combobox_layout.addLayout(button_layout)
        add_btn = QToolButton()
        add_btn.setToolTip(_('Add item (Alt+Shift+W)'))
        add_btn.setIcon(QIcon(I('plus.png')))
        add_btn.clicked.connect(self.add_title_item)
        add_btn.setShortcut(_('Alt+Shift+W'))
        button_layout.addWidget(add_btn)

        delete_btn = QToolButton()
        delete_btn.setToolTip(_('Delete item (Alt+Shift+S)'))
        delete_btn.setIcon(QIcon(I('minus.png')))
        delete_btn.clicked.connect(self.delete_title_item)
        delete_btn.setShortcut(_('Alt+Shift+S'))
        button_layout.addWidget(delete_btn)

        move_left_btn = QToolButton()
        move_left_btn.setToolTip(_('Move left (Alt+Shift+A)'))
        move_left_btn.setIcon(QIcon(I('back.png')))
        move_left_btn.clicked.connect(self.move_cols_left)
        move_left_btn.setShortcut(_('Alt+Shift+A'))
        button_layout.addWidget(move_left_btn)

        move_right_btn = QToolButton()
        move_right_btn.setToolTip(_('Move right (Alt+Shift+D)'))
        move_right_btn.setIcon(QIcon(I('forward.png')))
        move_right_btn.clicked.connect(self.move_cols_right)
        move_right_btn.setShortcut(_('Alt+Shift+D'))
        button_layout.addWidget(move_right_btn)

        self.item_options = [
        (CUSTOM_TEXT, True, _('Custom text')),
        (TITLE, True, _('Title')),
        (ORIGINAL_TITLE, True, _('Original title')),
        (PUBLISHER, True, _('Publisher')),
        (AUTHORS, True, _('Authors')),
        (REAL_AUTHORS, True, _('Real authors')),
        (TRANSLATION, True, _('Translators')),
        (ILLUSTRATION, True, _('Illustrators')),
        (COVER_AUTHORS, True, _('Book cover authors')),
        (ORIGINAL_YEAR, True, _('Original pubdate')),
        (PUB_YEAR, True, _('Publication date')),
        (PAGES, True, _('Pages count')),
        (COVER_TYPE, True, _('Cover type')),
        (PRINT_RUN, True, _('Print run')),
        (SERIES, True, _('Series')),
        (SERIES_INDEX, True, _('Series index')),
        (EDITION, True, _('Book Edition')),
        (EDITION_INDEX, True, _('Book Edition index')),
        (GENRES, True, _('Genre')),
        (TAGS, True, _('Tags')),
        (RATING, True, _('Rating (%)')),
        (RATING10, True, _('Rating (0-10)')),
        (RATING5, True, _('Rating (0-5)')),
        (RATING_COUNT, True, _('Rating count')),
        (ISBN, True, _('ISBN')),
        (ORIG_DK, True, _('Identifier databazeknih')),
        ]

        authors_field_group_box = QGroupBox(_('Select what will be added into "Authors" metadata field in Calibre'), self)
        layout.addWidget(authors_field_group_box)
        
        group_box_layout = QHBoxLayout()
        authors_field_group_box.setLayout(group_box_layout)


        self.authors_field_check = add_check_option(group_box_layout,
                                                    _('Authors'),
                                                    _('Names of authors will be added into calibre authors metadata field.'),
                                                    AUTHORS_INCLUDE)
        
        self.real_authors_field_check = add_check_option(group_box_layout,
                                                    _('Real Name Authors'),
                                                    _('Real names of authors (no pseudonyms) will be added into calibre authors metadata field.'),
                                                    REAL_AUTHS_INCLUDE)

        self.translators_field_check = add_check_option(group_box_layout,
                                                    _('Translators'),
                                                    _('Names of translators will be added into calibre authors metadata field.'),
                                                    TRANSLATORS_INCLUDE)

        self.cover_authors_field_check = add_check_option(group_box_layout,
                                                    _('Book cover authors'),
                                                    _('Names of book cover authors will be added into calibre authors metadata field.'),
                                                    COVER_AUTHORS_INCLUDE)

        self.illustrators_field_check = add_check_option(group_box_layout,
                                                    _('Illustrators'),
                                                    _('Names of illustrators will be added into calibre authors metadata field.'),
                                                    ILLUSTRATORS_INCLUDE)

        self.swap_authors_check = add_check_option(layout, _('Swap authors names with last names'), 
                                                   _('Enable this option if you want to get names in Authors metadata field in format "Surname Name"\n **DONT COMBINE with LN, FN Calibre option'),
                                                   SWAP_AUTHORS)
        
        self.one_author_check = add_check_option(layout, _('Get only one author'), 
                                                  _('If the information about the book contains more authors\n - plugin will add only first one.'), 
                                                  ONE_AUTHOR)
        layout.addStretch(1)

    def move_cols_left(self):
        self.title_line_table.setFocus()
        cols = self.title_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        first_sel_col = cols[0].column()
        if first_sel_col <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in selcols:
            self.title_line_table.swap_row_widgets(selcol - 1, selcol + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_col - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.title_line_table.scrollToItem(self.title_line_table.item(scroll_to_row, 0))
        self.title_line_string.setText(self.title_line_table.get_string())
        #self.renumber_series()

    def move_cols_right(self):
        self.title_line_table.setFocus()
        cols = self.title_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        last_sel_col = cols[-1].column()
        if last_sel_col == self.title_line_table.columnCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in reversed(selcols):
            self.title_line_table.swap_row_widgets(selcol + 2, selcol)

        scroll_to_row = last_sel_col + 1
        if scroll_to_row < self.title_line_table.columnCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.title_line_table.scrollToItem(self.title_line_table.item(scroll_to_row, 0))
        self.title_line_string.setText(self.title_line_table.get_string())
        # self.renumber_series()

    def add_title_item(self):
        self.title_line_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to title"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.title_line_table.get_data()
            new_item = [i for i in self.item_options if item in i][0]
            if new_item[0] == CUSTOM_TEXT:
                text, ok_text = QInputDialog().getText(self, _("Add custom text"),
                                            _("Write text:"))
                if not ok_text or not text:
                    return
                else:
                    new_item = (CUSTOM_TEXT, True, text)
            new_items.append(new_item)
            self.title_line_table.populate_table(new_items)
            self.title_line_string.setText(self.title_line_table.get_string())
            self.title_line_table.selectColumn(self.title_line_table.columnCount() - 1)
        if not item:
            return

    def delete_title_item(self):
        from calibre.gui2 import question_dialog
        self.title_line_table.setFocus()
        if not self.title_line_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for col in reversed(sorted(self.title_line_table.selectionModel().selectedColumns())):
            self.title_line_table.removeColumn(col.column())
        self.title_line_string.setText(self.title_line_table.get_string())

    def reset_title_items(self):
        from calibre.gui2 import question_dialog
        self.title_line_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set title builder to defaults?'),
                show_copy_button=False):
            return
        self.title_line_table.setColumnCount(0)
        self.title_line_table.populate_table(DEFAULT_STORE_VALUES[TITLE_LINE])
        self.title_line_string.setText(self.title_line_table.get_string())

class SeriesTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Items in Series metadata field
        self.series_or_editions = add_combobox_option(layout, _("Set what will be parsed into Calibre 'Series' column"), _('You can select series or editions to be parsed here.'),
                            SERIES_EDITIONS, choices=[_('Series only'), _('Editions only'), _('Series before Editions'), _('Editions before Series')])

        #Parse all or only Czech/Slovak part of series name
        self.series_localization = add_combobox_option(layout, _('Set which part of databazeknih series name information will be saved'), _('Databazeknih saves many series names in both Czech and Slovak languages separated with " / ".\n'\
                                                                 'You can select whole name of the series or Czech/Slovak part only to be saved and then remapped into calibre.'),
                            SERIES_LOCALIZATION, choices=[_('Whole name'), _('Czech part'), _('Slovak part')])


        # Calibre Mapping Grid
        mapping_group_box = QGroupBox(_('Databazeknih series and editions to calibre series and editions mappings'))
        mapping_group_box.setMaximumHeight(800)
        layout.addWidget(mapping_group_box)

        filter_layout = QHBoxLayout()
        self.series_filter_check = add_check_option(filter_layout, _('Filter series/editions via mappings'), _("Check this and only mapped series and editions below will get parsed to Calibre."), SERIES_FILTER)

        from calibre.gui2 import get_current_db
        all_series = list()
        for s in get_current_db().all_series():
            if s[1] not in all_series:
                all_series.append(s[1])
        self.table_widget = MappingsTableWidget(self, all_series, headers=(_('series/editions'), _('Series/editions')))
        tags_layout = RemapingTableLayout(self, self.table_widget, KEY_SERIES_MAPPINGS, 'series/editions', self.series_filter_check)
        mapping_group_box.setLayout(tags_layout)

class PublisherTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Items in Published date metadata field
        self.publication_date = add_combobox_option(layout, _("Set what will be parsed into Calibre 'Published date' column"), _('You can select date of first publication or this publication to be parsed here.'), 
                            PUBLICATION_DATE, choices=[_('Actual publication'), _('First publication')])
        

        #Publisher field
        publisher_builder_group_box = QGroupBox(_("Set what will be parsed into Calibre 'Publisher' column"))
        publisher_builder_group_box.setMaximumHeight(100)
        layout.addWidget(publisher_builder_group_box)

        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)
        tablebox_layout = QVBoxLayout()
        publisher_builder_group_box.setLayout(tablebox_layout)
        combobox_layout = QHBoxLayout()
        
        self.DEFAULT_PUBLISHER_ITEMS = [
            (PUBLISHER, True, _('Publisher')),
        ]
        publisher_items = []
        for i, tupl in enumerate(c.get(PUBLISHER_LINE, self.DEFAULT_PUBLISHER_ITEMS)):
            publisher_items.append((tupl[0], tupl[1], tupl[2]))

        self.publisher_line_table = LineBuilderTableWidget(self, publisher_items, header_labels=[_('Publisher: ')])
        self.publisher_line_table.populate_table(publisher_items)
        self.publisher_line_string = QLabel(self.publisher_line_table.get_string())
        title_line_layout = QHBoxLayout()
        title = QLabel(_('Publisher: '))
        font = title.font()
        font.setBold(True)
        title.setFont(font)
        title_line_layout.addWidget(title)
        title_line_layout.addWidget(self.publisher_line_string, 1)
        tablebox_layout.addLayout(title_line_layout)
        tablebox_layout.addLayout(combobox_layout)
        combobox_layout.addWidget(self.publisher_line_table)

        button_layout = QHBoxLayout()
        combobox_layout.addLayout(button_layout)
        add_btn = QToolButton()
        add_btn.setToolTip(_('Add item (Alt+Shift+W)'))
        add_btn.setIcon(QIcon(I('plus.png')))
        add_btn.clicked.connect(self.add_publisher_item)
        add_btn.setShortcut(_('Alt+Shift+W'))
        button_layout.addWidget(add_btn)

        delete_btn = QToolButton()
        delete_btn.setToolTip(_('Delete item (Alt+Shift+S)'))
        delete_btn.setIcon(QIcon(I('minus.png')))
        delete_btn.clicked.connect(self.delete_publisher_item)
        delete_btn.setShortcut(_('Alt+Shift+S'))
        button_layout.addWidget(delete_btn)

        move_left_btn = QToolButton()
        move_left_btn.setToolTip(_('Move left (Alt+Shift+A)'))
        move_left_btn.setIcon(QIcon(I('back.png')))
        move_left_btn.clicked.connect(self.move_cols_left)
        move_left_btn.setShortcut(_('Alt+Shift+A'))
        button_layout.addWidget(move_left_btn)

        move_right_btn = QToolButton()
        move_right_btn.setToolTip(_('Move right (Alt+Shift+D)'))
        move_right_btn.setIcon(QIcon(I('forward.png')))
        move_right_btn.clicked.connect(self.move_cols_right)
        move_right_btn.setShortcut(_('Alt+Shift+D'))
        button_layout.addWidget(move_right_btn)

        self.item_options = [
        (CUSTOM_TEXT, True, _('Custom text')),
        (PUBLISHER, True, _('Publisher')),
        (ORIGINAL_TITLE, True, _('Original title')),
        (TITLE, True, _('Title')),
        (AUTHORS, True, _('Authors')),
        (REAL_AUTHORS, True, _('Real authors')),
        (TRANSLATION, True, _('Translators')),
        (ILLUSTRATION, True, _('Illustrators')),
        (COVER_AUTHORS, True, _('Book cover authors')),
        (ORIGINAL_YEAR, True, _('Original pubdate')),
        (PUB_YEAR, True, _('Publication date')),
        (PAGES, True, _('Pages count')),
        (COVER_TYPE, True, _('Cover type')),
        (PRINT_RUN, True, _('Print run')),
        (SERIES, True, _('Series')),
        (SERIES_INDEX, True, _('Series index')),
        (EDITION, True, _('Book Edition')),
        (EDITION_INDEX, True, _('Book Edition index')),
        (GENRES, True, _('Genre')),
        (TAGS, True, _('Tags')),
        (RATING, True, _('Rating (%)')),
        (RATING10, True, _('Rating (0-10)')),
        (RATING5, True, _('Rating (0-5)')),
        (RATING_COUNT, True, _('Rating count')),
        (ISBN, True, _('ISBN')),
        (ORIG_DK, True, _('Identifier databazeknih')),
        ]

        # Calibre Mapping Grid
        mapping_group_box = QGroupBox(_('Databazeknih publishers to calibre publisher mappings'))
        mapping_group_box.setMaximumHeight(800)
        layout.addWidget(mapping_group_box)

        filter_layout = QHBoxLayout()
        self.publishers_filter_check = add_check_option(filter_layout, _('Filter publishers via mappings'), _("Check this and only mapped publishers below will get parsed to Calibre."), PUBLISHER_FILTER)
        from calibre.gui2 import get_current_db
        all_publishers = list()
        for s in get_current_db().all_publishers():
            if s[1] not in all_publishers:
                all_publishers.append(s[1])
        self.table_widget = MappingsTableWidget(self, all_publishers, headers=(_('publisher'), _('Publisher')))
        tags_layout = RemapingTableLayout(self, self.table_widget, KEY_PUBLISHER_MAPPINGS, 'publishers', self.publishers_filter_check)
        mapping_group_box.setLayout(tags_layout)

    def move_cols_left(self):
        self.publisher_line_table.setFocus()
        cols = self.publisher_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        first_sel_col = cols[0].column()
        if first_sel_col <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in selcols:
            self.publisher_line_table.swap_row_widgets(selcol - 1, selcol + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_col - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.publisher_line_table.scrollToItem(self.publisher_line_table.item(scroll_to_row, 0))
        self.publisher_line_string.setText(self.publisher_line_table.get_string())
        #self.renumber_series()

    def move_cols_right(self):
        self.publisher_line_table.setFocus()
        cols = self.publisher_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        last_sel_col = cols[-1].column()
        if last_sel_col == self.publisher_line_table.columnCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in reversed(selcols):
            self.publisher_line_table.swap_row_widgets(selcol + 2, selcol)

        scroll_to_row = last_sel_col + 1
        if scroll_to_row < self.publisher_line_table.columnCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.publisher_line_table.scrollToItem(self.publisher_line_table.item(scroll_to_row, 0))
        self.publisher_line_string.setText(self.publisher_line_table.get_string())
        # self.renumber_series()

    def add_publisher_item(self):
        self.publisher_line_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to publisher"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.publisher_line_table.get_data()
            new_item = [i for i in self.item_options if item in i][0]
            if new_item[0] == CUSTOM_TEXT:
                text, ok_text = QInputDialog().getText(self, _("Add custom text"),
                                            _("Write text:"))
                if not ok_text or not text:
                    return
                else:
                    new_item = (CUSTOM_TEXT, True, text)
            new_items.append(new_item)
            self.publisher_line_table.populate_table(new_items)
            self.publisher_line_string.setText(self.publisher_line_table.get_string())
            self.publisher_line_table.selectColumn(self.publisher_line_table.columnCount() - 1)
        if not item:
            return

    def delete_publisher_item(self):
        from calibre.gui2 import question_dialog
        self.publisher_line_table.setFocus()
        if not self.publisher_line_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for col in reversed(sorted(self.publisher_line_table.selectionModel().selectedColumns())):
            self.publisher_line_table.removeColumn(col.column())
        self.publisher_line_string.setText(self.publisher_line_table.get_string())

    def reset_publisher_items(self):
        from calibre.gui2 import question_dialog
        self.publisher_line_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set publisher builder to defaults?'),
                show_copy_button=False):
            return
        self.publisher_line_table.setColumnCount(0)
        self.publisher_line_table.populate_table(DEFAULT_STORE_VALUES[PUBLISHER_LINE])
        self.publisher_line_string.setText(self.publisher_line_table.get_string())

class TagTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)
        #Tags
        #tags_group_box = QGroupBox(_('Tags'), self)
        #self.l.addWidget(tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        tags_group_boxlayout = QVBoxLayout()
        self.setLayout(tags_group_boxlayout)

        #Append tags group box
        append_tags_group_box = QGroupBox(_('Select what will be added into "Tags" metadata field in Calibre'), self)
        #self.l.addWidget(append_tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        tags_group_boxlayout.addWidget(append_tags_group_box)

        table_layout = QHBoxLayout()
        append_tags_group_box.setLayout(table_layout)

        # here because of translations
        self.DEFAULT_TAG_ITEMS = [
            (GENRES, True, _('Genre')),
            (TAGS, False, _('Tags')),
            (PAGES, False, _('Pages count')),
            (COVER_TYPE, False, _('Cover type')),
            (PRINT_RUN, False, _('Print run')),
            (ORIGINAL_TITLE, False, _('Original title')),
            (TITLE, False, _('Title')),
            (ORIGINAL_YEAR, False, _('Original pubdate')),
            (PUB_YEAR, False, _('Publication date')),
            (PUBLISHER, False, _('Publisher')),
            (AUTHORS, False, _('Authors')),
            (REAL_AUTHORS, False, _('Real authors')),
            (TRANSLATION, False, _('Translators')),
            (ILLUSTRATION, False, _('Illustrators')),
            (COVER_AUTHORS, False, _('Book cover authors')),
            (SERIES, False, _('Series')),
            (EDITION, False, _('Book Edition')),
            (RATING, False, _('Rating (%)')),
            (RATING10, False, _('Rating (0-10)')),
            (RATING5, False, _('Rating (0-5)')),
            (RATING_COUNT, False, _('Rating count')),
            (ISBN, False, _('ISBN')),
            (ORIG_DK, False, _('Identifier databazeknih')),
            (AWARDS, False, _('Awards')),
            (TALES_IN_BOOK, False, _('Tales in book'))
        ]
        tag_items = []
        if len(c.get(APPEND_TO_TAG, [])) == len(self.DEFAULT_TAG_ITEMS):
            for i, tupl in enumerate(c.get(APPEND_TO_TAG, self.DEFAULT_TAG_ITEMS)):
                tag_items.append((tupl[0], tupl[1], self.DEFAULT_TAG_ITEMS[i][2]))
        else:
            tag_items = self.DEFAULT_TAG_ITEMS

        self.tag_items_table = TagBuilderTableWidget(self, tag_items)
        table_layout.addWidget(self.tag_items_table)
        self.tag_items_table.populate_table(tag_items)

        # Calibre Genre Mapping Grid
        genre_group_box = QGroupBox(_('Databazeknih genre and tag to calibre tag mappings'))
        genre_group_box.setMaximumHeight(800)
        tags_group_boxlayout.addWidget(genre_group_box)
        #self.l.addWidget(genre_group_box, self.l.rowCount(), 0, 1, 2)
        genre_group_box_layout = QVBoxLayout()
        genre_group_box.setLayout(genre_group_box_layout)

        filter_layout = QHBoxLayout()
        self.genre_filter_check = add_check_option(filter_layout, _('Filter genre/tags via mappings'), _("Check this and only mapped genre and tags below will get parsed to Calibre."), GENRE_FILTER)

        from calibre.gui2 import get_current_db
        all_tags = get_current_db().all_tags()
        self.table_widget = MappingsTableWidget(self, c[KEY_GENRE_MAPPINGS])
        tags_layout = RemapingTableLayout(self, self.table_widget, KEY_GENRE_MAPPINGS, 'genre/tag', self.genre_filter_check)
        genre_group_box_layout.addLayout(tags_layout)

class TagBuilderTableWidget(QTableWidget):

    def __init__(self, parent, tag_items):
        QTableWidget.__init__(self, parent)
        self.tag_items = tag_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        #self.setMinimumSize(50, 0)

    def populate_table(self, tag_items):
        self.tag_items = tag_items
        self.clear()
        self.setRowCount(len(tag_items))
        header_labels = [_('Tag item')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(tag_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyCheckableTableWidgetItem(item[2], checked=item[1])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        tag_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(row, 0).text())))
            active = self.item(row, 0).get_boolean_value()
            tag_items.append((item_id, active, item_name))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", tag_items)
        return tag_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class RemapingTableLayout(QVBoxLayout):
    def __init__(self, parent, table_widget, prefs, name, filter_check_widget=None):
        super(RemapingTableLayout, self).__init__(parent)
        self.parent = parent
        self.name = name
        self.prefs = prefs
        self.table_widget = table_widget
        
        if filter_check_widget:
            filter_check_layout = QVBoxLayout()
            filter_check_layout.addWidget(filter_check_widget)
            self.addLayout(filter_check_layout)

        hbox_layout = QHBoxLayout()
        self.addLayout(hbox_layout)

        hbox_layout.addWidget(self.table_widget)

        button_layout = QVBoxLayout()
        hbox_layout.addLayout(button_layout)

        add_mapping_button = QToolButton()
        add_mapping_button.setToolTip(_('Add mapping (Alt+A)'))
        add_mapping_button.setIcon(QIcon(I('plus.png')))
        add_mapping_button.clicked.connect(self.add_mapping)
        add_mapping_button.setShortcut(_('Alt+A'))
        button_layout.addWidget(add_mapping_button)

        remove_mapping_button = QToolButton()
        remove_mapping_button.setToolTip(_('Delete mapping (Alt+D)'))
        remove_mapping_button.setIcon(QIcon(I('minus.png')))
        remove_mapping_button.clicked.connect(self.delete_mapping)
        remove_mapping_button.setShortcut(_('Alt+D'))
        button_layout.addWidget(remove_mapping_button)

        rename_genre_button = QToolButton()
        rename_genre_button.setToolTip(_('Rename item (Alt+S)'))
        rename_genre_button.setIcon(QIcon(I('edit-undo.png')))
        rename_genre_button.clicked.connect(self.rename_genre)
        rename_genre_button.setShortcut(_('Alt+S'))
        button_layout.addWidget(rename_genre_button)

        reset_defaults_button = QToolButton()
        reset_defaults_button.setToolTip(_('Reset to plugin default mappings (Alt+R)'))
        reset_defaults_button.setIcon(QIcon(I('restart.png')))
        reset_defaults_button.clicked.connect(self.reset_to_defaults)
        reset_defaults_button.setShortcut(_('Alt+R'))
        button_layout.addWidget(reset_defaults_button)

        # Populate the table
        self.table_widget.populate_table(get_plugin_pref(STORE_NAME, prefs))

    def add_mapping(self):
        from calibre.gui2 import error_dialog
        new_genre_name, ok = QInputDialog.getText(self.parent, _('Add new mapping'),
                    _('Enter a Databazeknih %s name to create a mapping for:')%self.name, text='')
        if not ok:
            # Operation cancelled
            return
        new_genre_name = as_unicode(new_genre_name).strip().replace(',', ';')
        if not new_genre_name:
            return
        # Verify it does not clash with any other mappings in the list
        data = self.table_widget.get_data()
        for genre_name in data.keys():
            if genre_name.lower() == new_genre_name.lower():
                return error_dialog(self.parent, _('Add Failed'), _('A %s with the same name already exists')%self.name, show=True)
        data[new_genre_name] = []
        self.table_widget.populate_table(data)
        self.table_widget.select_genre(new_genre_name)

    def delete_mapping(self):
        from calibre.gui2 import question_dialog
        if not self.table_widget.selectionModel().hasSelection():
            return
        if not question_dialog(self.parent, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected %s mappings?')%self.name,
                show_copy_button=False):
            return
        for row in reversed(sorted(self.table_widget.selectionModel().selectedRows())):
            self.table_widget.removeRow(row.row())

    def rename_genre(self):
        from calibre.gui2 import error_dialog
        selected_genre = self.table_widget.get_selected_genre()
        if not selected_genre:
            return
        new_genre_name, ok = QInputDialog.getText(self.parent, _('Add new mapping'),
                    _('Enter a Databazeknih %s name to create a mapping for:')%self.name, text=selected_genre)
        if not ok:
            # Operation cancelled
            return
        new_genre_name = as_unicode(new_genre_name).strip().replace(',', ';')
        if not new_genre_name or new_genre_name == selected_genre:
            return
        data = self.table_widget.get_data()
        if new_genre_name.lower() != selected_genre.lower():
            # Verify it does not clash with any other mappings in the list
            for genre_name in data.keys():
                if genre_name.lower() == new_genre_name.lower():
                    return error_dialog(self.parent, _('Rename Failed'), _('A %s with the same name already exists')%self.name, show=True)
        data[new_genre_name] = data[selected_genre]
        del data[selected_genre]
        self.table_widget.populate_table(data)
        self.table_widget.select_genre(new_genre_name)

    def reset_to_defaults(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self.parent, _('Are you sure?'),
            _('Are you sure you want to reset to the plugin default %s mappings?')%self.name,
                show_copy_button=False):
            return
        self.table_widget.populate_table(DEFAULT_STORE_VALUES[self.prefs])

        #self.tab_widget.addTab(genre_group_box, _('Genre/Tag'))

class CommentsTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)
        layout = QVBoxLayout()
        self.setLayout(layout)
        # Comment builder
        comment_group_box = QGroupBox(_('Comment builder (What will be parsed into Comments field.)'))
        #self.l.addWidget(comment_group_box, self.l.rowCount()-3, 2, 3, 1)
        comment_group_box_layout = QVBoxLayout()
        layout.addWidget(comment_group_box)
        comment_group_box.setLayout(comment_group_box_layout)

        table_layout = QHBoxLayout()
        comment_group_box_layout.addLayout(table_layout)
        self.comments_arrange_table = CommentBuilderTableWidget(self, c.get(APPEND_TO_COMMENTS, DEFAULT_STORE_VALUES[APPEND_TO_COMMENTS]))
        table_layout.addWidget(self.comments_arrange_table)
        self.comments_arrange_table.populate_table(c.get(APPEND_TO_COMMENTS, DEFAULT_STORE_VALUES[APPEND_TO_COMMENTS]))

        table_button_layout = QVBoxLayout()
        table_layout.addLayout(table_button_layout)
        plus_button = QToolButton(self)
        plus_button.setToolTip(_('Add item (Alt+A)'))
        plus_button.setIcon(QIcon(I('plus.png')))
        plus_button.setShortcut(_('Alt+A'))
        plus_button.clicked.connect(self.add_comment_item)
        table_button_layout.addWidget(plus_button)
        minus_button = QToolButton(self)
        minus_button.setToolTip(_('Delete item (Alt+D)'))
        minus_button.setIcon(QIcon(I('minus.png')))
        minus_button.setShortcut(_('Alt+D'))
        minus_button.clicked.connect(self.delete_comment_item)
        table_button_layout.addWidget(minus_button)

        add_all_button = QToolButton(self)
        add_all_button.setToolTip(_('Add all items at once (Alt+Q)'))
        add_all_button.setIcon(QIcon(I('sort.png')))
        add_all_button.setShortcut(_('Alt+Q'))
        add_all_button.clicked.connect(self.add_all_comment_items)
        table_button_layout.addWidget(add_all_button)

        reset_button = QToolButton(self)
        reset_button.setToolTip(_('Reset to defaults (Alt+R)'))
        reset_button.setIcon(QIcon(I('restart.png')))
        reset_button.setShortcut(_('Alt+R'))
        reset_button.clicked.connect(self.reset_comment_items)
        table_button_layout.addWidget(reset_button)

        move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move item up (Alt+W)'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        move_up_button.setShortcut(_('Alt+Up'))
        move_up_button.setShortcut(_('Alt+W'))
        move_up_button.clicked.connect(self.move_rows_up)
        table_button_layout.addWidget(move_up_button)
        move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move item down (Alt+S)'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        move_down_button.setShortcut(_('Alt+Down'))
        move_down_button.setShortcut(_('Alt+S'))
        move_down_button.clicked.connect(self.move_rows_down)
        table_button_layout.addWidget(move_down_button)

        self.item_options = [
        (DESCRIPTION, True, _('Description')),
        (HR, True, _('——— (Horizontal line)')),
        (PAGES, True, _('Pages count')),
        (COVER_TYPE, True, _('Cover type')),
        (PRINT_RUN, True, _('Print run')),
        (ORIGINAL_TITLE, True, _('Original title')),
        (TITLE, True, _('Title')),
        (ORIGINAL_YEAR, True, _('Original pubdate')),
        (PUB_YEAR, True, _('Publication date')),
        (PUBLISHER, True, _('Publisher')),
        (AUTHORS, True, _('Authors')),
        (REAL_AUTHORS, True, _('Real authors')),
        (TRANSLATION, True, _('Translators')),
        (ILLUSTRATION, True, _('Illustrators')),
        (COVER_AUTHORS, True, _('Book cover authors')),
        (SERIES, True, _('Series')),
        (EDITION, True, _('Book Edition')),
        (GENRES, True, _('Genre')),
        (TAGS, True, _('Tags')),
        (RATING, True, _('Rating (%)')),
        (RATING10, True, _('Rating (0-10)')),
        (RATING5, True, _('Rating (0-5)')),
        (RATING_COUNT, True, _('Rating count')),
        (ISBN, True, _('ISBN')),
        (ORIG_DK, True, _('Identifier databazeknih')),
        (AWARDS, True, _('Awards')),
        (TRIVIA, True, _('Trivia')),
        (TALES_IN_BOOK, True, _('Tales in book'))
        ]

    def move_rows_up(self):
        self.comments_arrange_table.setFocus()
        rows = self.comments_arrange_table.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        first_sel_row = rows[0].row()
        if first_sel_row <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in selrows:
            self.comments_arrange_table.swap_row_widgets(selrow - 1, selrow + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_row - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.comments_arrange_table.scrollToItem(self.comments_arrange_table.item(scroll_to_row, 0))
        #self.renumber_series()

    def move_rows_down(self):
        self.comments_arrange_table.setFocus()
        rows = self.comments_arrange_table.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        last_sel_row = rows[-1].row()
        if last_sel_row == self.comments_arrange_table.rowCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in reversed(selrows):
            self.comments_arrange_table.swap_row_widgets(selrow + 2, selrow)

        scroll_to_row = last_sel_row + 1
        if scroll_to_row < self.comments_arrange_table.rowCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.comments_arrange_table.scrollToItem(self.comments_arrange_table.item(scroll_to_row, 0))
        # self.renumber_series()

    def add_all_comment_items(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete your order and add all available items?'),
                show_copy_button=False):
            return
        self.comments_arrange_table.setRowCount(0)
        #new_items = self.comments_arrange_table.get_data()
        new_items = ([i for i in self.item_options])
        self.comments_arrange_table.populate_table(new_items)
        self.comments_arrange_table.selectRow(self.comments_arrange_table.rowCount() - 1)

    def add_comment_item(self):
        self.comments_arrange_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to comments"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.comments_arrange_table.get_data()
            new_items.append([i for i in self.item_options if item in i][0])
            self.comments_arrange_table.populate_table(new_items)
            self.comments_arrange_table.selectRow(self.comments_arrange_table.rowCount() - 1)
        if not item:
            return

    def delete_comment_item(self):
        from calibre.gui2 import question_dialog
        self.comments_arrange_table.setFocus()
        if not self.comments_arrange_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for row in reversed(sorted(self.comments_arrange_table.selectionModel().selectedRows())):
            self.comments_arrange_table.removeRow(row.row())

    def reset_comment_items(self):
        from calibre.gui2 import question_dialog
        self.comments_arrange_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set comment builder to defaults?'),
                show_copy_button=False):
            return
        self.comments_arrange_table.setRowCount(0)
        self.comments_arrange_table.populate_table(DEFAULT_STORE_VALUES[APPEND_TO_COMMENTS])

class CommentBuilderTableWidget(QTableWidget):

    def __init__(self, parent, comment_items):
        QTableWidget.__init__(self, parent)
        self.comment_items = comment_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setMinimumSize(50, 0)

    def populate_table(self, comment_items):
        self.clear()
        self.setRowCount(len(comment_items))
        header_labels = [_('Comment item')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(comment_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyTableWidgetItem(item[2])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        comment_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str(self.item(row, 0).text()))
            #active = self.item(row, 0).get_boolean_value()
            comment_items.append((item_id, True, item_name))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", comment_items)
        return comment_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class MappingsTableWidget(QTableWidget):
    def __init__(self, parent, all_tags, headers=(_('genre'), _('Tag(s)'))):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tags_values = all_tags
        self.headers = headers

    def populate_table(self, tag_mappings):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(tag_mappings))
        header_labels = ['databazeknih ' + self.headers[0], _('Maps to Calibre') + ' ' + self.headers[1]]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)
        self.horizontalHeader().setStretchLastSection(True)

        for row, genre in enumerate(sorted(tag_mappings.keys(), key=lambda s: (s.lower(), s))):
            self.populate_table_row(row, genre, sorted(tag_mappings[genre]))

        self.set_minimum_column_width(0, 300)
        self.resizeColumnToContents(0)
        self.setSortingEnabled(False)
        if len(tag_mappings) > 0:
            self.selectRow(0)

    def set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, genre, tags):
        self.setItem(row, 0, ReadOnlyTableWidgetItem(genre))
        tags_value = ', '.join(tags)
        # Add a widget under the cell just for sorting purposes
        self.setItem(row, 1, QTableWidgetItem(tags_value))
        self.setCellWidget(row, 1, self.create_tags_edit(tags_value, row))

    def create_tags_edit(self, value, row):
        try:
            from calibre.gui2.complete2 import EditWithComplete
            tags_edit = EditWithComplete(self)
            tags_edit.set_add_separator(False)
            tags_edit.update_items_cache(self.tags_values)
            tags_edit.setText(value)
        except ImportError:
            from calibre.gui2.complete import MultiCompleteComboBox
            tags_edit = MultiCompleteComboBox(self)
            tags_edit.set_add_separator(False)
            tags_edit.update_items_cache(self.tags_values)
            tags_edit.setText(value)

        #tags_edit.editingFinished.connect(partial(self.tags_editing_finished, row, tags_edit))
        return tags_edit

    def tags_editing_finished(self, row, tags_edit):
        # Update our underlying widget for sorting
        self.item(row, 1).setText(tags_edit.text())

    def get_data(self):
        tag_mappings = {}
        for row in range(self.rowCount()):
            genre = as_unicode(self.item(row, 0).text()).strip()
            tags_text = as_unicode(self.cellWidget(row, 1).text()).strip()
            tag_values = tags_text.split(',')
            tags_list = []
            for tag in tag_values:
                if len(tag.strip()) > 0:
                    tags_list.append(tag.strip())
            tag_mappings[genre] = tags_list
        return tag_mappings

    def select_genre(self, genre_name):
        for row in range(self.rowCount()):
            if as_unicode(self.item(row, 0).text()) == genre_name:
                self.setCurrentCell(row, 1)
                return

    def get_selected_genre(self):
        if self.currentRow() >= 0:
            return as_unicode(self.item(self.currentRow(), 0).text())

class ReadOnlyTableWidgetItem(QTableWidgetItem):

    def __init__(self, text):
        if text is None:
            text = ''
        try:
            QTableWidgetItem.__init__(self, text, QTableWidgetItem.ItemType.UserType)
            self.setFlags(Qt.ItemFlag.ItemIsSelectable|Qt.ItemFlag.ItemIsEnabled)
        except (AttributeError, NameError) as e:
            QTableWidgetItem.__init__(self, text, QTableWidgetItem.UserType)
            self.setFlags(Qt.ItemIsSelectable|Qt.ItemIsEnabled)

class ReadOnlyCheckableTableWidgetItem(ReadOnlyTableWidgetItem):

    def __init__(self, text, checked=False, is_tristate=False):
        super(ReadOnlyCheckableTableWidgetItem, self).__init__(text)
        try:
            self.setFlags(Qt.ItemFlag(Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled ))
        except:
            self.setFlags(Qt.ItemFlags(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled ))
        if is_tristate:
            try:
                self.setFlags(self.flags() | Qt.ItemFlag.ItemIsAutoTristate)
            except:
                self.setFlags(self.flags() | Qt.ItemIsTristate)
        if checked:
            try:
                self.setCheckState(Qt.CheckState.Checked)
            except:
                self.setCheckState(Qt.Checked)
        else:
            if is_tristate and checked is None:
                try:
                    self.setCheckState(Qt.CheckState.PartiallyChecked)
                except:
                    self.setCheckState(Qt.PartiallyChecked)
            else:
                try:
                    self.setCheckState(Qt.CheckState.Unchecked)
                except:
                    self.setCheckState(Qt.Unchecked)

    def get_boolean_value(self):
        '''
        Return a boolean value indicating whether checkbox is checked
        If this is a tristate checkbox, a partially checked value is returned as None
        '''
        try:
            if self.checkState() == Qt.CheckState.PartiallyChecked:
                return None
            else:
                return self.checkState() == Qt.CheckState.Checked
        except:
            if self.checkState() == Qt.PartiallyChecked:
                return None
            else:
                return self.checkState() == Qt.Checked

class IdentifiersTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_plugin_prefs(STORE_NAME, fill_defaults=True)

        layout = QVBoxLayout()
        self.setLayout(layout)

        identifiers_group_boxlayout = QVBoxLayout()
        layout.addLayout(identifiers_group_boxlayout)
        #Append tags group box
        append_identifiers_group_box = QGroupBox(_('Select what will be added into "Identifiers" metadata field in Calibre'), self)
        #self.l.addWidget(append_tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        identifiers_group_boxlayout.addWidget(append_identifiers_group_box)

        table_layout = QVBoxLayout()
        append_identifiers_group_box.setLayout(table_layout)

        custom_cols_info = QLabel(_('How to create custom column from this Identificators? Check <a href="https://www.mobileread.com/forums/showpost.php?p=4361821&postcount=2">MobileRead</a> forum.'))
        custom_cols_info.setWordWrap(True)
        custom_cols_info.setOpenExternalLinks(True)
        table_layout.addWidget(custom_cols_info)
        # here because of translations
        # tuple (id, activated, description, id format)
        self.DEFAULT_IDENTIFIERS_ITEMS = [
            (ORIG_DK, True, _('Identifier databazeknih'), 'databazeknih'),
            (ISBN, True, _('ISBN'), 'isbn'),
            (RATING, False, _('Rating (%)'), 'dbk_rating100'),
            (RATING10, False, _('Rating (0-10)'), 'dbk_rating10'),
            (RATING5, False, _('Rating (0-5)'), 'dbk_rating5'),
            (RATING_COUNT, False, _('Rating count'), 'dbk_ratingcount'),
            (PAGES, False, _('Pages count'), 'dbk_pages'),
            (PRINT_RUN, False, _('Print run'), 'dbk_printrun'),
            (ORIGINAL_YEAR, False, _('Original pubdate'), 'dbk_origyear'),
            (PUB_YEAR, False, _('Publication date'), 'dbk_pubdate'),
            (COVER_TYPE, False, _('Cover type'), 'dbk_covertype'),
            (ORIGINAL_TITLE, False, _('Original title'), 'dbk_origtitle'),
            (TITLE, False, _('Title'), 'dbk_title'),
            (PUBLISHER, False, _('Publisher'), 'dbk_publisher'),
            (AUTHORS, False, _('Authors'), 'dbk_authors'),
            (REAL_AUTHORS, False, _('Real authors'), 'dbk_realauthors'),
            (TRANSLATION, False, _('Translators'), 'dbk_translators'),
            (ILLUSTRATION, False, _('Illustrators'), 'dbk_illustrators'),
            (COVER_AUTHORS, False, _('Book cover authors'), 'dbk_coverauthors'),
            (SERIES, False, _('Series'), 'dbk_series'),
            (EDITION, False, _('Book Edition'), 'dbk_edition'),
            (GENRES, False, _('Genre'), 'dbk_genres'),
            (TAGS, False, _('Tags'), 'dbk_tags'),
            (AWARDS, False, _('Awards'), 'dbk_awards'),
            (TALES_IN_BOOK, False, _('Tales in book'), 'dbk_bookparts'),
        ]
        #(DESCRIPTION, False, _('Description')),
        #(TRIVIA, False, _('Trivia')),
        identifier_items = []
        if len(c.get(APPEND_TO_IDENTIFIERS, [])) == len(self.DEFAULT_IDENTIFIERS_ITEMS):
            for i, tupl in enumerate(c.get(APPEND_TO_IDENTIFIERS, self.DEFAULT_IDENTIFIERS_ITEMS)):
                identifier_items.append((tupl[0], tupl[1], self.DEFAULT_IDENTIFIERS_ITEMS[i][2], self.DEFAULT_IDENTIFIERS_ITEMS[i][3]))
        else:
            identifier_items = self.DEFAULT_IDENTIFIERS_ITEMS

        self.identifier_items_table = IdentifiersBuilderTableWidget(self, identifier_items)
        table_layout.addWidget(self.identifier_items_table)
        self.identifier_items_table.populate_table(identifier_items)

        # Rating

        rating_group_box = QGroupBox(_('Rating'), self)
        rating_group_box_layout = QVBoxLayout()
        rating_group_box.setLayout(rating_group_box_layout)

        self.recalculate_rating_check = add_check_option(rating_group_box_layout,
                                                _('Recalculate ratings'),
                                                _('Enable this option if you want to parse more accurate rating (it will be calculated from individual ratings) from site.\nIf unchecked, only provided rating on main book page will be parsed.'),
                                                CALCULATE_RATING)
        self.recalculate_rating_spin = add_spin_option(rating_group_box_layout,
                                                _('Max decimal places'),
                                                _('With this option you can change accuracy of the float number.\n**ONLY in recalculated rating'),
                                                CALCULATE_RATING_ACCURACY, min_val=0, max_val=8)
        
        rating_group_box_layout.addStretch(1)
        layout.addWidget(rating_group_box)

        layout.addStretch(1)

class IdentifiersBuilderTableWidget(QTableWidget):

    def __init__(self, parent, identifier_items):
        QTableWidget.__init__(self, parent)
        self.identifier_items = identifier_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        #self.setMinimumSize(50, 0)

    def populate_table(self, identifier_items):
        self.identifier_items = identifier_items
        self.clear()
        self.setRowCount(len(identifier_items))
        header_labels = [_('Metadata Item'), _('Identifier format')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(identifier_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyCheckableTableWidgetItem(item[2], checked=item[1])
        identifier_format = ReadOnlyTableWidgetItem(item[3])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)
        self.setItem(row, 1, identifier_format)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        identifier_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(row, 0).text())))
            active = self.item(row, 0).get_boolean_value()
            identifier_format = as_unicode(str((self.item(row, 1).text())))
            identifier_items.append((item_id, active, item_name, identifier_format))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", identifier_items)
        return identifier_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class LineBuilderTableWidget(QTableWidget):

    def __init__(self, parent, identifier_items, header_labels=[_('Metadata Item')]):
        QTableWidget.__init__(self, parent)
        self.identifier_items = identifier_items
        self.setSortingEnabled(False)
        self.setSelectionBehavior(QAbstractItemView.SelectColumns)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setMinimumSize(0, 50)
        #self.setMaximumHeight(50)
        self.header_labels = header_labels

    def populate_table(self, identifier_items):
        self.identifier_items = identifier_items
        self.clear()
        self.setColumnCount(len(identifier_items))
        self.setRowCount(len(self.header_labels))
        self.setVerticalHeaderLabels(self.header_labels)
        self.horizontalHeader().setDefaultSectionSize(32)
        self.verticalHeader().setStretchLastSection(True)

        for col, item in enumerate(identifier_items):
            self.populate_table_col(col, item)

        #self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectColumn(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_col(self, col, item):
        name_widget = ReadOnlyTableWidgetItem(item[2])
        if item[0] != CUSTOM_TEXT:
            try:
                font = name_widget.font()
                font.setBold(True)
                name_widget.setFont(font)
            except:
                font = name_widget.font()
                font.setBold(True)
                name_widget.setFont(font)
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(0, col, name_widget)
        self.resizeColumnsToContents()

    def get_data(self):
        identifier_items = []
        for col in range(self.columnCount()):
            try:
                item_id = convert_qvariant(self.item(0, col).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(0, col).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(0, col).text())))
            identifier_items.append((item_id, True, item_name))
        return identifier_items
    
    def get_string(self):
        return ''.join([tupl[2] for tupl in self.get_data()])

    def swap_row_widgets(self, src_col, dest_col):
        self.blockSignals(True)
        self.insertColumn(dest_col)
        for row in range(self.rowCount()):
            self.setItem(row, dest_col, self.takeItem(row, src_col))
        self.removeColumn(src_col)
        self.blockSignals(False)
        self.resizeColumnsToContents()

    def select_and_scroll_to_col(self, col):
        self.selectRow(col)
        self.scrollToItem(self.currentItem())
