# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)

import re, datetime
from collections import defaultdict

from calibre.ebooks.metadata.sources.prefs import msprefs
from calibre.ebooks.metadata.book.base import Metadata
from calibre.ebooks.metadata import check_isbn
import calibre_plugins.xTrance.config as cfg
import calibre_plugins.xTrance.globals as g

def get_config_option(config_option):
    return g.plugin_prefs[g.STORE_NAME].get(config_option,
                                                g.DEFAULT_STORE_VALUES[config_option])

__license__ = 'GPL v3'
__copyright__ = '2015, David Forrester & Sosie'
__docformat__ = 'restructuredtext en'

class XtParser:
    def __init__(self, log, plugin, mi_list, matches, meta_dict):
        self.log = log
        self.plugin = plugin
        self.mi_list = mi_list
        self.matches = matches
        self.extra_metadata = meta_dict

    def parse_first_as_list(self, root, xpath, loginfo,
                            convert=lambda x: [node.strip() for node in x[0].split(';')]):
        try:
            nodes = root.xpath(xpath)
            return convert(nodes) if nodes else None
        except Exception:
            self.log.exception('Error parsing for %s with xpath: %s' % (loginfo, xpath))
            return None

    def parse_first(self, root, xpath, loginfo, convert=lambda x: x[0].strip()):
        try:
            nodes = root.xpath(xpath)
            return convert(nodes) if nodes else None
        except Exception:
            self.log.exception('Error parsing for %s with xpath: %s' % (loginfo, xpath))
            return None

    def parse_data(self):
        self.log.info('\nParsing %s results...'%len(self.matches))
        for i, match in enumerate(self.matches):
            root, author_match_relevance, title_relevance = match
            xtrance_id = self.parse_first(root, 'xtrance_id/text()', 'xtrance_id')
            title = self.parse_first(root, 'title/text()', 'title')
            subtitle = self.parse_first(root, 'podtitul/text()', 'subtitle')
            orig_title = self.parse_first(root, 'nazev_orig/text()', 'orig_title')
            original_publisher = self.parse_first(root,
                                                  'vydavatel_orig/text()',
                                                  'original_publisher')
            publisher_list = self.parse_first_as_list(root, 'publisher/text()', 'publisher')
            publisher = publisher_list[0] if publisher_list else None
            isbn = self.parse_first(root, 'identifier/text()', 'isbn')
            isbn = self.find_first_isbn(isbn) if isbn else None
            book_dimensions = self.parse_first(root, 'rozmer/text()', 'book_dimensions')
            language = self.parse_first(root, 'language/text()', 'language')
            orig_lang = self.parse_first(root, 'jazyk_orig/text()', 'orig_lang')

            authors = self.parse_first_as_list(root, 'author/text()', 'authors')
            illustrators = self.parse_first_as_list(root, 'ilustrator/text()', 'illustrators')
            translators = self.parse_first_as_list(root, 'prekladatel/text()', 'translators')
            tags = self.parse_first_as_list(root, 'tag/text()', 'tags')

            description = self.parse_first(root, 'description/text()', 'description')
            description = self.convert_bbcode_to_html(description) if description else None
            book_edition_list = self.parse_first_as_list(root, 'edice/text()', 'book_edition')
            book_edition_index_list = self.parse_first_as_list(root,
                                                               'edice_index/text()',
                                                               'book_edition_index')
            series_list = self.parse_first_as_list(root,
                                                   'calibre_series/text()',
                                                   'series')
            series_index_list = self.parse_first_as_list(root,
                                                         'series_index/text()',
                                                         'series_index')
            if series_list and series_index_list and len(series_list) == len(series_index_list):
                series = series_list[0]
                try:
                    series_index = series_index_list[0]
                    if series_index:
                        num_search = re.search(r'(\d+)', series_index)
                        series_index = num_search.group() if num_search else 0
                        series_index = float(series_index)
                    else:
                        series_index = 0.0
                except Exception as e:
                    self.log.exception('*** Error in series index parsing: %s'%(e))
                    series_index = None
            elif series_list:
                series = series_list[0]
                series_index = 0.0
            else:
                series = series_index = None
            if book_edition_list and book_edition_index_list and \
                len(book_edition_list) == len(book_edition_index_list):
                book_edition = book_edition_list[0]
                try:
                    book_edition_index = book_edition_index_list[0]
                    if book_edition_index:
                        num_search = re.search(r'(\d+)', book_edition_index)
                        book_edition_index = num_search.group() if num_search else 0
                        book_edition_index = float(book_edition_index)
                    else:
                        book_edition_index = 0.0
                except Exception as e:
                    self.log.exception('*** Error in book edition index parsing: %s'%(e))
                    book_edition_index = None
            elif book_edition_list:
                book_edition = book_edition_list[0]
                book_edition_index = 0.0
            else:
                book_edition = book_edition_index = None

            pages = self.parse_first(root, 'pocet_stran/text()', 'pages',
                                     convert=lambda x: x[0].strip('s. '))
            img_number = self.parse_first(root, 'pocet_obrazku/text()', 'img_number',
                                     convert=lambda x: x[0].strip('s. '))
            print_run = self.parse_first(root, 'naklad/text()', 'print_run',
                                         convert=lambda x: x[0].strip().replace(' ', ''))
            issue_number = self.parse_first(root, 'vydani/text()', 'issue_number')
            rating = self.parse_first(root, 'rating/text()', 'rating',
                                      convert=lambda x: float(x[0].strip())/20)
            rating100 = self.parse_first(root, 'rating/text()', 'rating',
                                         convert=lambda x: int(x[0].strip()))
            rating10 = self.parse_first(root, 'rating/text()', 'rating',
                                        convert=lambda x: float(x[0].strip())/10)
            pub_date = self.parse_first(root, 'pub_date/text()', 'pub_date')
            pub_date = self.convert_date_from_xml(pub_date) if pub_date else None
            orig_pubdate = self.parse_first(root, 'vydano_orig/text()', 'orig_pubdate')
            orig_pubdate = self.convert_date_from_xml(orig_pubdate) if orig_pubdate else None
            cover = self.parse_first(root, 'image/text()', 'cover')

            # basic metadata
            mi = Metadata(title, authors)
            if cover:
                mi.cover_url = cover
                self.plugin.cache_identifier_to_cover_url(xtrance_id, [xtrance_id])
            mi.isbn = check_isbn(isbn)
            mi.set_identifier('xtrance', xtrance_id)
            mi.tags = tags
            mi.comments = description
            mi.language = language
            mi.series = series
            mi.series_index = series_index
            mi.rating = rating
            mi.publisher = publisher
            mi.pubdate = pub_date
            mi.has_cover = bool(cover)
            mi.source_relevance = i

            # for issue preference handling
            mi.author_match_relevance = author_match_relevance
            mi.title_relevance = title_relevance
            mi.pubyear = int(pub_date.year) if pub_date else 0

            # additional metadata
            mi.xtrance_id = xtrance_id if xtrance_id else None
            mi.translators = translators if translators else None
            mi.illustrators = illustrators if illustrators else None
            mi.subtitle = subtitle if subtitle else None
            mi.orig_title = orig_title if orig_title else None
            mi.original_publisher = original_publisher if original_publisher else None
            mi.book_dimensions = book_dimensions.replace(',', '.') if book_dimensions else None
            mi.img_number = img_number if img_number else None
            mi.orig_lang = orig_lang if orig_lang else None
            mi.orig_pubdate = orig_pubdate if orig_pubdate else None
            mi.print_run = print_run if print_run else None
            mi.pages = pages if pages else None
            mi.issue_number = issue_number if issue_number else None
            mi.rating100 = rating100 if rating100 else None
            mi.rating10 = rating10 if rating10 else None
            mi.book_edition = book_edition if book_edition_list else None
            mi.book_edition_index = book_edition_index if book_edition_index else None

            self.plugin.clean_downloaded_metadata(mi)
            self.mi_list.append(mi)
            self.log.info('-- parsed book: %s; author : %s; ISBN : %s' % (title, authors, isbn))

        self._additional_metadata_handling()
        self._compare_issues()

    def _additional_metadata_handling(self):
        self.log.info('\nNow handle additional user preferences'
                      '(remapping, basic field editing etc.)...')
        # setting preferences
        authors_include = get_config_option(g.AUTHORS_INCLUDE)
        translators_include = get_config_option(g.TRANSLATORS_INCLUDE)
        illustrators_include = get_config_option(g.ILLUSTRATORS_INCLUDE)
        swap_authors = get_config_option(g.SWAP_AUTHORS)
        author_role = get_config_option(g.AUTHOR_ROLE)
        only_one_author = get_config_option(g.ONE_AUTHOR)

        lang_include = get_config_option(g.LANGUAGE_INCLUDE)

        publication_date = get_config_option(g.PUBLICATION_DATE)

        title_line = get_config_option(g.TITLE_LINE)
        publisher_line = get_config_option(g.PUBLISHER_LINE)
        series_line = get_config_option(g.SERIES_LINE)
        series_index_field = get_config_option(g.SERIES_INDEX_FIELD)

        append_to_tags = get_config_option(g.APPEND_TO_TAGS)
        append_to_identifiers = get_config_option(g.APPEND_TO_IDENTIFIERS)
        append_to_comments = get_config_option(g.APPEND_TO_COMMENTS)

        for mi in self.mi_list:
            # convert info back to local vars
            authors = mi.authors if mi.authors else []
            mi.authors = []
            title = mi.title
            mi.title = None
            tags = mi.tags if mi.tags else []
            mi.tags = []
            publisher = mi.publisher
            mi.publisher = None
            comments = '<p id="description">%s</p>'%mi.comments if mi.comments else ''
            mi.comments = ''
            pubdate = mi.pubdate
            pubyear = mi.pubdate.year if mi.pubdate else None
            mi.pubdate = None
            series, series_index = mi.series, mi.series_index
            edition, edition_index = mi.book_edition, mi.book_edition_index
            isbn = mi.isbn
            language = mi.language
            rating = mi.rating
            mi.language = None

            source_relevance = mi.source_relevance+1
            original_title = mi.orig_title
            original_year = mi.orig_pubdate.year if mi.orig_pubdate else None
            rating10 = mi.rating10
            rating100 = mi.rating100
            xtrance_id = mi.xtrance_id
            translators = mi.translators if mi.translators else []
            illustrators = mi.illustrators if mi.illustrators else []
            pages = mi.pages
            print_run = mi.print_run
            issue_number = mi.issue_number
            book_dimensions = mi.book_dimensions
            img_number = mi.img_number
            subtitle = mi.subtitle
            original_publisher = mi.original_publisher
            original_language = mi.orig_lang

            tags = self._convert_genres_to_calibre_tags(tags, get_config_option(g.TAGS_FILTER))
            series = self._convert_to_calibre(series, get_config_option(g.SERIES_MAPPINGS),
                                              get_config_option(g.SERIES_FILTER)) if series else None
            edition = self._convert_to_calibre(edition, get_config_option(g.SERIES_MAPPINGS),
                                               get_config_option(g.SERIES_FILTER)) if edition else None
            publisher = self._convert_to_calibre(publisher, get_config_option(g.PUBLISHER_MAPPINGS),
                                                 get_config_option(g.PUBLISHER_FILTER)) if publisher else None
            original_publisher = self._convert_to_calibre(original_publisher,
                                                          get_config_option(g.PUBLISHER_MAPPINGS),
                                                          get_config_option(g.PUBLISHER_FILTER)) if original_publisher else None

            # authors field creating
            authors_to_add = []
            if author_role:
                if authors_include:
                    authors_to_add.extend(['%s (Autor)'%a for a in authors])
                if translators_include and translators:
                    authors_to_add.extend(['%s (Překlad)'%a for a in translators])
                if illustrators_include and illustrators:
                    authors_to_add.extend(['%s (Ilustrace)'%a for a in illustrators])
            else:
                if authors_include:
                    authors_to_add.extend(authors)
                if translators_include and translators:
                    authors_to_add.extend(translators)
                if illustrators_include and illustrators:
                    authors_to_add.extend(illustrators)

            # Add only first
            if (only_one_author == 1 and authors_to_add and len(authors_to_add) > 1) or len(authors_to_add) == 1:
                authors_to_add = authors_to_add[:1]
                first_a = authors_to_add[0].split()
                if first_a[-1] in ('(Autor)'): # if (Autor) only.. don't write role
                    first_a.pop()
                authors_to_add[0] = ' '.join(first_a)

            # Swap authors from FN LN to LN FN format
            lookup = set()  # a temporary lookup set
            authors_to_add = [a for a in authors_to_add if a not in lookup and lookup.add(a) is None]
            if swap_authors and not msprefs.get('swap_author_names'):
                swapped = []
                for a in authors_to_add:
                    auth_parts = a.split()
                    if auth_parts[-1] in ('(Autor)', '(Ilustrace)', '(Překlad)'):
                        role = auth_parts.pop()
                        swapped.append('%s %s %s' %(auth_parts[-1], ' '.join(auth_parts[:-1]), role))
                    else:
                        swapped.append('%s %s' %(auth_parts[-1], ' '.join(auth_parts[:-1])))
                authors_to_add = swapped
            elif msprefs.get('swap_author_names'):
                swapped = []
                for a in authors_to_add:
                    auth_parts = a.split()
                    if auth_parts[-1] in ('(Autor)', '(Ilustrace)', '(Překlad)'):
                        role = auth_parts.pop()
                        swapped.append('%s %s %s' %(' '.join(auth_parts[:-1]), role, auth_parts[-1]))
                    else:
                        swapped.append('%s %s' %(auth_parts[-1], ' '.join(auth_parts[:-1])))
                authors_to_add = swapped

            mi.authors = authors_to_add

            if xtrance_id and not dict([[tupl[0], tupl[1]] for tupl in append_to_identifiers]).get(g.XTRANCE, True):
                mi.set_identifier('xtrance', None)
            
            if isbn and not dict([[tupl[0], tupl[1]] for tupl in append_to_identifiers]).get(g.ISBN, True):
                mi.set_identifier('isbn', None)
                mi.isbn = None

            # 0: found book pubdate, 1: first book pubdate
            if publication_date == 0 and pubdate:
                mi.pubdate = pubdate
            elif publication_date == 1 and original_year:
                mi.pubdate = original_year

            if lang_include == 0 and language:
                mi.language = language
            elif lang_include == 1 and original_language:
                mi.language = original_language

            try:
                mi.comments = ''
                # append_to_comments - list of tuples
                # (id_string, check_bool, visible_desc)
                def get_comment_variable(argument):
                    if argument == g.DESCRIPTION and comments:
                        return comments
                    elif argument == g.HR:
                        return '<hr>'
                    elif argument == g.PAGES and pages:
                        return '<p id="numberOfPages"><em>Počet stran:</em> %s</p>' %pages
                    elif argument == g.SOURCE_RELEVANCE and source_relevance:
                        return '<p id="sourceRelevance"><em>Pořadí ve vyhledávání:</em> %s</p>' %source_relevance
                    elif argument == g.PRINT_RUN and print_run:
                        return '<p id="printRun"><em>Náklad (ks):</em> %s</p>' %print_run
                    elif argument == g.ISSUE_NUMBER and issue_number:
                        return '<p id="issueNumber"><em>Vydání:</em> %s</p>' %issue_number
                    elif argument == g.IMG_NUMBER and img_number:
                        return '<p id="imgNumber"><em>Počet obrázků:</em> %s</p>' %img_number
                    elif argument == g.BOOK_DIMENSIONS and book_dimensions:
                        return '<p id="bookDimensions"><em>Rozměry:</em> %s</p>' %book_dimensions
                    elif argument == g.TITLE and title:
                        return '<p id="title"><em>Název:</em> %s</p>' %title
                    elif argument == g.SUBTITLE and subtitle:
                        return '<p id="subtitle"><em>Podtitul:</em> %s</p>' %subtitle
                    elif argument == g.ORIGINAL_TITLE and original_title:
                        return '<p id="origTitle"><em>Původní název:</em> %s</p>' %original_title
                    elif argument == g.ORIGINAL_YEAR and original_year:
                        return '<p id="origYear"><em>Rok vydání originálu:</em> %s</p>' %original_year
                    elif argument == g.PUB_YEAR and pubyear:
                        return '<p id="pubYear"><em>Rok vydání:</em> %s</p>' %pubyear
                    elif argument == g.PUBLISHER and publisher:
                        return '<p id="publisher"><em>Vydavatel:</em> %s</p>' %publisher
                    elif argument == g.ORIGINAL_PUBLISHER and original_publisher:
                        return '<p id="origPublisher"><em>Vydavatel originálu:</em> %s</p>' %original_publisher
                    elif argument == g.LANGUAGE and language:
                        return '<p id="language"><em>Jazyk:</em> %s</p>' %language
                    elif argument == g.ORIGINAL_LANGUAGE and original_language:
                        return '<p id="origLanguage"><em>Jazyk originálu:</em> %s</p>' %original_language
                    elif argument == g.RATING and rating:
                        return '<p id="rating100"><em>Hodnocení (%%):</em> %s</p>' %rating100
                    elif argument == g.RATING10 and rating10:
                        return '<p id="rating10"><em>Hodnocení (0-10):</em> %s</p>' %rating10
                    elif argument == g.RATING5 and rating:
                        return '<p id="rating5"><em>Hodnocení (0-5):</em> %s</p>' %rating
                    elif argument == g.ISBN and isbn:
                        return '<p id="isbn"><em>ISBN:</em> %s </p>' %isbn
                    elif argument == g.XTRANCE and xtrance_id:
                        return '<p id="xtrance"><em>xtrance:</em> %s</p>' %xtrance_id
                    elif argument == g.SERIES and series:
                        if series_index:
                            return '<p id="series"><em>Série:</em> %s [%s]</p>' %(series, series_index)
                        else:
                            return '<p id="series"><em>Série:</em> %s</p>' %series
                    elif argument == g.EDITION and edition:
                        if edition_index:
                            return '<p id="edition"><em>Edice:</em> %s [%s]</p>' %(edition, edition_index)
                        else:
                            return '<p id="edition"><em>Edice:</em> %s</p>' %edition
                    elif argument == g.AUTHORS and authors:
                        return '<p id="authors"><em>Autoři:</em> %s</p>'%(' & '.join(authors))
                    elif argument == g.TRANSLATORS and translators:
                        return '<p id="translator"><em>Překlad:</em> %s</p>'%(' & '.join(translators))
                    elif argument == g.ILLUSTRATORS and illustrators:
                        return '<p id="illustrators"><em>Ilustrace:</em> %s</p>'%(' & '.join(illustrators))
                    elif argument == g.TAGS and tags:
                        return '<p id="tags"><em>Štítky:</em> %s</p>' %', '.join(tags)
                    else:
                        return ''
                
                for item in append_to_comments:
                    if item[1]:
                        # converts parsed information into html paragraph string                  
                        mi.comments = '%s %s'%(mi.comments, get_comment_variable(item[0]))
            except:
                self.log.exception('Error adding more info to comments for mi title: %r'%mi.title)
                mi.comments = comments
                        
            try:
                def get_tag_variable(argument):
                    switch_check = {
                        g.TAGS: tags,
                        g.PAGES: pages,
                        g.SOURCE_RELEVANCE: source_relevance,
                        g.PRINT_RUN: print_run,
                        g.ISSUE_NUMBER: issue_number,
                        g.BOOK_DIMENSIONS: book_dimensions,
                        g.IMG_NUMBER: img_number,
                        g.AUTHORS: authors,
                        g.TRANSLATORS: translators,
                        g.ILLUSTRATORS: illustrators,
                        g.TITLE: title,
                        g.SUBTITLE: subtitle,
                        g.ORIGINAL_TITLE: original_title,
                        g.ORIGINAL_YEAR: original_year,
                        g.PUB_YEAR: pubyear,
                        g.PUBLISHER: publisher,
                        g.ORIGINAL_PUBLISHER: original_publisher,
                        g.LANGUAGE: language,
                        g.ORIGINAL_LANGUAGE: original_language,
                        g.SERIES: (series, series_index),
                        g.EDITION: (edition, edition_index),
                        g.RATING: rating100,
                        g.RATING5: rating,
                        g.RATING10: rating10,
                        g.ISBN: isbn,
                        g.XTRANCE: xtrance_id,
                    }
                    switch = {
                        g.TAGS: tags,
                        g.PAGES: ['Počet stran: %s' %pages],
                        g.SOURCE_RELEVANCE: ['Pořadí ve vyhledávání: %s' %source_relevance],
                        g.PRINT_RUN: ['Náklad (ks): %s' %print_run],
                        g.ISSUE_NUMBER: ['Vydání: %s' %issue_number],
                        g.IMG_NUMBER: ['Počet obrázků: %s' %img_number],
                        g.BOOK_DIMENSIONS: ['Rozměry: %s' %book_dimensions],
                        g.LANGUAGE: ['Jazyk: %s' %language],
                        g.ORIGINAL_LANGUAGE: ['Jazyk originálu: %s' %original_language],
                        g.ORIGINAL_YEAR: ['Rok vydání originálu: %s' %original_year],
                        g.PUB_YEAR: ['Rok vydání: %s' %pubyear],
                        g.RATING: ['Hodnocení (%%): %s' %rating],
                        g.RATING5: ['Hodnocení (0-5): %s' %rating],
                        g.RATING10: ['Hodnocení (0-10): %s' %rating10],
                        g.ISBN: ['ISBN: %s' %isbn],
                        g.XTRANCE: ['xtrance: %s' %xtrance_id]
                    }
                    if switch_check.get(argument, None):
                        if argument == g.TITLE and title:
                            return ['Název: %s' %title.replace(',', ';')]
                        if argument == g.SUBTITLE and subtitle:
                            return ['Podtitul: %s' %subtitle.replace(',', ';')]
                        if argument == g.PUBLISHER and publisher:
                            return ['Vydavatel: %s' %publisher.replace(',', ';')]
                        if argument == g.ORIGINAL_PUBLISHER and original_publisher:
                            return ['Vydavatel originálu: %s' %original_publisher.replace(',', ';')]
                        if argument == g.ORIGINAL_TITLE and original_title:
                            return ['Původní název: %s' %original_title.replace(',', ';')]
                        elif argument == g.EDITION and edition:
                            if edition_index:
                                return ['Edice: %s [%s]' %(edition, edition_index)]
                            else:
                                return ['Edice: %s' %edition]
                        elif argument == g.SERIES and series:
                            if series_index:
                                return ['Série: %s [%s]' %(series, series_index)]
                            else:
                                return ['Série: %s' %series]
                        elif argument == g.AUTHORS and authors:
                            return ['Autor: %s'%a.replace(',', ';') for a in authors]
                        elif argument == g.TRANSLATORS and translators:
                            return ['Překlad: %s'%t.replace(',', ';') for t in translators]
                        elif argument == g.ILLUSTRATORS and illustrators:
                            return ['Ilustrace: %s'%i.replace(',', ';') for i in illustrators]
                        else:
                            return switch.get(argument, '')
                    return ''

                for item in append_to_tags:
                    if item[1]:
                        mi.tags.extend(get_tag_variable(item[0]))
            except:
                self.log.exception('Error adding additional tags for title: %r'%title)

            
            try:
                def get_identifier_variable(argument):
                    switch_check = {
                        g.TAGS: tags,
                        g.PAGES: pages,
                        g.PRINT_RUN: print_run,
                        g.SOURCE_RELEVANCE: source_relevance,
                        g.ISSUE_NUMBER: issue_number,
                        g.IMG_NUMBER: img_number,
                        g.BOOK_DIMENSIONS: book_dimensions,
                        g.AUTHORS: authors,
                        g.TRANSLATORS: translators,
                        g.ILLUSTRATORS: illustrators,
                        g.TITLE: title,
                        g.SUBTITLE: subtitle,
                        g.ORIGINAL_TITLE: original_title,
                        g.ORIGINAL_YEAR: original_year,
                        g.PUB_YEAR: pubyear,
                        g.PUBLISHER: publisher,
                        g.ORIGINAL_PUBLISHER: original_publisher,
                        g.LANGUAGE: language,
                        g.ORIGINAL_LANGUAGE: original_language,
                        g.SERIES: (series, series_index),
                        g.EDITION: (edition, edition_index),
                        g.RATING: rating100,
                        g.RATING5: rating,
                        g.RATING10: rating10,
                        g.ISBN: isbn,
                        g.XTRANCE: xtrance_id,
                    }
                    return switch_check.get(argument, '')
                for item in append_to_identifiers:
                    appending = get_identifier_variable(item[0])
                    if item[0] in [g.TAGS] and appending:
                        appending = '| '.join([a.replace(',', ';') for a in appending])
                    elif item[0] in [g.AUTHORS, g.TRANSLATORS, g.ILLUSTRATORS] and appending:
                        appending = '| '.join([a.replace(',', ';') for a in appending])
                    elif item[0] in [g.SERIES, g.EDITION] and appending:
                        if appending[0] and appending[1]:
                            appending = ' '.join([appending[0].replace(',', '|'), str(appending[1])])
                        elif appending[0]:
                            appending = appending[0].replace(',', '|')
                        else:
                            appending = None
                    elif isinstance(appending, str) and appending:
                        appending = appending.replace(',', '|')
                    if item[1] and appending:
                        mi.identifiers[item[3]] = str(appending)
            except:
                self.log.exception('Error adding extra identifiers for title: %r'%title)

            def get_metadata_variable(argument):
                switch_check = {
                    g.TAGS: ', '.join(tags),
                    g.PAGES: pages,
                    g.ISSUE_NUMBER: issue_number,
                    g.SOURCE_RELEVANCE: source_relevance,
                    g.BOOK_DIMENSIONS: book_dimensions,
                    g.IMG_NUMBER: img_number,
                    g.PRINT_RUN: print_run,
                    g.AUTHORS: ' & '.join(authors),
                    g.ILLUSTRATORS: ' & '.join(illustrators),
                    g.TRANSLATORS: ' & '.join(translators),
                    g.TITLE: title,
                    g.SUBTITLE: subtitle,
                    g.ORIGINAL_TITLE: original_title,
                    g.ORIGINAL_YEAR: original_year,
                    g.ORIGINAL_PUBLISHER: original_publisher,
                    g.LANGUAGE: language,
                    g.ORIGINAL_LANGUAGE: original_language,
                    g.PUB_YEAR: pubyear,
                    g.PUBLISHER: publisher,
                    g.SERIES: series,
                    g.SERIES_INDEX: series_index,
                    g.EDITION: edition,
                    g.EDITION_INDEX: edition_index,
                    g.RATING: rating100,
                    g.RATING5: rating,
                    g.RATING10: rating10,
                    g.ISBN: isbn,
                    g.XTRANCE: xtrance_id,
                    g.CUSTOM_TEXT: argument
                }
                return switch_check.get(argument, None)
                
            try:
                mi.publisher = ''
                for item in publisher_line:
                    appending = get_metadata_variable(item[0])

                    if appending == g.CUSTOM_TEXT and item[1]:
                        mi.publisher += str(item[2])
                    elif item[1] and appending:
                        mi.publisher += str(appending)
            except:
                mi.publisher = publisher if publisher else None
                self.log.exception('Error parsing publisher for title: %r'%title)

            try:
                mi.series = ''
                for item in series_line:
                    appending = get_metadata_variable(item[0])

                    if appending == g.CUSTOM_TEXT and item[1]:
                        mi.series += str(item[2])
                    elif item[1] and appending:
                        mi.series += str(appending)
            except:
                mi.series = series if series else None
                self.log.exception('Error parsing series for title: %r'%title)

            try:
                mi.series_index = None
                appending = get_metadata_variable(series_index_field[0])
                if series_index_field[1] and appending:
                    mi.series_index = float(appending)
            except:
                mi.series_index = series_index if series_index else None
                self.log.exception('Error parsing series_index for title: %r'%title)

            try:
                mi.title = ''
                for item in title_line:
                    appending = get_metadata_variable(item[0])

                    if appending == g.CUSTOM_TEXT and item[1]:
                        mi.title += str(item[2])
                    elif item[1] and appending:
                        mi.title += str(appending)
            except:
                mi.title = title if title else None
                self.log.exception('Error parsing title for title: %r'%title)

    def _compare_issues(self):
        self.log.info('\nNow trying to find best issue in duplicate records...')
        wanted_lang = self.extra_metadata.get('language', None)
        wanted_pubyear = self.extra_metadata.get('pubdate', None)
        wanted_publisher = self.extra_metadata.get('publisher', None)

        if not wanted_lang and not wanted_publisher and not wanted_pubyear and \
            get_config_option(g.ISSUE_PREFERENCE) == 0 and get_config_option(g.MAX_COVERS) <= 1:
            self.log.info('No need for special issue merging. '
                          'Keeping all issues for built-in merging methods.')
            return

        # Create dict with records sharing same title and authors:
        # dict[token, list[all positions with same token]]
        tokens = [' '.join((mi.title, ''.join((mi.authors)))) for mi in self.mi_list]
        issues = defaultdict(list)
        for i, item in enumerate(tokens):
            issues[item].append(i)
        issues = {k:v for k,v in issues.items() if len(v)>1}
        if not issues.keys():
            self.log.info('No duplicate issue found.')

        # preference lang cs
        wanted_lang = 'cs' if wanted_lang is None and \
            get_config_option(g.ISSUE_PREFERENCE) in (1, 2) else wanted_lang
        # preference lang sk
        wanted_lang = 'sk' if wanted_lang is None and \
            get_config_option(g.ISSUE_PREFERENCE) in (3, 4) else wanted_lang

        # find best issue according to title identifiers or issue preferences settings
        delete_indices = set()
        for k, v in issues.items():
            self.log.info('Found duplicates of same Book: ', k, ' in records: ', v)
            best_issue = v[0]
            best_relevance = 10_000 # MAX_VAL
            best_source_relevance = 10_000 # MAX_VAL
            years = [self.mi_list[issue_index].pubyear
                     for issue_index in v if self.mi_list[issue_index].pubyear]

            # preference pubdate newest
            if wanted_pubyear is None and \
                get_config_option(g.ISSUE_PREFERENCE) in (1, 3):
                wanted_pubyear = max(years)
            # preference pubdate oldest
            elif wanted_pubyear is None and \
                get_config_option(g.ISSUE_PREFERENCE) in (2, 4):
                wanted_pubyear = min(years)

            for issue_index in v:
                mi = self.mi_list[issue_index]

                # Copy source_relevance to wanted issue
                if mi.source_relevance < best_source_relevance:
                    best_source_relevance = mi.source_relevance
                else:
                    mi.source_relevance = best_source_relevance

                # Calculate relevance of each book issue
                relevance = 3
                if mi.language and wanted_lang:
                    if mi.language == wanted_lang:
                        relevance -= 1
                    else:
                        relevance += 1_000
                if mi.publisher and wanted_publisher:
                    # compare with remapped version
                    c_mi_publisher = self._convert_to_calibre(mi.publisher,
                                                              get_config_option(g.PUBLISHER_MAPPINGS),
                                                              get_config_option(g.PUBLISHER_FILTER))
                    c_wanted_publisher = self._convert_to_calibre(wanted_publisher.replace('_', ' '),
                                                                  get_config_option(g.PUBLISHER_MAPPINGS),
                                                                  get_config_option(g.PUBLISHER_FILTER))
                    if c_mi_publisher and c_wanted_publisher and \
                        c_mi_publisher.lower().replace('_', '') == c_wanted_publisher.lower().replace('_', ''):
                        relevance -= 2
                    else:
                        relevance += 1_000
                if mi.pubyear and wanted_pubyear:
                    relevance += abs(mi.pubyear - int(wanted_pubyear))

                # Save index of best relevance issue
                if relevance < best_relevance:
                    best_issue = issue_index
                    best_relevance = relevance
            self.log.info('-- issue wanted: lang:', wanted_lang,
                          ' publi:', wanted_publisher,
                          ' pubyear:', wanted_pubyear)
            self.log.info('-- closest issue: lang:', self.mi_list[best_issue].language,
                          ' publi:', self.mi_list[best_issue].publisher, 
                          ' pubyear:', self.mi_list[best_issue].pubyear)
            # Add book record indices for delete into set

            self.mi_list[best_issue].cover_urls = [self.mi_list[best_issue].xtrance_id] if self.mi_list[best_issue].cover_url else []
            for issue_index in v:
                if issue_index != best_issue:
                    # add alternative covers into best issue
                    if self.mi_list[issue_index].cover_url:
                        self.mi_list[best_issue].cover_urls.append(self.mi_list[issue_index].xtrance_id)
                    delete_indices.add(issue_index)
            if self.mi_list[best_issue].cover_urls:
                self.log.info('cover urls:', self.mi_list[best_issue].cover_urls)
                self.plugin.cache_identifier_to_cover_url(self.mi_list[best_issue].xtrance_id, self.mi_list[best_issue].cover_urls)

        # Keep only wanted book issue in mi_list
        mi_to_keep = [mi for i, mi in enumerate(self.mi_list) if i not in delete_indices]
        self.mi_list.clear()
        self.mi_list.extend(mi_to_keep)

    @staticmethod
    def _convert_genres_to_calibre_tags(genre_tags, filter_check=False):
        # for each tag, add if we have a dictionary lookup
        calibre_tag_lookup = cfg.plugin_prefs[g.STORE_NAME][g.TAGS_MAPPINGS]
        calibre_tag_map = dict((k.lower(),v) for (k,v) in calibre_tag_lookup.items())
        tags_to_add = list()
        for genre_tag in genre_tags:
            if genre_tag.lower() in calibre_tag_map:
                tags = calibre_tag_map.get(genre_tag.lower(), None)
                if tags:
                    for tag in tags:
                        if tag not in tags_to_add:
                            tags_to_add.append(tag.replace(',', ';'))
            elif filter_check:
                continue
            else:
                tag = genre_tag
                if tag not in tags_to_add:
                    tags_to_add.append(tag.replace(',', ';'))

        return list(tags_to_add)

    @staticmethod
    def	_convert_to_calibre(remap_item, prefs, filter_check=False):
        # for each tag, add if we have a dictionary lookup
        calibre_map = dict((k.lower(),v) for (k,v) in prefs.items())
        calibre_remap_item = calibre_map.get(remap_item.lower(), None)
        if calibre_remap_item:
            return calibre_remap_item[0]
        elif filter_check:
            return None
        else:
            return remap_item

    @staticmethod
    def find_first_isbn(text):
        isbn_re = re.compile(
            r'(ISBN[-]*(1[03])*[ ]*(: ){0,1})*(([0-9Xx][- ]*){13})|([0-9Xx][- ]*){10}|'
            r'([0-9]{2}[-][0-9]{3}[-][0-9]{2})')
        isbn_match = re.search(isbn_re, text)
        return isbn_match.group() if isbn_match else None

    @staticmethod
    def convert_date_from_xml(date_str):
        date = re.search(r'\d{4}-\d{1,2}-\d{1,2}', date_str)
        date = datetime.datetime.strptime(date.group(), '%Y-%m-%d') if date else None
        if not date:
            date = re.search(r'\d{4}', date_str)
            date = datetime.datetime.strptime(date.group(), '%Y') if date else None
        return date

    @staticmethod
    def convert_bbcode_to_html(text):
        # Convert newlines
        text = re.sub(r'\r\n', '<br>', text)

        # All known tags map
        words_map = {'[i]': '<em>',
                    '[/i]': '</em>',
                    '[b]': '<strong>',
                    '[/b]': '</strong>',
                    '[u]': '<u>',
                    '[/u]': '</u>',
                    '[monospace]': '<code>',
                    '[/monospace]': '</code>',
                    '[green]': '<span style="color: #00f900">',
                    '[red]': '<span style="color: #ff2600">',
                    '[blue]': '<span style="color: #0433ff">',
                    '[/green]': '</span>',
                    '[/red]': '</span>',
                    '[/blue]': '</span>',
                    '[line][/line]': '<hr>',
                    '[line]': '<hr>',
                    ':lol:': '&#128514;', # emojis
                    ':wink:': '&#128521;',
                    ':shock:': '&#128563;',
                    ':sad:': '&#128532;',
                    ':oops:': '&#128542;',
                    ':cry:': '&#128546;',
                    ':evil:': '&#128127;',
                    ':twisted:': '&#128520;',
                    ':inlove:': '&#128525;',
                    ':roll:': '&#128580;',
                    ':sick:': '&#129298;',
                    ':heart:': '&#129505;',
                    ':dead:': '&#128128;',
                    ':wounded:': '&#129301;',
                    ':ghost:': '&#128123;',
                    ':snowman:': '&#9924;',
                    ':toiletpaper:': '&#129531;',
                    ':idea:': '&#128161;',
                    ':pig:': '&#128055;',
                    ':mrgreen:': '&#128527;',
                    ':arrow:': '&#10145;',
                    ':?:': '&#10067;',
                    ':!:': '&#10071;',
                    '8-)': '&#128123;',
                    ':P': '&#128523;',
                    ':-P': '&#128541;',
                    ':-D': '&#128512;',
                    ':)': '&#128522;',
                    ':-S': '&#128533;',
                    ':-O': '&#128558;',
                    ':x': '&#128545;',
                    ':|': '&#128528;',
                    }
        for word in words_map.keys():
            text = text.replace(word, words_map[word])

        # Convert url tag
        pattern = r'\[url=(.*?)\](.*?)\[/url\]'
        replacement = r'<a href="\1">\2</a>'
        text = re.sub(pattern, replacement, text)
        return text
