#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2024 seeder'
__docformat__ = 'restructuredtext en'

try:
    from qt.core import  Qt, QTableWidgetItem, QTableWidget, QAbstractItemView
except ImportError:
    try:
        from PyQt5.Qt import Qt, QTableWidgetItem, QTableWidget, QAbstractItemView
    except ImportError:
        from PyQt4.Qt import Qt, QTableWidgetItem, QTableWidget, QAbstractItemView
try:
    load_translations()
except NameError:
    pass # load_translations() added in calibre 1.9

import calibre_plugins.xTrance.globals as g
# python/pyqt backwards compability
from calibre import as_unicode
try:
    from calibre.gui2 import QVariant
    del QVariant
except ImportError:
    is_qt4 = False
    convert_qvariant = lambda x: x
else:
    is_qt4 = True
    def convert_qvariant(x):
        vt = x.type()
        if vt == x.String:
            return as_unicode(x.toString())
        if vt == x.List:
            return [convert_qvariant(i) for i in x.toList()]
        return x.toPyObject()

class TagBuilderTableWidget(QTableWidget):

    def __init__(self, parent, tag_items):
        QTableWidget.__init__(self, parent)
        self.tag_items = tag_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        #self.setMinimumSize(50, 0)

    def populate_table(self, tag_items):
        self.tag_items = tag_items
        self.clear()
        self.setRowCount(len(tag_items))
        header_labels = [_('Tag item')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(tag_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyCheckableTableWidgetItem(item[2], checked=item[1])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        tag_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(row, 0).text())))
            active = self.item(row, 0).get_boolean_value()
            tag_items.append((item_id, active, item_name))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", tag_items)
        return tag_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class CommentBuilderTableWidget(QTableWidget):

    def __init__(self, parent, comment_items):
        QTableWidget.__init__(self, parent)
        self.comment_items = comment_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setMinimumSize(50, 0)

    def populate_table(self, comment_items):
        self.clear()
        self.setRowCount(len(comment_items))
        header_labels = [_('Comment item')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(comment_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyTableWidgetItem(item[2])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        comment_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str(self.item(row, 0).text()))
            #active = self.item(row, 0).get_boolean_value()
            comment_items.append((item_id, True, item_name))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", comment_items)
        return comment_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class IdentifiersBuilderTableWidget(QTableWidget):

    def __init__(self, parent, identifier_items):
        QTableWidget.__init__(self, parent)
        self.identifier_items = identifier_items
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        #self.setMinimumSize(50, 0)

    def populate_table(self, identifier_items):
        self.identifier_items = identifier_items
        self.clear()
        self.setRowCount(len(identifier_items))
        header_labels = [_('Metadata Item'), _('Identifier format')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(True)

        for row, item in enumerate(identifier_items):
            self.populate_table_row(row, item)

        self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectRow(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item):
        name_widget = ReadOnlyCheckableTableWidgetItem(item[2], checked=item[1])
        identifier_format = ReadOnlyTableWidgetItem(item[3])
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(row, 0, name_widget)
        self.setItem(row, 1, identifier_format)

    def get_data(self):
        print("DownloadSourcesTableWidget::get_item_list - start")
        identifier_items = []
        for row in range(self.rowCount()):
            try:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(row, 0).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(row, 0).text())))
            active = self.item(row, 0).get_boolean_value()
            identifier_format = as_unicode(str((self.item(row, 1).text())))
            identifier_items.append((item_id, active, item_name, identifier_format))
        print("DownloadSourcesTableWidget::get_item_list - download_sources:", identifier_items)
        return identifier_items

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(self.columnCount()):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

class LineBuilderTableWidget(QTableWidget):

    def __init__(self, parent, identifier_items, header_labels=[_('Metadata Item')]):
        QTableWidget.__init__(self, parent)
        self.identifier_items = identifier_items
        self.setSortingEnabled(False)
        self.setSelectionBehavior(QAbstractItemView.SelectColumns)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setMinimumSize(0, 50)
        #self.setMaximumHeight(50)
        self.header_labels = header_labels

    def populate_table(self, identifier_items):
        self.identifier_items = identifier_items
        self.clear()
        self.setColumnCount(len(identifier_items))
        self.setRowCount(len(self.header_labels))
        self.setVerticalHeaderLabels(self.header_labels)
        self.horizontalHeader().setDefaultSectionSize(32)
        self.verticalHeader().setStretchLastSection(True)

        for col, item in enumerate(identifier_items):
            self.populate_table_col(col, item)

        #self.setMinimumColumnWidth(0, 150)
        self.resizeColumnsToContents()
        self.selectColumn(0)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_col(self, col, item):
        name_widget = ReadOnlyTableWidgetItem(item[2])
        if item[0] != g.CUSTOM_TEXT:
            try:
                font = name_widget.font()
                font.setBold(True)
                name_widget.setFont(font)
            except:
                font = name_widget.font()
                font.setBold(True)
                name_widget.setFont(font)
        try:
            name_widget.setData(Qt.ItemDataRole.UserRole, item[0])
        except:
            name_widget.setData(Qt.UserRole, item[0])
        self.setItem(0, col, name_widget)
        self.resizeColumnsToContents()

    def get_data(self):
        identifier_items = []
        for col in range(self.columnCount()):
            try:
                item_id = convert_qvariant(self.item(0, col).data(Qt.ItemDataRole.UserRole))
            except:
                item_id = convert_qvariant(self.item(0, col).data(Qt.UserRole))
            item_name = as_unicode(str((self.item(0, col).text())))
            identifier_items.append((item_id, True, item_name))
        return identifier_items
    
    def get_string(self):
        return ''.join([tupl[2] for tupl in self.get_data()])

    def swap_row_widgets(self, src_col, dest_col):
        self.blockSignals(True)
        self.insertColumn(dest_col)
        for row in range(self.rowCount()):
            self.setItem(row, dest_col, self.takeItem(row, src_col))
        self.removeColumn(src_col)
        self.blockSignals(False)
        self.resizeColumnsToContents()

    def select_and_scroll_to_col(self, col):
        self.selectRow(col)
        self.scrollToItem(self.currentItem())

class MappingsTableWidget(QTableWidget):
    def __init__(self, parent, all_tags, headers=(_('genre'), _('Tag(s)'))):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tags_values = all_tags
        self.headers = headers

    def populate_table(self, tag_mappings):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(tag_mappings))
        header_labels = ['xTrance ' + self.headers[0], _('Maps to Calibre') + ' ' + self.headers[1]]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)
        self.horizontalHeader().setStretchLastSection(True)

        for row, genre in enumerate(sorted(tag_mappings.keys(), key=lambda s: (s.lower(), s))):
            self.populate_table_row(row, genre, sorted(tag_mappings[genre]))

        self.set_minimum_column_width(0, 300)
        self.resizeColumnToContents(0)
        self.setSortingEnabled(False)
        if len(tag_mappings) > 0:
            self.selectRow(0)

    def set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, genre, tags):
        self.setItem(row, 0, ReadOnlyTableWidgetItem(genre))
        tags_value = ', '.join(tags)
        # Add a widget under the cell just for sorting purposes
        self.setItem(row, 1, QTableWidgetItem(tags_value))
        self.setCellWidget(row, 1, self.create_tags_edit(tags_value, row))

    def create_tags_edit(self, value, row):
        try:
            from calibre.gui2.complete2 import EditWithComplete
            tags_edit = EditWithComplete(self)
            tags_edit.set_add_separator(False)
            tags_edit.update_items_cache(self.tags_values)
            tags_edit.setText(value)
        except ImportError:
            from calibre.gui2.complete import MultiCompleteComboBox
            tags_edit = MultiCompleteComboBox(self)
            tags_edit.set_add_separator(False)
            tags_edit.update_items_cache(self.tags_values)
            tags_edit.setText(value)

        #tags_edit.editingFinished.connect(partial(self.tags_editing_finished, row, tags_edit))
        return tags_edit

    def tags_editing_finished(self, row, tags_edit):
        # Update our underlying widget for sorting
        self.item(row, 1).setText(tags_edit.text())

    def get_data(self):
        tag_mappings = {}
        for row in range(self.rowCount()):
            genre = as_unicode(self.item(row, 0).text()).strip()
            tags_text = as_unicode(self.cellWidget(row, 1).text()).strip()
            tag_values = tags_text.split(',')
            tags_list = []
            for tag in tag_values:
                if len(tag.strip()) > 0:
                    tags_list.append(tag.strip())
            tag_mappings[genre] = tags_list
        return tag_mappings

    def select_genre(self, genre_name):
        for row in range(self.rowCount()):
            if as_unicode(self.item(row, 0).text()) == genre_name:
                self.setCurrentCell(row, 1)
                return

    def get_selected_genre(self):
        if self.currentRow() >= 0:
            return as_unicode(self.item(self.currentRow(), 0).text())

class ReadOnlyTableWidgetItem(QTableWidgetItem):

    def __init__(self, text):
        if text is None:
            text = ''
        try:
            QTableWidgetItem.__init__(self, text, QTableWidgetItem.ItemType.UserType)
            self.setFlags(Qt.ItemFlag.ItemIsSelectable|Qt.ItemFlag.ItemIsEnabled)
        except (AttributeError, NameError) as e:
            QTableWidgetItem.__init__(self, text, QTableWidgetItem.UserType)
            self.setFlags(Qt.ItemIsSelectable|Qt.ItemIsEnabled)

class ReadOnlyCheckableTableWidgetItem(ReadOnlyTableWidgetItem):

    def __init__(self, text, checked=False, is_tristate=False):
        super(ReadOnlyCheckableTableWidgetItem, self).__init__(text)
        try:
            self.setFlags(Qt.ItemFlag(Qt.ItemFlag.ItemIsSelectable | Qt.ItemFlag.ItemIsUserCheckable | Qt.ItemFlag.ItemIsEnabled ))
        except:
            self.setFlags(Qt.ItemFlags(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled ))
        if is_tristate:
            try:
                self.setFlags(self.flags() | Qt.ItemFlag.ItemIsAutoTristate)
            except:
                self.setFlags(self.flags() | Qt.ItemIsTristate)
        if checked:
            try:
                self.setCheckState(Qt.CheckState.Checked)
            except:
                self.setCheckState(Qt.Checked)
        else:
            if is_tristate and checked is None:
                try:
                    self.setCheckState(Qt.CheckState.PartiallyChecked)
                except:
                    self.setCheckState(Qt.PartiallyChecked)
            else:
                try:
                    self.setCheckState(Qt.CheckState.Unchecked)
                except:
                    self.setCheckState(Qt.Unchecked)

    def get_boolean_value(self):
        '''
        Return a boolean value indicating whether checkbox is checked
        If this is a tristate checkbox, a partially checked value is returned as None
        '''
        try:
            if self.checkState() == Qt.CheckState.PartiallyChecked:
                return None
            else:
                return self.checkState() == Qt.CheckState.Checked
        except:
            if self.checkState() == Qt.PartiallyChecked:
                return None
            else:
                return self.checkState() == Qt.Checked
