# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2024, seeder'
__docformat__ = 'restructuredtext en'

from functools import total_ordering
from polyglot.builtins import cmp
import calibre_plugins.xTrance.globals as g
from calibre_plugins.xTrance.globals import get_pref, get_prefs, plugin_prefs

@total_ordering
class MetadataCompareKeyGen:
    def __init__(self, mi, source_plugin, title, authors, identifiers):
        if not mi:
            self.base = (2,2,2,2,2,2,2,2,2,2)
            self.comments_len = 0
            self.extra = 0
            return

        isbn = 1 if mi.isbn and identifiers.get('isbn', None) is not None \
                and mi.isbn == identifiers.get('isbn', None) else 2

        all_fields = 1 if source_plugin.test_fields(mi) is None else 2

        cl_title = title
        cl_title_mi = mi.title
        exact_title = 1 if title and \
                cl_title.strip() == cl_title_mi.strip() else 2

        contains_title = 1 if title and \
                cl_title.strip() in cl_title_mi.strip() else 2

        auths = []
        if authors:
            for a in authors:
                auths.append(a.split(" ")[-1])
        miauths = []
        for a in mi.authors:
            miauths.append(a.split(" ")[-1])

        author_segments = list(set(miauths) & set(auths)) #authors surname list compare

        has_cover = 2 if (not source_plugin.cached_cover_url_is_reliable or
                source_plugin.get_cached_cover_url(mi.identifiers) is None) else 1
        
        author_match_relevance = getattr(mi, 'author_match_relevance', 2)
        title_relevance = getattr(mi, 'title_relevance', 2)

        # compare wanted year from Identifiers in title (pubyear:2000 or pubdate:2000)
        wanted_year = source_plugin.meta_dict.get('pubdate', None)
        wanted_lang = source_plugin.meta_dict.get('language', None)
        issue_pref = g.plugin_prefs[g.STORE_NAME].get(g.ISSUE_PREFERENCE, g.DEFAULT_STORE_VALUES[g.ISSUE_PREFERENCE])
        if not wanted_year and issue_pref in (1, 3):
            wanted_year = 10_000 #MAX_VAL
        elif not wanted_year and issue_pref in (2, 4):
            wanted_year = 0 #MIN_VAL
        if not wanted_lang and issue_pref in (1, 2):
            wanted_lang = 'cs'
        elif not wanted_lang and issue_pref in (3, 4):
            wanted_lang = 'sk'
        pubyear = getattr(mi, 'pubyear', None)
        language = getattr(mi, 'language', None)
        if wanted_year and pubyear:
            closest_year = abs(int(wanted_year) - int(pubyear))
        else:
            closest_year = 0
        closest_lang = 0 if language == wanted_lang else 2
        

        self.base = (author_match_relevance, exact_title, title_relevance, isbn, contains_title, -len(author_segments), closest_lang, closest_year, all_fields, has_cover)

        #self.comments_len = len(mi.comments.strip() if mi.comments else '')
        self.comments_len = 0
        self.extra = (getattr(mi, 'source_relevance', 0), )

    def compare_to_other(self, other):
        a = cmp(self.base, other.base)
        if a != 0:
            return a
        cx, cy = self.comments_len, other.comments_len
        if cx and cy:
            t = (cx + cy) / 20
            delta = cy - cx
            if abs(delta) > t:
                return -1 if delta < 0 else 1
        return cmp(self.extra, other.extra)

    def __eq__(self, other):
        return self.compare_to_other(other) == 0

    def __ne__(self, other):
        return self.compare_to_other(other) != 0

    def __lt__(self, other):
        return self.compare_to_other(other) < 0

    def __le__(self, other):
        return self.compare_to_other(other) <= 0

    def __gt__(self, other):
        return self.compare_to_other(other) > 0

    def __ge__(self, other):
        return self.compare_to_other(other) >= 0
