#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2024 seeder'
__docformat__ = 'restructuredtext en'

try:
    from qt.core import  Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidget, \
                        QIcon, QInputDialog, QTabWidget, QWidget
except ImportError:
    try:
        from PyQt5.Qt import Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidget, \
                        QIcon, QInputDialog, QTabWidget, QWidget
    except ImportError:
        from PyQt4.Qt import Qt, QToolButton, QSpinBox, QLabel, QGroupBox, QCheckBox, QComboBox, QHBoxLayout, QVBoxLayout, QLineEdit, QTableWidget, \
                        QIcon, QInputDialog, QTabWidget, QWidget

from calibre import as_unicode
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget

import calibre_plugins.xTrance.globals as g
from calibre_plugins.xTrance.globals import get_pref, get_prefs, plugin_prefs
from calibre_plugins.xTrance.config_components import TagBuilderTableWidget, CommentBuilderTableWidget, \
                                    IdentifiersBuilderTableWidget, LineBuilderTableWidget, MappingsTableWidget

try:
    load_translations()
except NameError:
    pass # load_translations() added in calibre 1.9

def add_spin_option(parent, name, help_desc, pref, min_val=1, max_val=100):
    spinbox_layout = QHBoxLayout()
    parent.addLayout(spinbox_layout)
    
    label = QLabel(name)
    label.setToolTip(help_desc)
    spinbox_layout.addWidget(label)
    
    spinbox = QSpinBox()
    spinbox.setMinimum(min_val)
    spinbox.setMaximum(max_val)
    spinbox.setMaximumWidth(60)
    spinbox.setProperty('value', get_pref(pref))
    spinbox.setProperty('option_name', pref)
    spinbox_layout.addWidget(spinbox)
    
    spinbox_layout.addStretch(1)
    return spinbox

def add_combobox_option(parent, name, help_desc, pref, choices):
    combobox_layout = QHBoxLayout()
    parent.addLayout(combobox_layout)
    
    label = QLabel(name)
    label.setToolTip(help_desc)
    combobox_layout.addWidget(label)

    combobox = QComboBox()
    combobox.setProperty('option_name', pref)
    [combobox.addItem(c) for c in choices]
    combobox.setCurrentIndex(get_pref(pref))
    combobox.setMaximumWidth(180)
    combobox_layout.addWidget(combobox)

    return combobox

def add_check_option(parent, name, help_desc=None, pref=None):
    checkbox = QCheckBox(name)
    if help_desc:
        checkbox.setToolTip(help_desc)
    if pref is not None:
        checkbox.setProperty('option_name', pref)
        checkbox.setChecked(get_pref(pref))
    else:
        checkbox.setChecked(True)
        checkbox.setEnabled(False)
    parent.addWidget(checkbox)
    
    return checkbox

def get_widget_value(widget):
    if isinstance(widget, QComboBox):
        return int(widget.currentIndex())
    elif isinstance(widget, QCheckBox):
        return widget.isChecked()
    elif isinstance(widget, QSpinBox):
        return int(as_unicode(widget.value()))
    elif isinstance(widget, QLineEdit):
        return str(widget.text())
    elif isinstance(widget, QTableWidget):
        return widget.get_data()
    
def set_widget_value(widget, value):
    if isinstance(widget, QComboBox):
        widget.setCurrentIndex(value)
    elif isinstance(widget, QCheckBox):
        widget.setChecked(value)
    elif isinstance(widget, QSpinBox):
        widget.setValue(int(value))
    elif isinstance(widget, QLineEdit):
        widget.setText(str(value))
    elif isinstance(widget, QTableWidget):
        widget.populate_table(value)
        
class ConfigWidget(DefaultConfigWidget):

    def set_default_prefs(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to reset plugin to default settings?'),
                show_copy_button=False):
            return
        plugin_prefs[g.STORE_NAME] = g.DEFAULT_STORE_VALUES
        set_widget_value(self.search_tab.max_downloads_spin, plugin_prefs[g.STORE_NAME][g.MAX_DOWNLOADS])
        set_widget_value(self.search_tab.max_covers_spin, plugin_prefs[g.STORE_NAME][g.MAX_COVERS])
        set_widget_value(self.search_tab.title_whole_search_check, plugin_prefs[g.STORE_NAME][g.SEARCH_WHOLE_TITLE])
        set_widget_value(self.tag_tab.tags_filter_check, plugin_prefs[g.STORE_NAME][g.TAGS_FILTER])
        set_widget_value(self.series_tab.series_filter_check, plugin_prefs[g.STORE_NAME][g.SERIES_FILTER])
        set_widget_value(self.publisher_tab.publishers_filter_check, plugin_prefs[g.STORE_NAME][g.PUBLISHER_FILTER])
        set_widget_value(self.tag_tab.table_widget, plugin_prefs[g.STORE_NAME][g.TAGS_MAPPINGS])
        set_widget_value(self.series_tab.table_widget, plugin_prefs[g.STORE_NAME][g.SERIES_MAPPINGS])
        set_widget_value(self.publisher_tab.table_widget, plugin_prefs[g.STORE_NAME][g.PUBLISHER_MAPPINGS])
        set_widget_value(self.authors_tab.one_author_check, plugin_prefs[g.STORE_NAME][g.ONE_AUTHOR])
        set_widget_value(self.search_tab.issue_pref_combo, plugin_prefs[g.STORE_NAME][g.ISSUE_PREFERENCE])
        set_widget_value(self.authors_tab.authors_field_check, plugin_prefs[g.STORE_NAME][g.AUTHORS_INCLUDE])
        set_widget_value(self.authors_tab.translators_field_check, plugin_prefs[g.STORE_NAME][g.TRANSLATORS_INCLUDE])
        set_widget_value(self.authors_tab.illustrators_field_check, plugin_prefs[g.STORE_NAME][g.ILLUSTRATORS_INCLUDE])
        set_widget_value(self.comments_tab.comments_arrange_table, plugin_prefs[g.STORE_NAME][g.APPEND_TO_COMMENTS])
        self.tag_tab.tag_items_table.populate_table(self.tag_tab.DEFAULT_TAGS_ITEMS)
        self.identifiers_tab.identifier_items_table.populate_table(self.identifiers_tab.DEFAULT_IDENTIFIERS_ITEMS)
        self.authors_tab.title_line_table.populate_table(self.authors_tab.DEFAULT_TITLE_ITEMS)
        self.publisher_tab.publisher_line_table.populate_table(self.publisher_tab.DEFAULT_PUBLISHER_ITEMS)
        self.series_tab.series_line_table.populate_table(self.series_tab.DEFAULT_SERIES_ITEMS)
        set_widget_value(self.series_tab.series_index_field, plugin_prefs[g.STORE_NAME][g.SERIES_INDEXING_ITEM])
        set_widget_value(self.publisher_tab.publication_date, plugin_prefs[g.STORE_NAME][g.PUBLICATION_DATE])
        set_widget_value(self.authors_tab.swap_authors_check, plugin_prefs[g.STORE_NAME][g.SWAP_AUTHORS])
        set_widget_value(self.authors_tab.authors_role_check, plugin_prefs[g.STORE_NAME][g.AUTHOR_ROLE])
        set_widget_value(self.identifiers_tab.lang_check, plugin_prefs[g.STORE_NAME][g.LANGUAGE_INCLUDE])
        

    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = get_prefs(fill_defaults=True)

        top_box = QHBoxLayout(self)
        if plugin.config_message:
            config_message = QLabel(plugin.config_message)
            config_message.setWordWrap(True)
            config_message.setOpenExternalLinks(True)
            top_box.addWidget(config_message)

        reset_plugin_btn = QToolButton()
        reset_plugin_btn.setToolTip(_('Reset plugin to defaults (Alt+Shift+R)'))
        reset_plugin_btn.setIcon(QIcon(I('restart.png')))
        reset_plugin_btn.clicked.connect(self.set_default_prefs)
        reset_plugin_btn.setShortcut(_('Alt+Shift+R'))
        reset_plugin_btn.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
        reset_plugin_btn.setText(_('Restart plugin to defaults'))
        top_box.addWidget(reset_plugin_btn)

        try:
            self.overl.insertLayout(0, top_box)
        except AttributeError:
            self.l.addLayout(top_box, 0, 0, 1, 1)
            
        self.gb.setMaximumHeight(70)

        tab = QTabWidget(self)
        self.l.addWidget(tab, self.l.rowCount(), 0, 1, 1)

        self.search_tab = SearchTab()
        self.authors_tab = AuthorsTab()
        self.tag_tab = TagTab()
        self.series_tab = SeriesTab()
        self.publisher_tab = PublisherTab()
        self.comments_tab = CommentsTab()
        self.identifiers_tab = IdentifiersTab()

        tab.addTab(self.search_tab, _('Search'))
        tab.addTab(self.authors_tab, _('Title and Authors'))
        tab.addTab(self.tag_tab, _('Tags'))
        tab.addTab(self.series_tab, _('Series'))
        tab.addTab(self.publisher_tab, _('Publishers and Date'))
        tab.addTab(self.comments_tab, _('Comments'))
        tab.addTab(self.identifiers_tab, _('Identifiers and Language'))

    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        new_prefs[g.MAX_DOWNLOADS] = get_widget_value(self.search_tab.max_downloads_spin)
        new_prefs[g.MAX_COVERS] = get_widget_value(self.search_tab.max_covers_spin)
        new_prefs[g.SEARCH_WHOLE_TITLE] = get_widget_value(self.search_tab.title_whole_search_check)
        new_prefs[g.TAGS_FILTER] = get_widget_value(self.tag_tab.tags_filter_check)
        new_prefs[g.SERIES_FILTER] = get_widget_value(self.series_tab.series_filter_check)
        new_prefs[g.PUBLISHER_FILTER] = get_widget_value(self.publisher_tab.publishers_filter_check)
        new_prefs[g.TAGS_MAPPINGS] = get_widget_value(self.tag_tab.table_widget)
        new_prefs[g.SERIES_MAPPINGS] = get_widget_value(self.series_tab.table_widget)
        new_prefs[g.PUBLISHER_MAPPINGS] = get_widget_value(self.publisher_tab.table_widget)
        new_prefs[g.ONE_AUTHOR] = get_widget_value(self.authors_tab.one_author_check)
        new_prefs[g.ISSUE_PREFERENCE] = get_widget_value(self.search_tab.issue_pref_combo)
        new_prefs[g.AUTHORS_INCLUDE] = get_widget_value(self.authors_tab.authors_field_check)
        new_prefs[g.TRANSLATORS_INCLUDE] = get_widget_value(self.authors_tab.translators_field_check)
        new_prefs[g.ILLUSTRATORS_INCLUDE] = get_widget_value(self.authors_tab.illustrators_field_check)

        new_prefs[g.APPEND_TO_COMMENTS] = get_widget_value(self.comments_tab.comments_arrange_table)
        new_prefs[g.APPEND_TO_TAGS] = get_widget_value(self.tag_tab.tag_items_table)
        new_prefs[g.APPEND_TO_IDENTIFIERS] = get_widget_value(self.identifiers_tab.identifier_items_table)
        new_prefs[g.TITLE_LINE] = get_widget_value(self.authors_tab.title_line_table)
        new_prefs[g.PUBLISHER_LINE] = get_widget_value(self.publisher_tab.publisher_line_table)
        new_prefs[g.SERIES_LINE] = get_widget_value(self.series_tab.series_line_table)
        new_prefs[g.SERIES_INDEXING_ITEM], new_prefs[g.SERIES_INDEX_FIELD] =  [(i, item) for i, item in enumerate(self.series_tab.index_item_options) if i == get_widget_value(self.series_tab.series_index_field)][0]
        new_prefs[g.PUBLICATION_DATE] = get_widget_value(self.publisher_tab.publication_date)
        new_prefs[g.SWAP_AUTHORS] = get_widget_value(self.authors_tab.swap_authors_check)
        new_prefs[g.AUTHOR_ROLE] = get_widget_value(self.authors_tab.authors_role_check)
        new_prefs[g.LANGUAGE_INCLUDE] = get_widget_value(self.identifiers_tab.lang_check)
        plugin_prefs[g.STORE_NAME] = new_prefs

class SearchTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_prefs(fill_defaults=True)

        #Search option groupbox
        #other_group_box = QGroupBox(_('Search'), self)
        other_group_box_layout = QVBoxLayout()
        self.setLayout(other_group_box_layout)      
        
        self.max_downloads_spin = add_spin_option(other_group_box_layout, _('Maximum title/author search matches to evaluate (1 = fastest):'), 
                                                  _('Number of maximum search results which you can get from DK.'),
                                                  g.MAX_DOWNLOADS)

        search_group_box = QGroupBox(_('Searching priority'), self)
        search_group_box_layout = QHBoxLayout()
        search_group_box.setLayout(search_group_box_layout)


        title_identifiers_search_check = add_check_option(search_group_box_layout, 
                                                        _('Title ids'),
                                                        _("You can add identifiers into 'Title' field (in format identifier:value e.g. pubdate:2000 or publisher:new_publisher)\nbefore you start metadata download and they will be used prior built-in one."\
                                                          "Identifiers can be combined in title\n\n"\
                                                          "Identifiers available:\n"\
                                                          "'publisher' for book issue publisher specification (use underscore '_' for spacing multiple words)\n"\
                                                          "'language' and 'lang' for book issue language specification\n"\
                                                          "'pubdate' and 'pubyear' published edition year specification\n(e.g. pubdate:2024 will force plugin to prior search for issues closest to year 2024)"))
        
        search_group_box_layout.addWidget(QLabel('>', self))

        title_beginning_search_check = add_check_option(search_group_box_layout,
                                                        _('Title beginning'),
                                                        _('At first plugin tries to find records comparing Title from beginning')
                                                        )
        search_group_box_layout.addWidget(QLabel('>', self))
        self.title_whole_search_check = add_check_option(search_group_box_layout,
                                                        _('Whole Title'),
                                                        _('Then plugin tries to find all matches which includes Title query'),
                                                        g.SEARCH_WHOLE_TITLE)
        search_group_box_layout.addStretch(1)
        other_group_box_layout.addWidget(search_group_box)

        # Issue choice
        book_issues_group_box = QGroupBox(_('Book issues'), self)
        book_issues_group_box_layout = QHBoxLayout()
        book_issues_group_box.setLayout(book_issues_group_box_layout)
        self.issue_pref_combo = add_combobox_option(book_issues_group_box_layout, _('Choose preferred book issue'), _('Set which issue of the book will be preferred. You can choose between czech/slovak and oldest/newest.\nDefault will be chosen if specified issue not found.'), 
                            g.ISSUE_PREFERENCE, choices=[_('Default'), _('Czech newest'), _('Czech oldest'), _('Slovak newest'), _('Slovak oldest')])
        
        other_group_box_layout.addWidget(book_issues_group_box)

        # Covers download
        covers_group_box = QGroupBox(_('Covers'), self)
        covers_group_box_layout = QVBoxLayout()
        covers_group_box.setLayout(covers_group_box_layout)
        self.max_covers_spin = add_spin_option(covers_group_box_layout, _('Maximum covers from other issues to evaluate (1 = fastest):'), _('Set max covers to download. Faster evaluating if set to 1 with Default preferred issue set\n\n'\
                                                                                                                                            'When set to 0 - no covers will be downloaded.\n'\
                                                                                                                                            'When set to 1 and preferred book issue option set to default - only default found issue cover will be downloaded.\n'\
                                                                                                                                            'When set to >1 all covers from merging issues are downloaded'),
                            g.MAX_COVERS, min_val=0)
        other_group_box_layout.addWidget(covers_group_box)

        other_group_box_layout.addStretch(1)

class AuthorsTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Title field
        title_builder_group_box = QGroupBox(_("Set what will be parsed into Calibre 'Title' column"))
        title_builder_group_box.setMaximumHeight(100)
        layout.addWidget(title_builder_group_box)

        c = get_prefs(fill_defaults=True)
        tablebox_layout = QVBoxLayout()
        title_builder_group_box.setLayout(tablebox_layout)
        combobox_layout = QHBoxLayout()
        
        self.DEFAULT_TITLE_ITEMS = [
            (g.TITLE, True, _('Title')),
        ]
        title_items = []
        for i, tupl in enumerate(c.get(g.TITLE_LINE, self.DEFAULT_TITLE_ITEMS)):
            title_items.append((tupl[0], tupl[1], tupl[2]))

        self.title_line_table = LineBuilderTableWidget(self, title_items, header_labels=[_('Title: ')])
        self.title_line_table.populate_table(title_items)
        self.title_line_string = QLabel(self.title_line_table.get_string())
        title_line_layout = QHBoxLayout()
        title = QLabel(_('Title: '))
        font = title.font()
        font.setBold(True)
        title.setFont(font)
        title_line_layout.addWidget(title)
        title_line_layout.addWidget(self.title_line_string, 1)
        tablebox_layout.addLayout(title_line_layout)
        tablebox_layout.addLayout(combobox_layout)
        combobox_layout.addWidget(self.title_line_table)

        button_layout = QHBoxLayout()
        combobox_layout.addLayout(button_layout)
        add_btn = QToolButton()
        add_btn.setToolTip(_('Add item (Alt+Shift+W)'))
        add_btn.setIcon(QIcon(I('plus.png')))
        add_btn.clicked.connect(self.add_title_item)
        add_btn.setShortcut(_('Alt+Shift+W'))
        button_layout.addWidget(add_btn)

        delete_btn = QToolButton()
        delete_btn.setToolTip(_('Delete item (Alt+Shift+S)'))
        delete_btn.setIcon(QIcon(I('minus.png')))
        delete_btn.clicked.connect(self.delete_title_item)
        delete_btn.setShortcut(_('Alt+Shift+S'))
        button_layout.addWidget(delete_btn)

        move_left_btn = QToolButton()
        move_left_btn.setToolTip(_('Move left (Alt+Shift+A)'))
        move_left_btn.setIcon(QIcon(I('back.png')))
        move_left_btn.clicked.connect(self.move_cols_left)
        move_left_btn.setShortcut(_('Alt+Shift+A'))
        button_layout.addWidget(move_left_btn)

        move_right_btn = QToolButton()
        move_right_btn.setToolTip(_('Move right (Alt+Shift+D)'))
        move_right_btn.setIcon(QIcon(I('forward.png')))
        move_right_btn.clicked.connect(self.move_cols_right)
        move_right_btn.setShortcut(_('Alt+Shift+D'))
        button_layout.addWidget(move_right_btn)

        self.item_options = [
        (g.CUSTOM_TEXT, True, _('Custom text')),
        (g.TITLE, True, _('Title')),
        (g.SUBTITLE, True, _('Subtitle')),
        (g.ORIGINAL_TITLE, True, _('Original title')),
        (g.PUBLISHER, True, _('Publisher')),
        (g.ORIGINAL_PUBLISHER, True, _('Original publisher')),
        (g.AUTHORS, True, _('Authors')),
        (g.TRANSLATORS, True, _('Translators')),
        (g.ILLUSTRATORS, True, _('Illustrators')),
        (g.BOOK_DIMENSIONS, True, _('Book dimensions')),
        (g.ISSUE_NUMBER, True, _('Issue number')),
        (g.IMG_NUMBER, True, _('Images count')),
        (g.PAGES, True, _('Pages count')),
        (g.PUB_YEAR, True, _('Publication date')),
        (g.ORIGINAL_YEAR, True, _('Original pubdate')),
        (g.PRINT_RUN, True, _('Print run')),
        (g.SERIES, True, _('Series')),
        (g.SERIES_INDEX, True, _('Series index')),
        (g.EDITION, True, _('Book Edition')),
        (g.EDITION_INDEX, True, _('Book Edition index')),
        (g.TAGS, True, _('Tags')),
        (g.RATING, True, _('Rating (%)')),
        (g.RATING10, True, _('Rating (0-10)')),
        (g.RATING5, True, _('Rating (0-5)')),
        (g.ISBN, True, _('ISBN')),
        (g.XTRANCE, True, _('Identifier xtrance')),
        (g.LANGUAGE, True, _('Language')),
        (g.ORIGINAL_LANGUAGE, True, _('Original language')),
        (g.SOURCE_RELEVANCE, True, _('Source relevance'))
        ]

        authors_field_group_box = QGroupBox(_('Select what will be added into "Authors" metadata field in Calibre'), self)
        layout.addWidget(authors_field_group_box)
        
        group_box_layout = QHBoxLayout()
        authors_field_group_box.setLayout(group_box_layout)


        self.authors_field_check = add_check_option(group_box_layout,
                                                    _('Authors'),
                                                    _('Names of authors will be added into calibre authors metadata field.'),
                                                    g.AUTHORS_INCLUDE)

        self.translators_field_check = add_check_option(group_box_layout,
                                                    _('Translators'),
                                                    _('Names of translators will be added into calibre authors metadata field.'),
                                                    g.TRANSLATORS_INCLUDE)

        self.illustrators_field_check = add_check_option(group_box_layout,
                                                    _('Illustrators'),
                                                    _('Names of illustrators will be added into calibre authors metadata field.'),
                                                    g.ILLUSTRATORS_INCLUDE)

        self.authors_role_check = add_check_option(layout, _('Save authors with role'),
                                                   _('Writes role (author, translator, illustrator...) after every author in Authors field.\n'
                                                     '**WONT add role when only Author grabbing is checked'),
                                                   g.AUTHOR_ROLE)

        self.swap_authors_check = add_check_option(layout, _('Swap authors names with last names'),
                                                   _('Enable this option if you want to get names in Authors metadata field in format "Surname Name"\n **Global LN, FN Calibre option is prefered'),
                                                   g.SWAP_AUTHORS)
        
        self.one_author_check = add_check_option(layout, _('Get only one author'),
                                                  _('If the information about the book contains more authors\n - plugin will add only first one.'), 
                                                  g.ONE_AUTHOR)

        layout.addStretch(1)

    def move_cols_left(self):
        self.title_line_table.setFocus()
        cols = self.title_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        first_sel_col = cols[0].column()
        if first_sel_col <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in selcols:
            self.title_line_table.swap_row_widgets(selcol - 1, selcol + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_col - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.title_line_table.scrollToItem(self.title_line_table.item(scroll_to_row, 0))
        self.title_line_string.setText(self.title_line_table.get_string())
        #self.renumber_series()

    def move_cols_right(self):
        self.title_line_table.setFocus()
        cols = self.title_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        last_sel_col = cols[-1].column()
        if last_sel_col == self.title_line_table.columnCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in reversed(selcols):
            self.title_line_table.swap_row_widgets(selcol + 2, selcol)

        scroll_to_row = last_sel_col + 1
        if scroll_to_row < self.title_line_table.columnCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.title_line_table.scrollToItem(self.title_line_table.item(scroll_to_row, 0))
        self.title_line_string.setText(self.title_line_table.get_string())
        # self.renumber_series()

    def add_title_item(self):
        self.title_line_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to title"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.title_line_table.get_data()
            new_item = [i for i in self.item_options if item in i][0]
            if new_item[0] == g.CUSTOM_TEXT:
                text, ok_text = QInputDialog().getText(self, _("Add custom text"),
                                            _("Write text:"))
                if not ok_text or not text:
                    return
                else:
                    new_item = (g.CUSTOM_TEXT, True, text)
            new_items.append(new_item)
            self.title_line_table.populate_table(new_items)
            self.title_line_string.setText(self.title_line_table.get_string())
            self.title_line_table.selectColumn(self.title_line_table.columnCount() - 1)
        if not item:
            return

    def delete_title_item(self):
        from calibre.gui2 import question_dialog
        self.title_line_table.setFocus()
        if not self.title_line_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for col in reversed(sorted(self.title_line_table.selectionModel().selectedColumns())):
            self.title_line_table.removeColumn(col.column())
        self.title_line_string.setText(self.title_line_table.get_string())

    def reset_title_items(self):
        from calibre.gui2 import question_dialog
        self.title_line_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set title builder to defaults?'),
                show_copy_button=False):
            return
        self.title_line_table.setColumnCount(0)
        self.title_line_table.populate_table(g.DEFAULT_STORE_VALUES[g.TITLE_LINE])
        self.title_line_string.setText(self.title_line_table.get_string())

class SeriesTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Series field
        series_builder_group_box = QGroupBox(_("Set what will be parsed into Calibre 'Series' column"))
        series_builder_group_box.setMaximumHeight(100)
        layout.addWidget(series_builder_group_box)

        c = get_prefs(fill_defaults=True)
        tablebox_layout = QVBoxLayout()
        series_builder_group_box.setLayout(tablebox_layout)
        combobox_layout = QHBoxLayout()
        
        self.DEFAULT_SERIES_ITEMS = [
            (g.SERIES, True, _('Series')),
        ]
        series_items = []
        for i, tupl in enumerate(c.get(g.SERIES_LINE, self.DEFAULT_SERIES_ITEMS)):
            series_items.append((tupl[0], tupl[1], tupl[2]))

        self.series_line_table = LineBuilderTableWidget(self, series_items, header_labels=[_('Series: ')])
        self.series_line_table.populate_table(series_items)
        self.series_line_string = QLabel(self.series_line_table.get_string())
        series_line_layout = QHBoxLayout()
        title = QLabel(_('Series: '))
        font = title.font()
        font.setBold(True)
        title.setFont(font)
        series_line_layout.addWidget(title)
        series_line_layout.addWidget(self.series_line_string, 1)
        tablebox_layout.addLayout(series_line_layout)
        tablebox_layout.addLayout(combobox_layout)
        combobox_layout.addWidget(self.series_line_table)

        button_layout = QHBoxLayout()
        combobox_layout.addLayout(button_layout)
        add_btn = QToolButton()
        add_btn.setToolTip(_('Add item (Alt+Shift+W)'))
        add_btn.setIcon(QIcon(I('plus.png')))
        add_btn.clicked.connect(self.add_series_item)
        add_btn.setShortcut(_('Alt+Shift+W'))
        button_layout.addWidget(add_btn)

        delete_btn = QToolButton()
        delete_btn.setToolTip(_('Delete item (Alt+Shift+S)'))
        delete_btn.setIcon(QIcon(I('minus.png')))
        delete_btn.clicked.connect(self.delete_series_item)
        delete_btn.setShortcut(_('Alt+Shift+S'))
        button_layout.addWidget(delete_btn)

        move_left_btn = QToolButton()
        move_left_btn.setToolTip(_('Move left (Alt+Shift+A)'))
        move_left_btn.setIcon(QIcon(I('back.png')))
        move_left_btn.clicked.connect(self.move_cols_left)
        move_left_btn.setShortcut(_('Alt+Shift+A'))
        button_layout.addWidget(move_left_btn)

        move_right_btn = QToolButton()
        move_right_btn.setToolTip(_('Move right (Alt+Shift+D)'))
        move_right_btn.setIcon(QIcon(I('forward.png')))
        move_right_btn.clicked.connect(self.move_cols_right)
        move_right_btn.setShortcut(_('Alt+Shift+D'))
        button_layout.addWidget(move_right_btn)

        self.item_options = [
        (g.CUSTOM_TEXT, True, _('Custom text')),
        (g.SERIES, True, _('Series')),
        (g.SERIES_INDEX, True, _('Series index')),
        (g.EDITION, True, _('Book Edition')),
        (g.EDITION_INDEX, True, _('Book Edition index')),
        (g.PUBLISHER, True, _('Publisher')),
        (g.ORIGINAL_PUBLISHER, True, _('Original publisher')),
        (g.ORIGINAL_TITLE, True, _('Original title')),
        (g.TITLE, True, _('Title')),
        (g.SUBTITLE, True, _('Subtitle')),
        (g.AUTHORS, True, _('Authors')),
        (g.TRANSLATORS, True, _('Translators')),
        (g.ILLUSTRATORS, True, _('Illustrators')),
        (g.BOOK_DIMENSIONS, True, _('Book dimensions')),
        (g.ISSUE_NUMBER, True, _('Issue number')),
        (g.IMG_NUMBER, True, _('Images count')),
        (g.PUB_YEAR, True, _('Publication date')),
        (g.ORIGINAL_YEAR, True, _('Original pubdate')),
        (g.PAGES, True, _('Pages count')),
        (g.PRINT_RUN, True, _('Print run')),
        (g.TAGS, True, _('Tags')),
        (g.RATING, True, _('Rating (%)')),
        (g.RATING10, True, _('Rating (0-10)')),
        (g.RATING5, True, _('Rating (0-5)')),
        (g.ISBN, True, _('ISBN')),
        (g.XTRANCE, True, _('Identifier xtrance')),
        (g.LANGUAGE, True, _('Language')),
        (g.ORIGINAL_LANGUAGE, True, _('Original language')),
        (g.SOURCE_RELEVANCE, True, _('Source relevance'))
        ]

        #Items in Series index metadata field
        self.index_item_options = [
        (g.SERIES_INDEX, True, _('Series index')),
        (g.EDITION_INDEX, True, _('Book Edition index')),
        (g.ISSUE_NUMBER, True, _('Issue number')),
        (g.IMG_NUMBER, True, _('Images count')),
        (g.PUB_YEAR, True, _('Publication date')),
        (g.ORIGINAL_YEAR, True, _('Original pubdate')),
        (g.PAGES, True, _('Pages count')),
        (g.PRINT_RUN, True, _('Print run')),
        (g.RATING, True, _('Rating (%)')),
        (g.RATING10, True, _('Rating (0-10)')),
        (g.RATING5, True, _('Rating (0-5)')),
        (g.XTRANCE, True, _('Identifier xtrance')),
        (g.SOURCE_RELEVANCE, True, _('Source relevance'))
        ]
        self.series_index_field = add_combobox_option(layout, _("Set what will be parsed into Calibre 'series_index'"), _('You can select any number value here.'),
                            g.SERIES_INDEXING_ITEM, choices=[i[2] for i in self.index_item_options])

        # Calibre Mapping Grid
        mapping_group_box = QGroupBox(_('xTrance series and editions to calibre series and editions mappings'))
        mapping_group_box.setMaximumHeight(800)
        layout.addWidget(mapping_group_box)

        filter_layout = QHBoxLayout()
        self.series_filter_check = add_check_option(filter_layout, _('Filter series/editions via mappings'), _("Check this and only mapped series and editions below will get parsed to Calibre."), g.SERIES_FILTER)

        from calibre.gui2 import get_current_db
        all_series = list()
        for s in get_current_db().all_series():
            if s[1] not in all_series:
                all_series.append(s[1])
        self.table_widget = MappingsTableWidget(self, all_series, headers=(_('series/editions'), _('Series/editions')))
        tags_layout = RemapingTableLayout(self, self.table_widget, g.SERIES_MAPPINGS, 'series/editions', self.series_filter_check)
        mapping_group_box.setLayout(tags_layout)

    def move_cols_left(self):
        self.series_line_table.setFocus()
        cols = self.series_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        first_sel_col = cols[0].column()
        if first_sel_col <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in selcols:
            self.series_line_table.swap_row_widgets(selcol - 1, selcol + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_col - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.series_line_table.scrollToItem(self.series_line_table.item(scroll_to_row, 0))
        self.series_line_string.setText(self.series_line_table.get_string())
        #self.renumber_series()

    def move_cols_right(self):
        self.series_line_table.setFocus()
        cols = self.series_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        last_sel_col = cols[-1].column()
        if last_sel_col == self.series_line_table.columnCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in reversed(selcols):
            self.series_line_table.swap_row_widgets(selcol + 2, selcol)

        scroll_to_row = last_sel_col + 1
        if scroll_to_row < self.series_line_table.columnCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.series_line_table.scrollToItem(self.series_line_table.item(scroll_to_row, 0))
        self.series_line_string.setText(self.series_line_table.get_string())
        # self.renumber_series()

    def add_series_item(self):
        self.series_line_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to series"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.series_line_table.get_data()
            new_item = [i for i in self.item_options if item in i][0]
            if new_item[0] == g.CUSTOM_TEXT:
                text, ok_text = QInputDialog().getText(self, _("Add custom text"),
                                            _("Write text:"))
                if not ok_text or not text:
                    return
                else:
                    new_item = (g.CUSTOM_TEXT, True, text)
            new_items.append(new_item)
            self.series_line_table.populate_table(new_items)
            self.series_line_string.setText(self.series_line_table.get_string())
            self.series_line_table.selectColumn(self.series_line_table.columnCount() - 1)
        if not item:
            return

    def delete_series_item(self):
        from calibre.gui2 import question_dialog
        self.series_line_table.setFocus()
        if not self.series_line_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for col in reversed(sorted(self.series_line_table.selectionModel().selectedColumns())):
            self.series_line_table.removeColumn(col.column())
        self.series_line_string.setText(self.series_line_table.get_string())

    def reset_series_items(self):
        from calibre.gui2 import question_dialog
        self.series_line_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set series builder to defaults?'),
                show_copy_button=False):
            return
        self.series_line_table.setColumnCount(0)
        self.series_line_table.populate_table(g.DEFAULT_STORE_VALUES[g.SERIES_LINE])
        self.series_line_string.setText(self.series_line_table.get_string())

class PublisherTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        layout = QVBoxLayout()
        self.setLayout(layout)

        #Items in Published date metadata field
        self.publication_date = add_combobox_option(layout, _("Set what will be parsed into Calibre 'Published date' column"), _('You can select original pubdate or this actual publication pubdate to be parsed here.'), 
                            g.PUBLICATION_DATE, choices=[_('Actual publication date'), _('Original publication date')])

        #Publisher field
        publisher_builder_group_box = QGroupBox(_("Set what will be parsed into Calibre 'Publisher' column"))
        publisher_builder_group_box.setMaximumHeight(100)
        layout.addWidget(publisher_builder_group_box)

        c = get_prefs(fill_defaults=True)
        tablebox_layout = QVBoxLayout()
        publisher_builder_group_box.setLayout(tablebox_layout)
        combobox_layout = QHBoxLayout()
        
        self.DEFAULT_PUBLISHER_ITEMS = [
            (g.PUBLISHER, True, _('Publisher')),
        ]
        publisher_items = []
        for i, tupl in enumerate(c.get(g.PUBLISHER_LINE, self.DEFAULT_PUBLISHER_ITEMS)):
            publisher_items.append((tupl[0], tupl[1], tupl[2]))

        self.publisher_line_table = LineBuilderTableWidget(self, publisher_items, header_labels=[_('Publisher: ')])
        self.publisher_line_table.populate_table(publisher_items)
        self.publisher_line_string = QLabel(self.publisher_line_table.get_string())
        title_line_layout = QHBoxLayout()
        title = QLabel(_('Publisher: '))
        font = title.font()
        font.setBold(True)
        title.setFont(font)
        title_line_layout.addWidget(title)
        title_line_layout.addWidget(self.publisher_line_string, 1)
        tablebox_layout.addLayout(title_line_layout)
        tablebox_layout.addLayout(combobox_layout)
        combobox_layout.addWidget(self.publisher_line_table)

        button_layout = QHBoxLayout()
        combobox_layout.addLayout(button_layout)
        add_btn = QToolButton()
        add_btn.setToolTip(_('Add item (Alt+Shift+W)'))
        add_btn.setIcon(QIcon(I('plus.png')))
        add_btn.clicked.connect(self.add_publisher_item)
        add_btn.setShortcut(_('Alt+Shift+W'))
        button_layout.addWidget(add_btn)

        delete_btn = QToolButton()
        delete_btn.setToolTip(_('Delete item (Alt+Shift+S)'))
        delete_btn.setIcon(QIcon(I('minus.png')))
        delete_btn.clicked.connect(self.delete_publisher_item)
        delete_btn.setShortcut(_('Alt+Shift+S'))
        button_layout.addWidget(delete_btn)

        move_left_btn = QToolButton()
        move_left_btn.setToolTip(_('Move left (Alt+Shift+A)'))
        move_left_btn.setIcon(QIcon(I('back.png')))
        move_left_btn.clicked.connect(self.move_cols_left)
        move_left_btn.setShortcut(_('Alt+Shift+A'))
        button_layout.addWidget(move_left_btn)

        move_right_btn = QToolButton()
        move_right_btn.setToolTip(_('Move right (Alt+Shift+D)'))
        move_right_btn.setIcon(QIcon(I('forward.png')))
        move_right_btn.clicked.connect(self.move_cols_right)
        move_right_btn.setShortcut(_('Alt+Shift+D'))
        button_layout.addWidget(move_right_btn)

        self.item_options = [
        (g.CUSTOM_TEXT, True, _('Custom text')),
        (g.PUBLISHER, True, _('Publisher')),
        (g.ORIGINAL_PUBLISHER, True, _('Original publisher')),
        (g.ORIGINAL_TITLE, True, _('Original title')),
        (g.TITLE, True, _('Title')),
        (g.SUBTITLE, True, _('Subtitle')),
        (g.AUTHORS, True, _('Authors')),
        (g.TRANSLATORS, True, _('Translators')),
        (g.ILLUSTRATORS, True, _('Illustrators')),
        (g.BOOK_DIMENSIONS, True, _('Book dimensions')),
        (g.ISSUE_NUMBER, True, _('Issue number')),
        (g.IMG_NUMBER, True, _('Images count')),
        (g.PUB_YEAR, True, _('Publication date')),
        (g.ORIGINAL_YEAR, True, _('Original pubdate')),
        (g.PAGES, True, _('Pages count')),
        (g.PRINT_RUN, True, _('Print run')),
        (g.SERIES, True, _('Series')),
        (g.SERIES_INDEX, True, _('Series index')),
        (g.EDITION, True, _('Book Edition')),
        (g.EDITION_INDEX, True, _('Book Edition index')),
        (g.TAGS, True, _('Tags')),
        (g.RATING, True, _('Rating (%)')),
        (g.RATING10, True, _('Rating (0-10)')),
        (g.RATING5, True, _('Rating (0-5)')),
        (g.ISBN, True, _('ISBN')),
        (g.XTRANCE, True, _('Identifier xtrance')),
        (g.LANGUAGE, True, _('Language')),
        (g.ORIGINAL_LANGUAGE, True, _('Original language')),
        (g.SOURCE_RELEVANCE, True, _('Source relevance'))
        ]

        # Calibre Mapping Grid
        mapping_group_box = QGroupBox(_('xTrance publishers to calibre publisher mappings'))
        mapping_group_box.setMaximumHeight(800)
        layout.addWidget(mapping_group_box)

        filter_layout = QHBoxLayout()
        self.publishers_filter_check = add_check_option(filter_layout, _('Filter publishers via mappings'), _("Check this and only mapped publishers below will get parsed to Calibre."), g.PUBLISHER_FILTER)
        from calibre.gui2 import get_current_db
        all_publishers = list()
        for s in get_current_db().all_publishers():
            if s[1] not in all_publishers:
                all_publishers.append(s[1])
        self.table_widget = MappingsTableWidget(self, all_publishers, headers=(_('publisher'), _('Publisher')))
        tags_layout = RemapingTableLayout(self, self.table_widget, g.PUBLISHER_MAPPINGS, 'publishers', self.publishers_filter_check)
        mapping_group_box.setLayout(tags_layout)

    def move_cols_left(self):
        self.publisher_line_table.setFocus()
        cols = self.publisher_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        first_sel_col = cols[0].column()
        if first_sel_col <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in selcols:
            self.publisher_line_table.swap_row_widgets(selcol - 1, selcol + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_col - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.publisher_line_table.scrollToItem(self.publisher_line_table.item(scroll_to_row, 0))
        self.publisher_line_string.setText(self.publisher_line_table.get_string())
        #self.renumber_series()

    def move_cols_right(self):
        self.publisher_line_table.setFocus()
        cols = self.publisher_line_table.selectionModel().selectedColumns()
        if len(cols) == 0:
            return
        last_sel_col = cols[-1].column()
        if last_sel_col == self.publisher_line_table.columnCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selcols = []
        for col in cols:
            selcols.append(col.column())
        selcols.sort()
        for selcol in reversed(selcols):
            self.publisher_line_table.swap_row_widgets(selcol + 2, selcol)

        scroll_to_row = last_sel_col + 1
        if scroll_to_row < self.publisher_line_table.columnCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.publisher_line_table.scrollToItem(self.publisher_line_table.item(scroll_to_row, 0))
        self.publisher_line_string.setText(self.publisher_line_table.get_string())
        # self.renumber_series()

    def add_publisher_item(self):
        self.publisher_line_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to publisher"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.publisher_line_table.get_data()
            new_item = [i for i in self.item_options if item in i][0]
            if new_item[0] == g.CUSTOM_TEXT:
                text, ok_text = QInputDialog().getText(self, _("Add custom text"),
                                            _("Write text:"))
                if not ok_text or not text:
                    return
                else:
                    new_item = (g.CUSTOM_TEXT, True, text)
            new_items.append(new_item)
            self.publisher_line_table.populate_table(new_items)
            self.publisher_line_string.setText(self.publisher_line_table.get_string())
            self.publisher_line_table.selectColumn(self.publisher_line_table.columnCount() - 1)
        if not item:
            return

    def delete_publisher_item(self):
        from calibre.gui2 import question_dialog
        self.publisher_line_table.setFocus()
        if not self.publisher_line_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for col in reversed(sorted(self.publisher_line_table.selectionModel().selectedColumns())):
            self.publisher_line_table.removeColumn(col.column())
        self.publisher_line_string.setText(self.publisher_line_table.get_string())

    def reset_publisher_items(self):
        from calibre.gui2 import question_dialog
        self.publisher_line_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set publisher builder to defaults?'),
                show_copy_button=False):
            return
        self.publisher_line_table.setColumnCount(0)
        self.publisher_line_table.populate_table(g.DEFAULT_STORE_VALUES[g.PUBLISHER_LINE])
        self.publisher_line_string.setText(self.publisher_line_table.get_string())

class TagTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_prefs(fill_defaults=True)
        #Tags
        #tags_group_box = QGroupBox(_('Tags'), self)
        #self.l.addWidget(tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        tags_group_boxlayout = QVBoxLayout()
        self.setLayout(tags_group_boxlayout)

        #Append tags group box
        append_tags_group_box = QGroupBox(_('Select what will be added into "Tags" metadata field in Calibre'), self)
        #self.l.addWidget(append_tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        tags_group_boxlayout.addWidget(append_tags_group_box)

        table_layout = QHBoxLayout()
        append_tags_group_box.setLayout(table_layout)

        # here because of translations
        self.DEFAULT_TAGS_ITEMS = [
            (g.TAGS, True, _('Tags')),
            (g.PAGES, False, _('Pages count')),
            (g.PRINT_RUN, False, _('Print run')),
            (g.BOOK_DIMENSIONS, False, _('Book dimensions')),
            (g.ISSUE_NUMBER, False, _('Issue number')),
            (g.IMG_NUMBER, False, _('Images count')),
            (g.ORIGINAL_TITLE, False, _('Original title')),
            (g.TITLE, False, _('Title')),
            (g.SUBTITLE, False, _('Subtitle')),
            (g.PUB_YEAR, False, _('Publication date')),
            (g.ORIGINAL_YEAR, False, _('Original pubdate')),
            (g.PUBLISHER, False, _('Publisher')),
            (g.ORIGINAL_PUBLISHER, False, _('Original publisher')),
            (g.AUTHORS, False, _('Authors')),
            (g.TRANSLATORS, False, _('Translators')),
            (g.ILLUSTRATORS, False, _('Illustrators')),
            (g.SERIES, False, _('Series')),
            (g.EDITION, False, _('Book Edition')),
            (g.RATING, False, _('Rating (%)')),
            (g.RATING10, False, _('Rating (0-10)')),
            (g.RATING5, False, _('Rating (0-5)')),
            (g.ISBN, False, _('ISBN')),
            (g.XTRANCE, False, _('Identifier xtrance')),
            (g.LANGUAGE, False, _('Language')),
            (g.ORIGINAL_LANGUAGE, False, _('Original language')),
            (g.SOURCE_RELEVANCE, False, _('Source relevance'))
        ]
        tag_items = []
        if len(c.get(g.APPEND_TO_TAGS, [])) == len(self.DEFAULT_TAGS_ITEMS):
            for i, tupl in enumerate(c.get(g.APPEND_TO_TAGS, self.DEFAULT_TAGS_ITEMS)):
                tag_items.append((tupl[0], tupl[1], self.DEFAULT_TAGS_ITEMS[i][2]))
        else:
            tag_items = self.DEFAULT_TAGS_ITEMS

        self.tag_items_table = TagBuilderTableWidget(self, tag_items)
        table_layout.addWidget(self.tag_items_table)
        self.tag_items_table.populate_table(tag_items)

        # Calibre Genre Mapping Grid
        genre_group_box = QGroupBox(_('xTrance tag to calibre tag mappings'))
        genre_group_box.setMaximumHeight(800)
        tags_group_boxlayout.addWidget(genre_group_box)
        #self.l.addWidget(genre_group_box, self.l.rowCount(), 0, 1, 2)
        genre_group_box_layout = QVBoxLayout()
        genre_group_box.setLayout(genre_group_box_layout)

        filter_layout = QHBoxLayout()
        self.tags_filter_check = add_check_option(filter_layout, _('Filter tags via mappings'), _("Check this and only mapped tags below will get parsed to Calibre."), g.TAGS_FILTER)

        from calibre.gui2 import get_current_db
        all_tags = get_current_db().all_tags()
        self.table_widget = MappingsTableWidget(self, c[g.TAGS_MAPPINGS])
        tags_layout = RemapingTableLayout(self, self.table_widget, g.TAGS_MAPPINGS, 'tag', self.tags_filter_check)
        genre_group_box_layout.addLayout(tags_layout)

class RemapingTableLayout(QVBoxLayout):
    def __init__(self, parent, table_widget, prefs, name, filter_check_widget=None):
        super(RemapingTableLayout, self).__init__(parent)
        self.parent = parent
        self.name = name
        self.prefs = prefs
        self.table_widget = table_widget
        
        if filter_check_widget:
            filter_check_layout = QVBoxLayout()
            filter_check_layout.addWidget(filter_check_widget)
            self.addLayout(filter_check_layout)

        hbox_layout = QHBoxLayout()
        self.addLayout(hbox_layout)

        hbox_layout.addWidget(self.table_widget)

        button_layout = QVBoxLayout()
        hbox_layout.addLayout(button_layout)

        add_mapping_button = QToolButton()
        add_mapping_button.setToolTip(_('Add mapping (Alt+A)'))
        add_mapping_button.setIcon(QIcon(I('plus.png')))
        add_mapping_button.clicked.connect(self.add_mapping)
        add_mapping_button.setShortcut(_('Alt+A'))
        button_layout.addWidget(add_mapping_button)

        remove_mapping_button = QToolButton()
        remove_mapping_button.setToolTip(_('Delete mapping (Alt+D)'))
        remove_mapping_button.setIcon(QIcon(I('minus.png')))
        remove_mapping_button.clicked.connect(self.delete_mapping)
        remove_mapping_button.setShortcut(_('Alt+D'))
        button_layout.addWidget(remove_mapping_button)

        rename_genre_button = QToolButton()
        rename_genre_button.setToolTip(_('Rename item (Alt+S)'))
        rename_genre_button.setIcon(QIcon(I('edit-undo.png')))
        rename_genre_button.clicked.connect(self.rename_genre)
        rename_genre_button.setShortcut(_('Alt+S'))
        button_layout.addWidget(rename_genre_button)

        reset_defaults_button = QToolButton()
        reset_defaults_button.setToolTip(_('Reset to plugin default mappings (Alt+R)'))
        reset_defaults_button.setIcon(QIcon(I('restart.png')))
        reset_defaults_button.clicked.connect(self.reset_to_defaults)
        reset_defaults_button.setShortcut(_('Alt+R'))
        button_layout.addWidget(reset_defaults_button)

        # Populate the table
        self.table_widget.populate_table(get_pref(prefs))

    def add_mapping(self):
        from calibre.gui2 import error_dialog
        new_genre_name, ok = QInputDialog.getText(self.parent, _('Add new mapping'),
                    _('Enter a xTrance %s name to create a mapping for:')%self.name, text='')
        if not ok:
            # Operation cancelled
            return
        new_genre_name = as_unicode(new_genre_name).strip().replace(',', ';')
        if not new_genre_name:
            return
        # Verify it does not clash with any other mappings in the list
        data = self.table_widget.get_data()
        for genre_name in data.keys():
            if genre_name.lower() == new_genre_name.lower():
                return error_dialog(self.parent, _('Add Failed'), _('A %s with the same name already exists')%self.name, show=True)
        data[new_genre_name] = []
        self.table_widget.populate_table(data)
        self.table_widget.select_genre(new_genre_name)

    def delete_mapping(self):
        from calibre.gui2 import question_dialog
        if not self.table_widget.selectionModel().hasSelection():
            return
        if not question_dialog(self.parent, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected %s mappings?')%self.name,
                show_copy_button=False):
            return
        for row in reversed(sorted(self.table_widget.selectionModel().selectedRows())):
            self.table_widget.removeRow(row.row())

    def rename_genre(self):
        from calibre.gui2 import error_dialog
        selected_genre = self.table_widget.get_selected_genre()
        if not selected_genre:
            return
        new_genre_name, ok = QInputDialog.getText(self.parent, _('Add new mapping'),
                    _('Enter a xTrance %s name to create a mapping for:')%self.name, text=selected_genre)
        if not ok:
            # Operation cancelled
            return
        new_genre_name = as_unicode(new_genre_name).strip().replace(',', ';')
        if not new_genre_name or new_genre_name == selected_genre:
            return
        data = self.table_widget.get_data()
        if new_genre_name.lower() != selected_genre.lower():
            # Verify it does not clash with any other mappings in the list
            for genre_name in data.keys():
                if genre_name.lower() == new_genre_name.lower():
                    return error_dialog(self.parent, _('Rename Failed'), _('A %s with the same name already exists')%self.name, show=True)
        data[new_genre_name] = data[selected_genre]
        del data[selected_genre]
        self.table_widget.populate_table(data)
        self.table_widget.select_genre(new_genre_name)

    def reset_to_defaults(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self.parent, _('Are you sure?'),
            _('Are you sure you want to reset to the plugin default %s mappings?')%self.name,
                show_copy_button=False):
            return
        self.table_widget.populate_table(g.DEFAULT_STORE_VALUES[self.prefs])

class CommentsTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_prefs(fill_defaults=True)
        layout = QVBoxLayout()
        self.setLayout(layout)
        # Comment builder
        comment_group_box = QGroupBox(_('Comment builder (What will be parsed into Comments field.)'))
        #self.l.addWidget(comment_group_box, self.l.rowCount()-3, 2, 3, 1)
        comment_group_box_layout = QVBoxLayout()
        layout.addWidget(comment_group_box)
        comment_group_box.setLayout(comment_group_box_layout)

        table_layout = QHBoxLayout()
        comment_group_box_layout.addLayout(table_layout)
        self.comments_arrange_table = CommentBuilderTableWidget(self, c.get(g.APPEND_TO_COMMENTS, g.DEFAULT_STORE_VALUES[g.APPEND_TO_COMMENTS]))
        table_layout.addWidget(self.comments_arrange_table)
        self.comments_arrange_table.populate_table(c.get(g.APPEND_TO_COMMENTS, g.DEFAULT_STORE_VALUES[g.APPEND_TO_COMMENTS]))

        table_button_layout = QVBoxLayout()
        table_layout.addLayout(table_button_layout)
        plus_button = QToolButton(self)
        plus_button.setToolTip(_('Add item (Alt+A)'))
        plus_button.setIcon(QIcon(I('plus.png')))
        plus_button.setShortcut(_('Alt+A'))
        plus_button.clicked.connect(self.add_comment_item)
        table_button_layout.addWidget(plus_button)
        minus_button = QToolButton(self)
        minus_button.setToolTip(_('Delete item (Alt+D)'))
        minus_button.setIcon(QIcon(I('minus.png')))
        minus_button.setShortcut(_('Alt+D'))
        minus_button.clicked.connect(self.delete_comment_item)
        table_button_layout.addWidget(minus_button)

        add_all_button = QToolButton(self)
        add_all_button.setToolTip(_('Add all items at once (Alt+Q)'))
        add_all_button.setIcon(QIcon(I('sort.png')))
        add_all_button.setShortcut(_('Alt+Q'))
        add_all_button.clicked.connect(self.add_all_comment_items)
        table_button_layout.addWidget(add_all_button)

        reset_button = QToolButton(self)
        reset_button.setToolTip(_('Reset to defaults (Alt+R)'))
        reset_button.setIcon(QIcon(I('restart.png')))
        reset_button.setShortcut(_('Alt+R'))
        reset_button.clicked.connect(self.reset_comment_items)
        table_button_layout.addWidget(reset_button)

        move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move item up (Alt+W)'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        move_up_button.setShortcut(_('Alt+Up'))
        move_up_button.setShortcut(_('Alt+W'))
        move_up_button.clicked.connect(self.move_rows_up)
        table_button_layout.addWidget(move_up_button)
        move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move item down (Alt+S)'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        move_down_button.setShortcut(_('Alt+Down'))
        move_down_button.setShortcut(_('Alt+S'))
        move_down_button.clicked.connect(self.move_rows_down)
        table_button_layout.addWidget(move_down_button)

        self.item_options = [
        (g.DESCRIPTION, True, _('Description')),
        (g.HR, True, _('——— (Horizontal line)')),
        (g.PAGES, True, _('Pages count')),
        (g.PRINT_RUN, True, _('Print run')),
        (g.BOOK_DIMENSIONS, True, _('Book dimensions')),
        (g.IMG_NUMBER, True, _('Images count')),
        (g.ISSUE_NUMBER, True, _('Issue number')),
        (g.ORIGINAL_TITLE, True, _('Original title')),
        (g.TITLE, True, _('Title')),
        (g.SUBTITLE, True, _('Subtitle')),
        (g.PUB_YEAR, True, _('Publication date')),
        (g.ORIGINAL_YEAR, True, _('Original pubdate')),
        (g.PUBLISHER, True, _('Publisher')),
        (g.ORIGINAL_PUBLISHER, True, _('Original publisher')),
        (g.AUTHORS, True, _('Authors')),
        (g.TRANSLATORS, True, _('Translators')),
        (g.ILLUSTRATORS, True, _('Illustrators')),
        (g.SERIES, True, _('Series')),
        (g.EDITION, True, _('Book Edition')),
        (g.TAGS, True, _('Tags')),
        (g.RATING, True, _('Rating (%)')),
        (g.RATING10, True, _('Rating (0-10)')),
        (g.RATING5, True, _('Rating (0-5)')),
        (g.ISBN, True, _('ISBN')),
        (g.XTRANCE, True, _('Identifier xtrance')),
        (g.LANGUAGE, True, _('Language')),
        (g.ORIGINAL_LANGUAGE, True, _('Original language')),
        (g.SOURCE_RELEVANCE, True, _('Source relevance'))
        ]

    def move_rows_up(self):
        self.comments_arrange_table.setFocus()
        rows = self.comments_arrange_table.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        first_sel_row = rows[0].row()
        if first_sel_row <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in selrows:
            self.comments_arrange_table.swap_row_widgets(selrow - 1, selrow + 1)
        #self.books[selrow-1], self.books[selrow] = self.books[selrow], self.books[selrow-1]

        scroll_to_row = first_sel_row - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.comments_arrange_table.scrollToItem(self.comments_arrange_table.item(scroll_to_row, 0))
        #self.renumber_series()

    def move_rows_down(self):
        self.comments_arrange_table.setFocus()
        rows = self.comments_arrange_table.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        last_sel_row = rows[-1].row()
        if last_sel_row == self.comments_arrange_table.rowCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in reversed(selrows):
            self.comments_arrange_table.swap_row_widgets(selrow + 2, selrow)

        scroll_to_row = last_sel_row + 1
        if scroll_to_row < self.comments_arrange_table.rowCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.comments_arrange_table.scrollToItem(self.comments_arrange_table.item(scroll_to_row, 0))
        # self.renumber_series()

    def add_all_comment_items(self):
        from calibre.gui2 import question_dialog
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete your order and add all available items?'),
                show_copy_button=False):
            return
        self.comments_arrange_table.setRowCount(0)
        #new_items = self.comments_arrange_table.get_data()
        new_items = ([i for i in self.item_options])
        self.comments_arrange_table.populate_table(new_items)
        self.comments_arrange_table.selectRow(self.comments_arrange_table.rowCount() - 1)

    def add_comment_item(self):
        self.comments_arrange_table.setFocus()

        item, ok = QInputDialog().getItem(self, _("Add to comments"),
                                            _("Choose item:"), [tuple_item[2] for tuple_item in self.item_options], 0, False)
        if ok and item:
            new_items = self.comments_arrange_table.get_data()
            new_items.append([i for i in self.item_options if item in i][0])
            self.comments_arrange_table.populate_table(new_items)
            self.comments_arrange_table.selectRow(self.comments_arrange_table.rowCount() - 1)
        if not item:
            return

    def delete_comment_item(self):
        from calibre.gui2 import question_dialog
        self.comments_arrange_table.setFocus()
        if not self.comments_arrange_table.selectionModel().hasSelection():
                return
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to delete the selected item?'),
                show_copy_button=False):
            return
        for row in reversed(sorted(self.comments_arrange_table.selectionModel().selectedRows())):
            self.comments_arrange_table.removeRow(row.row())

    def reset_comment_items(self):
        from calibre.gui2 import question_dialog
        self.comments_arrange_table.setFocus()
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                _('Are you sure you want to set comment builder to defaults?'),
                show_copy_button=False):
            return
        self.comments_arrange_table.setRowCount(0)
        self.comments_arrange_table.populate_table(g.DEFAULT_STORE_VALUES[g.APPEND_TO_COMMENTS])

class IdentifiersTab(QWidget):
    def __init__(self):
        QWidget.__init__(self)
        c = get_prefs(fill_defaults=True)

        layout = QVBoxLayout()
        self.setLayout(layout)

        identifiers_group_boxlayout = QVBoxLayout()
        layout.addLayout(identifiers_group_boxlayout)
        #Append identifiers group box
        append_identifiers_group_box = QGroupBox(_('Select what will be added into "Identifiers" metadata field in Calibre'), self)
        #self.l.addWidget(append_tags_group_box, self.l.rowCount()-2, 1, 2, 1)
        identifiers_group_boxlayout.addWidget(append_identifiers_group_box)

        table_layout = QVBoxLayout()
        append_identifiers_group_box.setLayout(table_layout)

        custom_cols_info = QLabel(_('How to create custom column from this Identificators? Check <a href="https://www.mobileread.com/forums/showpost.php?p=4361821&postcount=2">MobileRead</a> forum.'))
        custom_cols_info.setWordWrap(True)
        custom_cols_info.setOpenExternalLinks(True)
        table_layout.addWidget(custom_cols_info)
        # here because of translations
        # tuple (id, activated, description, id format)
        self.DEFAULT_IDENTIFIERS_ITEMS = [
            (g.XTRANCE, True, _('Identifier xTrance'), 'xtrance'),
            (g.ISBN, True, _('ISBN'), 'isbn'),
            (g.RATING, False, _('Rating (%)'), 'xtr_rating100'),
            (g.RATING10, False, _('Rating (0-10)'), 'xtr_rating10'),
            (g.RATING5, False, _('Rating (0-5)'), 'xtr_rating5'),
            (g.PAGES, False, _('Pages count'), 'xtr_pages'),
            (g.PRINT_RUN, False, _('Print run'), 'xtr_printrun'),
            (g.ISSUE_NUMBER, False, _('Issue number'), 'xtr_issuenumber'),
            (g.IMG_NUMBER, False, _('Images count'), 'xtr_imgnumber'),
            (g.BOOK_DIMENSIONS, False, _('Book dimensions'), 'xtr_dimensions'),
            (g.PUB_YEAR, False, _('Publication date'), 'xtr_pubdate'),
            (g.ORIGINAL_YEAR, False, _('Original pubdate'), 'xtr_origyear'),
            (g.TITLE, False, _('Title'), 'xtr_title'),
            (g.SUBTITLE, False, _('Subtitle'), 'xtr_subtitle'),
            (g.ORIGINAL_TITLE, False, _('Original title'), 'xtr_origtitle'),
            (g.PUBLISHER, False, _('Publisher'), 'xtr_publisher'),
            (g.ORIGINAL_PUBLISHER, False, _('Original publisher'), 'xtr_origpublisher'),
            (g.AUTHORS, False, _('Authors'), 'xtr_authors'),
            (g.TRANSLATORS, False, _('Translators'), 'xtr_translators'),
            (g.ILLUSTRATORS, False, _('Illustrators'), 'xtr_illustrators'),
            (g.SERIES, False, _('Series'), 'xtr_series'),
            (g.EDITION, False, _('Book Edition'), 'xtr_edition'),
            (g.TAGS, False, _('Tags'), 'xtr_tags'),
            (g.LANGUAGE, False, _('Language'), 'xtr_lang'),
            (g.ORIGINAL_LANGUAGE, False, _('Original language'), 'xtr_origlang'),
            (g.SOURCE_RELEVANCE, False, _('Source relevance'), 'xtr_relevance')
        ]
        identifier_items = []
        if len(c.get(g.APPEND_TO_IDENTIFIERS, [])) == len(self.DEFAULT_IDENTIFIERS_ITEMS):
            for i, tupl in enumerate(c.get(g.APPEND_TO_IDENTIFIERS, self.DEFAULT_IDENTIFIERS_ITEMS)):
                identifier_items.append((tupl[0], tupl[1], self.DEFAULT_IDENTIFIERS_ITEMS[i][2], self.DEFAULT_IDENTIFIERS_ITEMS[i][3]))
        else:
            identifier_items = self.DEFAULT_IDENTIFIERS_ITEMS

        self.identifier_items_table = IdentifiersBuilderTableWidget(self, identifier_items)
        table_layout.addWidget(self.identifier_items_table)
        self.identifier_items_table.populate_table(identifier_items)

        # Language
        lang_group_box = QGroupBox(_('Languages'), self)
        lang_group_box_layout = QHBoxLayout()
        lang_group_box.setLayout(lang_group_box_layout)
        self.lang_check = add_combobox_option(lang_group_box_layout, _("Set what language will be parsed into Calibre"), _('You can select actual issue language or original book language.'),
                            g.LANGUAGE_INCLUDE, choices=[_('Actual language'), _('Original language')])
        
        lang_group_box_layout.addStretch(1)
        layout.addWidget(lang_group_box)

        layout.addStretch(1)
