﻿/****************************************************************************
*
* SAMPLE: LZSS Library
* Copyright (C) 2004, 2006, 2007 by
* Michael Dipperstein (mdipper@alumni.engr.ucsb.edu)
*
* This file is part of the lzss library.
*
* The lzss library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 3 of the
* License, or (at your option) any later version.
*
* The lzss library is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
* General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*
* RES File Text Decompression algorithm is tweak by Nick Rapallo and
* Michael Scott.  The changes to the original algorithm by 
* Michael Dipperstein are made this IMP Decompression specifically
*
*
*
***************************************************************************/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Xml;

using System.IO;
using BBeBLib;

namespace ConvertIMPGUI
{
    public partial class frmMain : Form
    {
        XmlDocument IMPXMLDoc = new XmlDocument();
        XmlElement FolderNode = null;
        XmlElement CategoriesNode = null;
        XmlElement BooksNode = null;
        string IMPXMLFilePath = Application.StartupPath + @"\Library.xml";
        //string IMPLRSFilePath = Application.StartupPath + @"\tmp.lrs";
        CIMPFile impFile = null;

        bool bSortAuthor = false;
        bool bSortTitle = false;
        bool bSortCategory = false;
        bool bSortIMPType = false;
        bool bSortSize = false;

        #region Constructor
        public frmMain()
        {
            InitializeComponent();

            if (!File.Exists(IMPXMLFilePath))
                CreateLibraryFile();

            IMPXMLDoc.Load(IMPXMLFilePath);
        }
        #endregion

        #region CreateLibraryFile
        private void CreateLibraryFile()
        {
            XmlElement root = IMPXMLDoc.CreateElement("IMPGUI");
            XmlElement folder = IMPXMLDoc.CreateElement("Folder");
            XmlElement category = IMPXMLDoc.CreateElement("Categories");
            XmlElement book = IMPXMLDoc.CreateElement("Books");

            root.AppendChild(folder);
            root.AppendChild(category);
            root.AppendChild(book);

            IMPXMLDoc.AppendChild(root);

            IMPXMLDoc.Save(IMPXMLFilePath);

            IMPXMLDoc = new XmlDocument();
        }
        #endregion

        #region frmMain_Load
        private void frmMain_Load(object sender, EventArgs e)
        {
            FolderNode = IMPXMLDoc.DocumentElement["Folder"];
            CategoriesNode = IMPXMLDoc.DocumentElement["Categories"];
            BooksNode = IMPXMLDoc.DocumentElement["Books"];

            if (FolderNode.InnerText != "")
                txtIMPFolder.Text = FolderNode.InnerText;

            cboLibraryCategories.Items.Add("All");

            foreach (XmlElement category in CategoriesNode.ChildNodes)
                cboLibraryCategories.Items.Add(category.InnerText);

            cboLibraryCategories.SelectedIndex = 0;
        }
        #endregion

        #region LoadBooks
        private void LoadBooks(bool bRefreshBookList)
        {
            Cursor = Cursors.WaitCursor;
            Application.DoEvents();

            lstBookShelf.Items.Clear();

            if (chkIncludeSubdirectories.Checked)
                LoadAllDirectories(bRefreshBookList, txtIMPFolder.Text);
            else
                LoadSingleDirectory(bRefreshBookList, txtIMPFolder.Text);

            Cursor = Cursors.Default;
            Application.DoEvents();
        }
        #endregion

        #region LoadAllDirectories
        private void LoadAllDirectories(bool bRefreshBookList, string path)
        {
            LoadSingleDirectory(bRefreshBookList, path);

            string[] dirs = Directory.GetDirectories(path);
            foreach (string dir in dirs)
                LoadAllDirectories(bRefreshBookList, dir);
        }
        #endregion

        #region LoadSingleDirectory
        private void LoadSingleDirectory(bool bRefreshBookList, string path)
        {
            string cat = "";

            if (cboLibraryCategories.SelectedIndex != 0)
                cat = cboLibraryCategories.SelectedItem.ToString();

            if (bRefreshBookList)
            {
                string[] files = Directory.GetFiles(path);

                pnlProgress.Visible = true;
                lblFileProgress.Text = "";
                fileProgress.Value = 0;
                fileProgress.Minimum = 0;
                fileProgress.Maximum = files.Length;

                Application.DoEvents();

                foreach (string file in files)
                {
                    fileProgress.Value = fileProgress.Value + 1;
                    lblFileProgress.Text = "File " + fileProgress.Value + " of " + fileProgress.Maximum;
                    Application.DoEvents();

                    if (file.ToLower().EndsWith(".imp"))
                    {
                        FileInfo fi = new FileInfo(file);
                        impFile = new CIMPFile(file);

                        impFile.ReadForDisplay();

                        #region Add Category if Not in List
                        if (cboLibraryCategories.Items.Count == 1)
                        {
                            if (!cboLibraryCategories.Items.Contains(impFile.BookProperties.Bookshelf_Category))
                            {
                                cboLibraryCategories.Items.Add(impFile.BookProperties.Bookshelf_Category);
                                XmlElement c = IMPXMLDoc.CreateElement("Category");
                                c.InnerText = impFile.BookProperties.Bookshelf_Category;
                                CategoriesNode.AppendChild(c);
                                IMPXMLDoc.Save(IMPXMLFilePath);
                            }
                        }
                        #endregion

                        #region Search For Book (Add If Not Found)
                        bool bFound = false;
                        foreach (XmlElement b in BooksNode.ChildNodes)
                        {
                            if (b["Title"].InnerText == impFile.BookProperties.Title &&
                                b["Author"].InnerText == impFile.BookProperties.FirstName &&
                                b["IMPType"].InnerText == impFile.Header.impTypeDevice &&
                                b["File"].InnerText == file)
                            {
                                bFound = true;
                                break;
                            }
                        }

                        if (!bFound)
                        {
                            XmlElement bi = IMPXMLDoc.CreateElement("Book");
                            XmlElement bf = IMPXMLDoc.CreateElement("File");
                            XmlElement bt = IMPXMLDoc.CreateElement("Title");
                            XmlElement ba = IMPXMLDoc.CreateElement("Author");
                            XmlElement bc = IMPXMLDoc.CreateElement("Category");
                            XmlElement bimp = IMPXMLDoc.CreateElement("IMPType");

                            bf.InnerText = file;
                            bt.InnerText = impFile.BookProperties.Title;
                            ba.InnerText = impFile.BookProperties.FirstName;
                            bc.InnerText = impFile.BookProperties.Bookshelf_Category;
                            bimp.InnerText = impFile.Header.impTypeDevice;

                            bi.AppendChild(bt);
                            bi.AppendChild(ba);
                            bi.AppendChild(bc);
                            bi.AppendChild(bf);
                            bi.AppendChild(bimp);
                            BooksNode.AppendChild(bi);
                        }
                        #endregion

                        #region Load From IMP Files
                        if (cat == "")
                        {
                            ListViewItem item = new ListViewItem();
                            item.Tag = impFile;
                            item.Text = impFile.BookProperties.FirstName;
                            item.SubItems.Add(impFile.BookProperties.Title);
                            item.SubItems.Add(impFile.BookProperties.Bookshelf_Category);
                            item.SubItems.Add(impFile.Header.impTypeDevice);
                            item.SubItems.Add(fi.Length + " bytes");

                            lstBookShelf.Items.Add(item);
                        }
                        else if (impFile.BookProperties.Bookshelf_Category == cat)
                        {
                            ListViewItem item = new ListViewItem();
                            item.Tag = impFile;
                            item.Text = impFile.BookProperties.FirstName;
                            item.SubItems.Add(impFile.BookProperties.Title);
                            item.SubItems.Add(impFile.BookProperties.Bookshelf_Category);
                            item.SubItems.Add(impFile.Header.impTypeDevice);
                            item.SubItems.Add(fi.Length + " bytes");

                            lstBookShelf.Items.Add(item);
                        }
                        #endregion
                    }
                }
                IMPXMLDoc.Save(IMPXMLFilePath);
            }
            else
            {
                #region Load From XML
                pnlProgress.Visible = true;
                lblFileProgress.Text = "";
                fileProgress.Value = 0;
                fileProgress.Minimum = 0;
                fileProgress.Maximum = BooksNode.ChildNodes.Count;
                Application.DoEvents();

                List<XmlElement> rb = new List<XmlElement>();

                foreach (XmlElement book in BooksNode.ChildNodes)
                {
                    fileProgress.Value = fileProgress.Value + 1;
                    lblFileProgress.Text = "File " + fileProgress.Value + " of " + fileProgress.Maximum;
                    Application.DoEvents();

                    if (File.Exists(book["File"].InnerText))
                    {
                        if (cat == "")
                        {
                            FileInfo fi = new FileInfo(book["File"].InnerText);
                            impFile = new CIMPFile(book["File"].InnerText);
                            ListViewItem item = new ListViewItem();
                            item.Tag = impFile;
                            item.Text = book["Author"].InnerText;
                            item.SubItems.Add(book["Title"].InnerText);
                            item.SubItems.Add(book["Category"].InnerText);
                            item.SubItems.Add(book["IMPType"].InnerText);
                            item.SubItems.Add(fi.Length + " bytes");

                            lstBookShelf.Items.Add(item);
                        }
                        else if (book["Category"].InnerText == cat)
                        {
                            FileInfo fi = new FileInfo(book["File"].InnerText);
                            impFile = new CIMPFile(book["File"].InnerText);
                            ListViewItem item = new ListViewItem();
                            item.Tag = impFile;
                            item.Text = book["Author"].InnerText;
                            item.SubItems.Add(book["Title"].InnerText);
                            item.SubItems.Add(book["Category"].InnerText);
                            item.SubItems.Add(book["IMPType"].InnerText);
                            item.SubItems.Add(fi.Length + " bytes");

                            lstBookShelf.Items.Add(item);
                        }
                    }
                    else
                    {
                        rb.Add(book);
                    }
                }

                foreach (XmlElement r in rb)
                {
                    BooksNode.RemoveChild(r);
                }

                IMPXMLDoc.Save(IMPXMLFilePath);
                #endregion
            }

            pnlProgress.Visible = false;
            lblFileProgress.Text = "";
            fileProgress.Value = 0;
            fileProgress.Minimum = 0;
        }
        #endregion

        #region mnuFile_Exit
        private void mnuFile_Exit_Click(object sender, EventArgs e)
        {
            this.Close();
        }
        #endregion

        #region LoadIMPFile
        private void LoadIMPFile(string IMPFileName)
        {
            Cursor = Cursors.WaitCursor;
            Application.DoEvents();

            impFile = new CIMPFile(IMPFileName);

            frmProgress frm = new frmProgress();
            frm.IMPFile = impFile;
            frm.Show();
            Application.DoEvents();
            frm.StartProcessing();
            impFile = frm.IMPFile;
            frm.Close();
            frm.Dispose();
            frm = null;

            //txtOutputFileName.Text = impFile.BookProperties.FirstName + " - " + impFile.BookProperties.Title + ".txt";
            LoadIMPTree();

            IMPTabs.Enabled = true;
            //txtOutputDirectory.Enabled = true;
            //txtOutputFileName.Enabled = true;
            mnuCreateHTML.Enabled = true;

            Cursor = Cursors.Default;
            Application.DoEvents();
        }
        #endregion

        #region LoadIMPTree
        private void LoadIMPTree()
        {
            IMPTree.Nodes.Clear();

            TreeNode headerNode = IMPTree.Nodes.Add("Header");
            TreeNode bookNode = IMPTree.Nodes.Add("Book Properties");
            TreeNode tocNode = IMPTree.Nodes.Add("Table Of Contents");
            TreeNode swIndexesNode = IMPTree.Nodes.Add("SW Indexes");
            TreeNode imageRunNode = IMPTree.Nodes.Add("Book Images");
            TreeNode styleNode = IMPTree.Nodes.Add("Styles");
            TreeNode extStyleNode = IMPTree.Nodes.Add("Extended Styles");
            TreeNode styleRunNode = IMPTree.Nodes.Add("Style Runs");
            TreeNode pPicNode = IMPTree.Nodes.Add("PPics");
            TreeNode Pcz1Node = IMPTree.Nodes.Add("Pcz1");
            TreeNode PcZ1Node = IMPTree.Nodes.Add("PcZ1");
            TreeNode tableNode = IMPTree.Nodes.Add("Tables");
            TreeNode tableRowNode = IMPTree.Nodes.Add("Table Rows");
            TreeNode tableCellNode = IMPTree.Nodes.Add("Table Cells");

            #region Header Display
            headerNode.Nodes.Add("IMP Version (" + impFile.Header.version + ")");

            headerNode.Nodes.Add("IMP Device (" + impFile.Header.impTypeDevice + ")");

            headerNode.Nodes.Add("RES File Count (" + impFile.Header.fileCount + ")");

            headerNode.Nodes.Add("Bytes Remaining In Header (" + impFile.Header.bytesRemaining + ")");

            if (impFile.Header.compression)
                headerNode.Nodes.Add("Compressed");
            else
                headerNode.Nodes.Add("UnCompressed");

            if (impFile.Header.encryption)
                headerNode.Nodes.Add("Encrypted");
            else
                headerNode.Nodes.Add("UnEncrypted");

            if (impFile.Header.zoomState == 0)
                headerNode.Nodes.Add("Zoom State ( Small & Large View )");
            else if (impFile.Header.zoomState == 1)
                headerNode.Nodes.Add("Zoom State ( Small View )");
            else if (impFile.Header.zoomState == 1)
                headerNode.Nodes.Add("Zoom State ( Large View )");
            #endregion

            #region Book Display
            bookNode.Nodes.Add("Category (" + impFile.BookProperties.Bookshelf_Category + ")");
            bookNode.Nodes.Add("SubCategory (" + impFile.BookProperties.Subcategory + ")");
            bookNode.Nodes.Add("Title (" + impFile.BookProperties.Title + ")");
            bookNode.Nodes.Add("Author First Name (" + impFile.BookProperties.FirstName + ")");
            bookNode.Nodes.Add("Author Middle Name (" + impFile.BookProperties.MiddleName + ")");
            bookNode.Nodes.Add("Author Last Name (" + impFile.BookProperties.LastName + ")");
            bookNode.Nodes.Add("ID (" + impFile.BookProperties.ID + ")");
            #endregion

            #region TOC Display
            foreach (IMPTableContentsItem item in impFile.TableOfContents)
            {
                TreeNode tocItem = null;

                if (item.fileType == "")
                    tocItem = tocNode.Nodes.Add("TOC File (DATA)");
                else
                    tocItem = tocNode.Nodes.Add("TOC File (" + item.fileType + ")");
                tocItem.Nodes.Add("File Type (" + item.fileType + ")");
                tocItem.Nodes.Add("Data Size (" + item.fileSize + " bytes)");
                tocItem.Nodes.Add("Indexes (" + item.indexes + ")");
            }
            #endregion

            #region SWIndexes Display
            foreach (SWIndex item in impFile.SWIndexes)
            {
                TreeNode swItem = swIndexesNode.Nodes.Add("Index (" + item.fileType + ")");
            }
            #endregion

            #region ImageRuns Display
            int indexGIF = 0;
            int indexJPEG = 0;
            int indexPNG = 0;
            int indexPICT = 0;

            foreach (IMGRunItem item in impFile.ImageRuns)
            {
                TreeNode imgItem = imageRunNode.Nodes.Add("Image (" + item.imageFileType + ")");
                imgItem.Nodes.Add("Width (" + item.imageWidth + ")");
                imgItem.Nodes.Add("Height (" + item.imageHeight + ")");
                imgItem.Nodes.Add("Location In DATA (" + item.dataOffset + ")");
                imgItem.Nodes.Add("Image ID (" + string.Format("{0:X4}", item.imageID) + ")");
                imgItem.Nodes.Add("Image Type (" + item.imageFileType + ")");

                if (item.imageFileType == "GIF ")
                {
                    indexGIF++;
                    MemoryStream ms = impFile.GetImage(item.imageID, ref item.dataFile);
                    item.Image = Image.FromStream(ms);
                }
                else if (item.imageFileType == "JPEG")
                {
                    indexJPEG++;
                    MemoryStream ms = impFile.GetImage(item.imageID, ref item.dataFile);
                    item.Image = Image.FromStream(ms);
                }
                else if (item.imageFileType == "PNG ")
                {
                    indexPNG++;
                    MemoryStream ms = impFile.GetImage(item.imageID, ref item.dataFile);
                    item.Image = Image.FromStream(ms);
                }
                else if (item.imageFileType == "PICT")
                {
                    indexPICT++;
                    MemoryStream ms = impFile.GetImage(item.imageID, ref item.dataFile);
                    item.Image = Image.FromStream(ms);
                }
                imgItem.Tag = item;
            }
            #endregion

            #region Styles Display
            int styleIndex = 0;
            foreach (IMPStyleItem item in impFile.TextStyles)
            {
                styleIndex++;
                TreeNode stylItem = styleNode.Nodes.Add("Style (" + styleIndex + ")");
                stylItem.Nodes.Add("Element (" + item.element + ")");
                stylItem.Nodes.Add("Font Family (" + item.fontFamily + ")");
                stylItem.Nodes.Add("Font Family Name (" + item.fontFamilyName + ")");
                stylItem.Nodes.Add("Style (" + item.style + ")");
                stylItem.Nodes.Add("Font Size Number (" + item.fontRealSize + ")");
                stylItem.Nodes.Add("Font Size (" + item.fontSize + ")");
                stylItem.Nodes.Add("Text Align (" + item.textAlign + ")");
                stylItem.Nodes.Add("Text Color (" + string.Format("0x{0:X2} R 0x{1:X2} G 0x{2:X2} B", item.textColorRed, item.textColorGreen, item.textColorBlue) + ")");
                stylItem.Nodes.Add("Text BGColor (" + string.Format("0x{0:X2} R 0x{1:X2} G 0x{2:X2} B", item.textBGColorRed, item.textBGColorGreen, item.textBGColorBlue) + ")");
                stylItem.Nodes.Add("Top Margin (" + item.marginTop + ")");
                stylItem.Nodes.Add("Text Indent (" + item.textIndent + ")");
                stylItem.Nodes.Add("Right Margin (" + item.marginRight + ")");
                stylItem.Nodes.Add("Left Margin (" + item.marginLeft + ")");
                stylItem.Nodes.Add("OEB Column Number (" + item.oebColumnNumber + ")");
                stylItem.Tag = item;
            }
            #endregion

            #region Extended Styles Display
            int extStyleIndex = 0;
            foreach (IMPExtendedStyleItem item in impFile.TextExtendedStyles)
            {
                extStyleIndex++;
                TreeNode extStylItem = extStyleNode.Nodes.Add("Extended Style (" + extStyleIndex + ")");
                extStylItem.Nodes.Add("Resource1 (" + item.Resource1 + ")");
                extStylItem.Nodes.Add("x-sbp-orphan-pull (" + item.OrphanPull + ")");
                extStylItem.Nodes.Add("x-sbp-widow-push (" + item.WidowPush + ")");
                extStylItem.Nodes.Add("Resource2 (" + item.Resource2 + ")");
                extStylItem.Tag = item;
            }
            #endregion

            #region Style Run Display
            int StyleRunIndex = 0;
            foreach (IMPStyleRunItem item in impFile.StyleRuns)
            {
                StyleRunIndex++;
                TreeNode styleRun = styleRunNode.Nodes.Add("Style Run (" + StyleRunIndex + ")");
                styleRun.Nodes.Add("Style Index (" + item.StyleIndex + ")");
                styleRun.Nodes.Add("Offset (" + item.Offset + ")");
            }
            #endregion

            #region PPic Display
            int PPicIndex = 0;
            foreach (IMPPPic item in impFile.PPics)
            {
                PPicIndex++;
                TreeNode pPicItem = pPicNode.Nodes.Add("PPIC (" + PPicIndex + ")");
                //pPicItem.Nodes.Add("Has Pictures (" + item.HasPictures.ToString() + ")");
                //pPicItem.Nodes.Add("Has Borders (" + item.HasBorders.ToString() + ")");
                pPicItem.Nodes.Add("Pictures Count (" + item.CountOfImages + ")");
                pPicItem.Nodes.Add("Borders Count (" + item.CountOfCellAndTableBorders + ")");
                pPicItem.Tag = item;
            }
            #endregion

            #region Pcz1 Display
            int Pcz1Index = 0;
            foreach (IMPPcZ1 item in impFile.Pcz1)
            {
                Pcz1Index++;
                TreeNode Pcz1Item = Pcz1Node.Nodes.Add("Pcz1 (" + Pcz1Index + ")");
                Pcz1Item.Nodes.Add("ID (" + item.ID + ")");
                Pcz1Item.Nodes.Add("Length (" + item.Length + ")");
                Pcz1Item.Nodes.Add("Offset (" + item.Offset + ")");
                Pcz1Item.Tag = item;
            }
            #endregion

            #region PcZ1 Display
            int PcZ1Index = 0;
            foreach (IMPPcZ1 item in impFile.PcZ1)
            {
                PcZ1Index++;
                TreeNode PcZ1Item = PcZ1Node.Nodes.Add("PcZ1 (" + PcZ1Index + ")");
                PcZ1Item.Nodes.Add("ID (" + item.ID + ")");
                PcZ1Item.Nodes.Add("Length (" + item.Length + ")");
                PcZ1Item.Nodes.Add("Offset (" + item.Offset + ")");
                PcZ1Item.Tag = item;
            }
            #endregion

            #region Table Display
            int TableIndex = 0;
            foreach (IMPTableItem item in impFile.Tables)
            {
                TableIndex++;
                TreeNode tableItem = tableNode.Nodes.Add("Table (" + TableIndex + ")");
                tableItem.Nodes.Add("Align (" + item.AlignType + ")");
                if (item.Width != -1)
                    tableItem.Nodes.Add("Width (" + item.Width + "%)");
                else
                    tableItem.Nodes.Add("Width (" + item.Width + ")");
                tableItem.Nodes.Add("Border (" + item.Border + ")");
                tableItem.Nodes.Add("CellSpacing (" + item.CellSpacing + ")");
                tableItem.Nodes.Add("CellPadding (" + item.CellPadding + ")");
                tableItem.Nodes.Add("Caption Present (" + item.CaptionPresent.ToString() + ")");
                tableItem.Nodes.Add("Caption Length (" + item.CaptionLength + ")");
                tableItem.Nodes.Add("List-Style-Type (" + item.ListStyleType + ")");
                tableItem.Nodes.Add("TRow ID (" + item.TRowID + ")");
                tableItem.Tag = item;
            }
            #endregion

            #region TableRow Display
            int TableRowGroupIndex = 0;
            foreach (IMPTableRowGroup rowGroup in impFile.TableRows)
            {
                TableRowGroupIndex++;
                TreeNode tableRowGroup = tableRowNode.Nodes.Add("Table Row Group (" + TableRowGroupIndex + ")");
                tableRowGroup.Tag = rowGroup;

                int TableRowIndex = 0;
                foreach (IMPTableRowItem item in rowGroup.Rows)
                {
                    TableRowIndex++;
                    TreeNode tableRowItem = tableRowGroup.Nodes.Add("Table Row (" + TableRowIndex + ")");
                    tableRowItem.Nodes.Add("Row ID (" + item.TRowID + ")");

                    if (impFile.REB1200)
                    {
                        tableRowItem.Nodes.Add("Row Type (" + item.RowType + ")");
                        tableRowItem.Nodes.Add("Border (" + item.Border + ")");
                        tableRowItem.Nodes.Add("TCell ID (" + item.TCelID + ")");
                        tableRowItem.Nodes.Add("Offset (" + item.Offset + ")");
                        tableRowItem.Nodes.Add("Length (" + item.Length + ")");
                    }
                    else
                    {
                    }

                    tableRowItem.Tag = item;
                }
            }
            #endregion

            #region TableRowCell Display
            int TableCellGroupIndex = 0;
            foreach (IMPTableCellGroup cellGroup in impFile.TableRowCells)
            {
                TableCellGroupIndex++;
                TreeNode tableCellGroup = tableCellNode.Nodes.Add("Table Cell Group (" + TableCellGroupIndex + ")");
                tableCellNode.Tag = cellGroup;

                int TableRowCellIndex = 0;
                foreach (IMPTableCellItem item in cellGroup.Cells)
                {
                    TableRowCellIndex++;
                    TreeNode tableRowCellItem = tableCellGroup.Nodes.Add("Table Cell (" + TableRowCellIndex + ")");
                    tableRowCellItem.Nodes.Add("Cell ID (" + item.TCelID + ")");

                    if (impFile.REB1200)
                    {
                        tableRowCellItem.Nodes.Add("Cell Type (" + item.CellType + ")");
                        tableRowCellItem.Nodes.Add("VAlign (" + item.VerticalAlign + ")");
                        tableRowCellItem.Nodes.Add("Width (" + item.Width + ")");
                        tableRowCellItem.Nodes.Add("Height (" + item.Height + ")");
                        tableRowCellItem.Nodes.Add("Offset (" + item.Offset + ")");
                        tableRowCellItem.Nodes.Add("Length (" + item.Length + ")");
                        if (item.BGColor == Color.White)
                            tableRowCellItem.Nodes.Add("BackGround Color (None)");
                        else
                            tableRowCellItem.Nodes.Add("BackGround Color (" + string.Format("0x{0:X2} R 0x{1:X2} G 0x{2:X2} B", item.BGColorRed, item.BGColorGreen, item.BGColorBlue) + ")");
                    }
                    else
                    {
                    }
                    tableRowCellItem.Tag = item;
                }
            }
            #endregion
        }
        #endregion

        #region IMPTree_AfterSelect
        private void IMPTree_AfterSelect(object sender, TreeViewEventArgs e)
        {
            if (e.Node.Text.StartsWith("Image "))
            {
                if (e.Node.Tag != null)
                {
                    IMGRunItem resImage = (IMGRunItem)e.Node.Tag;
                    imageBox.Image = resImage.Image;
                    imageBox.Tag = resImage;
                }
            }
        }
        #endregion
        
        #region IMPTabs_SelectedIndexChanged
        private void IMPTabs_SelectedIndexChanged(object sender, EventArgs e)
        {
            txtIMPTreeText.Text = "";
            pnlIMPTreeProgress.Visible = false;
            txtIMPTreeText.Visible = false;

            if (lstBookShelf.SelectedItems.Count == 0)
                return;
            else
            {
                impFile = (CIMPFile)lstBookShelf.SelectedItems[0].Tag;
                //txtOutputDirectory.Text = txtIMPFolder.Text;

                Cursor = Cursors.WaitCursor;
                Application.DoEvents();

                frmProgress frm = new frmProgress();
                frm.IMPFile = impFile;
                frm.Show();
                Application.DoEvents();
                frm.StartProcessing();
                impFile = frm.IMPFile;
                frm.Close();
                frm.Dispose();
                frm = null;

                LoadIMPTree();

                mnuCreateHTML.Enabled = true;

                Cursor = Cursors.Default;
                Application.DoEvents();
            }

            if (IMPTabs.SelectedTab == tabBookProperties)
            {
                cboBP_Category.Items.Clear();
                foreach (string sCat in cboLibraryCategories.Items)
                    cboBP_Category.Items.Add(sCat);
                txtBP_Category.Text = impFile.BookProperties.Bookshelf_Category;
                txtBP_FirstName.Text = impFile.BookProperties.FirstName;
                txtBP_ID.Text = impFile.BookProperties.ID;
                txtBP_LastName.Text = impFile.BookProperties.LastName;
                txtBP_MiddleName.Text = impFile.BookProperties.MiddleName;
                txtBP_SubCategory.Text = impFile.BookProperties.Subcategory;
                txtBP_Title.Text = impFile.BookProperties.Title;
                txtBP_RESFileName.Text = impFile.BookProperties.ResFile;

                txtNewIMPFile.Text = impFile.BookProperties.FirstName + " - " + impFile.BookProperties.Title;
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace("/", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace(@"\", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace(":", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace("?", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace("*", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace("<", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace(">", " ");
                txtNewIMPFile.Text = txtNewIMPFile.Text.Replace("|", " ");
                txtNewIMPFile.Text += ".imp";
            }
            else if (IMPTabs.SelectedTab == tabContent)
            {
                if (txtContent.Text == "")
                {
                    MessageBox.Show("May take a while depending on size of IMP File");

                    Cursor = Cursors.WaitCursor;
                    Application.DoEvents();

                    impFile.ReverseResourceFile();

                    txtContent.Text = impFile.Content;

                    Cursor = Cursors.Default;
                    Application.DoEvents();
                }
            }
        }
        #endregion

        #region btnSaveBookProperties
        private void btnSaveBookProperties_Click(object sender, EventArgs e)
        {
            IMPBookProperties newProperties = new IMPBookProperties();
            newProperties.Bookshelf_Category = txtBP_Category.Text;
            newProperties.FirstName = txtBP_FirstName.Text;
            newProperties.ID = txtBP_ID.Text;
            newProperties.LastName = txtBP_LastName.Text;
            newProperties.MiddleName = txtBP_MiddleName.Text;
            newProperties.Subcategory = txtBP_SubCategory.Text;
            newProperties.Title = txtBP_Title.Text;
            newProperties.ResFile = txtBP_RESFileName.Text;

            impFile.ChangeBookProperties(newProperties);

            LoadIMPFile(impFile.IMPFilePath);
            LoadIMPTree();

            IMPTabs.SelectedIndex = 0;
        }
        #endregion

        #region btnRenameIMPFile
        private void btnRenameIMPFile_Click(object sender, EventArgs e)
        {
            // Rename File
            string origFile = impFile.IMPFilePath;
            string newFile = origFile.Substring(0, origFile.LastIndexOf(@"\") + 1) + txtNewIMPFile.Text;

            try
            {

                if (File.Exists(newFile))
                {
                    if (MessageBox.Show("Do you want to overwrite file: " + newFile, "Overwrite File", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                    {
                        File.Copy(origFile, newFile, true);

                        LoadIMPFile(newFile);
                        LoadIMPTree();

                        IMPTabs.SelectedIndex = 0;

                        File.Delete(origFile);

                        //txtInputFilePath.Text = newFile;
                    }
                    else
                        return;
                }
                else
                {
                    File.Copy(origFile, newFile, true);

                    LoadIMPFile(newFile);
                    LoadIMPTree();

                    IMPTabs.SelectedIndex = 0;

                    File.Delete(origFile);

                    //txtInputFilePath.Text = newFile;
                }
            }
            catch
            {
                MessageBox.Show("The Original IMP File is in use.");
            }
        }
        #endregion

        #region mnuRenameAllIMPFiles
        private void mnuRenameAllIMPFiles_Click(object sender, EventArgs e)
        {
            string directory = "";

            if (folderDlg.ShowDialog() == DialogResult.OK)
            {
                Cursor = Cursors.WaitCursor;
                Application.DoEvents();

                directory = folderDlg.SelectedPath;

                frmRenameProgress frm = new frmRenameProgress();
                frm.IMPDirectory = directory;
                frm.ShowDialog(this);
                frm.Dispose();
                frm = null;

                Cursor = Cursors.Default;
                Application.DoEvents();
            }
        }
        #endregion

        #region btnGenerateNewRESFileName
        private void btnGenerateNewRESFileName_Click(object sender, EventArgs e)
        {
            txtBP_RESFileName.Text = impFile.BookProperties.FirstName + " - " + impFile.BookProperties.Title;
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace("/", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace(@"\", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace(":", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace("?", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace("*", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace("<", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace(">", " ");
            txtBP_RESFileName.Text = txtBP_RESFileName.Text.Replace("|", " ");
            txtBP_RESFileName.Text += ".RES";
        }
        #endregion

        #region btnSaveImage
        private void btnSaveImage_Click(object sender, EventArgs e)
        {
            IMGRunItem imgFile = (IMGRunItem)imageBox.Tag;

            string imageFile = imgFile.imageFileType + "__" + string.Format("{0:X4}", imgFile.imageID);

            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Bmp.Guid)
                imageFile += ".bmp";
            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Gif.Guid)
                imageFile += ".gif";
            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Jpeg.Guid)
                imageFile += ".jpg";
            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Png.Guid)
                imageFile += ".png";
            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Tiff.Guid)
                imageFile += ".tif";
            if (imageBox.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Wmf.Guid)
                imageFile += ".wmf";

            saveDlg.FileName = imageFile;
            saveDlg.Title = "Save Image As...";
            if (saveDlg.ShowDialog() == DialogResult.OK)
            {
                imageBox.Image.Save(saveDlg.FileName);
            }
        }
        #endregion

        #region btnGenerateBookID
        private void btnGenerateBookID_Click(object sender, EventArgs e)
        {
            txtBP_ID.Text = "ebook:guid=" + Guid.NewGuid().ToString().Replace("{", "").Replace("}", "").ToLower();
        }
        #endregion

        #region btnSaveText
        private void btnSaveText_Click(object sender, EventArgs e)
        {
            pnlIMPTreeProgress.Visible = false;
            txtIMPTreeText.Visible = false;

            string textFile = impFile.BookProperties.ResFile + ".txt";
            saveDlg.FileName = textFile;
            saveDlg.Title = "Save Text Content...";
            if (saveDlg.ShowDialog() == DialogResult.OK)
            {
                StreamWriter sw = new StreamWriter(saveDlg.FileName);
                sw.Write(txtContent.Text);
                sw.Close();
            }
        }
        #endregion

        #region btnFormatFML
        private void btnFormatFML_Click(object sender, EventArgs e)
        {
            if (txtBP_FirstName.Tag == null)
                txtBP_FirstName.Tag = txtBP_FirstName.Text;

            string author = txtBP_FirstName.Text;

            string[] names = author.Split(' ');

            if (names.Length == 2)
            {
                string f = names[0];
                string l = names[1];

                txtBP_FirstName.Text = l + ", " + f;
            }
            else if (names.Length > 2)
            {
                string f = names[0];
                string m = names[1];

                txtBP_FirstName.Text = "";

                for (int i = 2; i < names.Length; i++)
                    txtBP_FirstName.Text += names[i] + " ";

                txtBP_FirstName.Text = txtBP_FirstName.Text.Trim();
                txtBP_FirstName.Text += ", " + f + " " + m;
            }
        }
        #endregion

        #region btnUndo
        private void btnUndo_Click(object sender, EventArgs e)
        {
            if (txtBP_FirstName.Tag != null)
                txtBP_FirstName.Text = txtBP_FirstName.Tag.ToString();
        }
        #endregion

        #region mnuCreateHTML
        private void mnuCreateHTML_Click(object sender, EventArgs e)
        {
            CreateHTML();
        }
        #endregion

        #region btnSaveHTML
        private void btnSaveHTML_Click(object sender, EventArgs e)
        {
            pnlIMPTreeProgress.Visible = false;
            txtIMPTreeText.Visible = false;

            CreateHTML();
        }
        #endregion

        #region CreateHTML
        private void CreateHTML()
        {
            Cursor = Cursors.WaitCursor;
            Application.DoEvents();

            FileInfo fi = new FileInfo(impFile.IMPFilePath);

            string output = fi.DirectoryName + @"\" + impFile.BookProperties.Title + @"\";

            if (!Directory.Exists(output))
                Directory.CreateDirectory(output);

            foreach (IMGRunItem item in impFile.ImageRuns)
            {
                string imageFile = item.imageFileType + "__" + string.Format("{0:X4}", item.imageID);

                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Bmp.Guid)
                    imageFile += ".bmp";
                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Gif.Guid)
                    imageFile += ".gif";
                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Jpeg.Guid)
                    imageFile += ".jpg";
                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Png.Guid)
                    imageFile += ".png";
                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Tiff.Guid)
                    imageFile += ".tif";
                if (item.Image.RawFormat.Guid == System.Drawing.Imaging.ImageFormat.Wmf.Guid)
                    imageFile += ".wmf";

                item.Image.Save(output + @"\" + imageFile);

                item.imageFileName = output + @"\" + imageFile;
            }

            string html = "";

            html = "<!DOCTYPE html PUBLIC \"+//ISBN 0-9673008-1-9//DTD OEB 1.0.1 Document//EN\" \"http://openebook.org/dtds/oeb-1.0.1/oebdoc101.dtd\">\r\n\r\n";
            html += "<html><head><title>" + impFile.BookProperties.Title + "</title></head>\r\n";

            html += "<body>\r\n";
            html += impFile.ReverseResourceFileToHTML();
            html += "</body>\r\n";
            html += "</html>";

            StreamWriter sw = new StreamWriter(output + @"\" + impFile.BookProperties.ResFile + ".html");
            sw.Write(html);
            sw.Close();

            Cursor = Cursors.Default;
            Application.DoEvents();

            System.Diagnostics.Process.Start(output + @"\" + impFile.BookProperties.ResFile + ".html");
        }
        #endregion

        #region mnuAbout
        private void mnuAbout_Click(object sender, EventArgs e)
        {
            frmAbout frm = new frmAbout();
            frm.ShowDialog();
            frm.Dispose();
            frm = null;
        }
        #endregion

        #region btnBrowse
        private void btnBrowse_Click(object sender, EventArgs e)
        {
            if (folderDlg.ShowDialog() == DialogResult.OK)
            {
                txtIMPFolder.Text = folderDlg.SelectedPath;

                FolderNode.InnerText = txtIMPFolder.Text;

                IMPXMLDoc.Save(IMPXMLFilePath);

                cboLibraryCategories.SelectedItem = 0;
                LoadBooks(true);
            }
        }
        #endregion

        #region cboLibraryCategories_SelectedIndexChanged
        private void cboLibraryCategories_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (txtIMPFolder.Text != "")
                LoadBooks(false);
        }
        #endregion

        #region btnRefresh
        private void btnRefresh_Click(object sender, EventArgs e)
        {
            BooksNode.RemoveAll();

            LoadBooks(true);
        }
        #endregion

        #region cboBP_Category_SelectedIndexChanged
        private void cboBP_Category_SelectedIndexChanged(object sender, EventArgs e)
        {
            txtBP_Category.Text = cboBP_Category.SelectedItem.ToString();
        }
        #endregion

        #region lstBookShelf_ColumnClick
        private void lstBookShelf_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (e.Column == 0)
            {
                bSortAuthor = !bSortAuthor;
                lstBookShelf.ListViewItemSorter = new ListViewItemStringComparer(e.Column, bSortAuthor);
            }
            else if (e.Column == 1)
            {
                bSortTitle = !bSortTitle;
                lstBookShelf.ListViewItemSorter = new ListViewItemStringComparer(e.Column, bSortTitle);
            }
            else if (e.Column == 2)
            {
                bSortCategory = !bSortCategory;
                lstBookShelf.ListViewItemSorter = new ListViewItemStringComparer(e.Column, bSortCategory);
            }
            else if (e.Column == 3)
            {
                bSortIMPType = !bSortIMPType;
                lstBookShelf.ListViewItemSorter = new ListViewItemStringComparer(e.Column, bSortIMPType);
            }
            else if (e.Column == 4)
            {
                bSortSize = !bSortSize;
                lstBookShelf.ListViewItemSorter = new ListViewItemFileSizeComparer(e.Column, bSortSize);
            }
        }
        #endregion

        #region btnSaveLRF
        private void btnSaveLRF_Click(object sender, EventArgs e)
        {
            pnlIMPTreeProgress.Visible = false;
            txtIMPTreeText.Visible = false;

            Cursor = Cursors.WaitCursor;
            Application.DoEvents();

            FileInfo fi = new FileInfo(impFile.IMPFilePath);

            string output = fi.DirectoryName + @"\" + impFile.BookProperties.Title + @"\";

            if (!Directory.Exists(output))
                Directory.CreateDirectory(output);

            string textFile = output + impFile.BookProperties.ResFile + ".txt";
            StreamWriter sw = new StreamWriter(textFile);
            sw.Write(impFile.Content);
            sw.Close();

            BindingParams config = new BindingParams();
            config.MetaData = new BookMetaData();
            config.MetaData.BookInfo.Author = impFile.BookProperties.FirstName;
            config.MetaData.BookInfo.BookID = impFile.BookProperties.ID;
            config.MetaData.BookInfo.Category = impFile.BookProperties.Bookshelf_Category;
            config.MetaData.DocInfo.CreationDate = DateTime.Now.ToShortDateString();
            config.MetaData.DocInfo.Language = "en";
            config.MetaData.DocInfo.Creator = "ConvertIMPGUI";
            config.MetaData.DocInfo.Producer = "ConvertIMPGUI";

            LegacyBBeB book = new LegacyBBeB(config.MetaData);
            book.BeginBook(); 
            PseudoHtmlReader reader = new PseudoHtmlReader(book);
            reader.ReadFile(textFile);
            book.FinalizeBook(Application.StartupPath + @"\BookThumbnail.gif");
            book.WriteToFile(output + impFile.BookProperties.ResFile + ".lrf"); 

            Cursor = Cursors.Default;
            Application.DoEvents();

            System.Diagnostics.Process.Start(output + impFile.BookProperties.ResFile + ".lrf");
        }
        #endregion

        #region btnShowTreeText
        private void btnShowTreeText_Click(object sender, EventArgs e)
        {
            Cursor = Cursors.WaitCursor;

            lblIMPTreeTextProgress.Text = "";
            IMPTreeTextProgress.Value = 0;
            IMPTreeTextProgress.Minimum = 0;

            pnlIMPTreeProgress.Visible = true;
            txtIMPTreeText.Visible = true;

            Application.DoEvents();

            int nodeCount = CountNodes(IMPTree.Nodes);

            IMPTreeTextProgress.Maximum = nodeCount;
            Application.DoEvents();

            OutputTreeNode(IMPTree.Nodes);

            pnlIMPTreeProgress.Visible = false;

            Cursor = Cursors.Default;
            Application.DoEvents();
        }
        #endregion

        #region CountNodes
        private int CountNodes(TreeNodeCollection nodes)
        {
            int count = nodes.Count;

            foreach (TreeNode node in nodes)
                count += CountNodes(node.Nodes);

            return count;
        }
        #endregion

        #region OutputTreeNode
        private void OutputTreeNode(TreeNodeCollection nodes)
        {
            string line = "";

            foreach (TreeNode node in nodes)
            {
                IMPTreeTextProgress.Value = IMPTreeTextProgress.Value + 1;
                lblIMPTreeTextProgress.Text = IMPTreeTextProgress.Value + " of " + IMPTreeTextProgress.Maximum;
                Application.DoEvents();

                line = "";
                for (int i = 0; i < node.Level; i++)
                {
                    line += "  ";
                }

                line += node.Text + "\r\n";

                txtIMPTreeText.Text += line;

                OutputTreeNode(node.Nodes);
            }
        }
        #endregion

        private void mnuIMPTxtToLRF_Click(object sender, EventArgs e)
        {
            frmConvertIMPTxtToLRF frm = new frmConvertIMPTxtToLRF();
            frm.List = lstBookShelf.Items;
            frm.ShowDialog();
            frm.Dispose();
            frm = null;
        }
    }
}
