#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2013, Tobias Zeumer'
__docformat__ = 'restructuredtext en'

from PyQt5 import (QtGui, QtWidgets)
from PyQt5.Qt import (QLabel, QGridLayout, Qt, QGroupBox, QCheckBox, QHBoxLayout, QVBoxLayout, QLineEdit)
from PyQt6.QtWidgets import (QSpinBox)
from calibre.gui2 import question_dialog, error_dialog
from calibre.gui2.metadata.config import ConfigWidget as DefaultConfigWidget
#from calibre.gui2.preferences.ConfigWidgetBase import
from calibre.utils.config import JSONConfig

from collections import OrderedDict
from calibre_plugins.SRU.common_utils import KeyValueComboBox

# Define possible values
#   ['NAME', 'NAME (LONG)', 'SRU-BASE', 'CATALOG-SEARCH-URL', 'PRIMARY LANGUAGE', 'PICA'],
# LOC-Sru other url: http://z3950.loc.gov:7090/voyager; http://www.loc.gov/standards/sru/resources/lcServers.html
KEY_SOURCE = [
    ['GBV', 'Gemeinsamer Bibliotheksverbund', 'http://sru.k10plus.de/gvk', 'http://gso.gbv.de/DB=2.1/PPNSET?PPN=', 'ger', '1'],
    ['LOC', 'Library of Congress', 'http://z3950.loc.gov:7090/voyager', 'http://lccn.loc.gov/', 'eng', '']
]
CHOOSE_SOURCE = OrderedDict([('0', KEY_SOURCE[0][1]),
                           ('1', KEY_SOURCE[1][1])])

# Variables to use in plugin code, with defaults
STORE_NAME = 'Options'
KEY_MAX_DOWNLOADS = 'key_max_downloads'
KEY_SELECTED_SOURCE = 'key_selected_source'
KEY_SOURCE_NAME = 'key_source_name'
KEY_SOURCE_SRU = 'key_source_sru'
KEY_SOURCE_WEB = 'key_source_web'
KEY_SOURCE_PICA = 'key_source_pica'
KEY_TAG_GENRE  = 'key_tag_genre'
KEY_TAG_TOPIC1 = 'key_tag_topic1'
KEY_TAG_TOPIC2 = 'key_tag_topic2'

# Default (for reset)
DEFAULT_STORE_VALUES = {
    KEY_MAX_DOWNLOADS: '10',
    KEY_SELECTED_SOURCE: '0',
    KEY_SOURCE_NAME: KEY_SOURCE[0][0],
    KEY_SOURCE_SRU: KEY_SOURCE[0][2],
    KEY_SOURCE_WEB: KEY_SOURCE[0][3],
    KEY_SOURCE_PICA: bool(KEY_SOURCE[0][5]),
    KEY_TAG_GENRE: '',
    KEY_TAG_TOPIC1: '',
    KEY_TAG_TOPIC2: ''
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/SRU')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES


class ConfigWidget(DefaultConfigWidget):

    '''
    http://www.pythoncentral.io/pyside-pyqt-tutorial-interactive-widgets-and-layout-containers/?ModPagespeed=noscript
    '''
    def __init__(self, plugin):
        DefaultConfigWidget.__init__(self, plugin)
        c = plugin_prefs[STORE_NAME]

        # FIRST grid
        options_group_box = QGroupBox('Source Options', self)
        self.l.addWidget(options_group_box, self.l.rowCount(), 0, 1, 1)
        options_group_box_layout = QVBoxLayout()
        options_group_box.setLayout(options_group_box_layout)

        # chose DATA SOURCE - not active yet
        sourceSelect_layout = QHBoxLayout()
        options_group_box_layout.addLayout(sourceSelect_layout)
        sourceSelect_label = QLabel('Which source to query:', self)
        sourceSelect_label.setToolTip('GB = chose for german book, LOC for english books')
        sourceSelect_layout.addWidget(sourceSelect_label)
        self.sourceSelectCombo = KeyValueComboBox(self, CHOOSE_SOURCE, c.get(KEY_SELECTED_SOURCE, DEFAULT_STORE_VALUES[KEY_SELECTED_SOURCE]))
        sourceSelect_layout.addWidget(self.sourceSelectCombo)

        # SECOND grid
        options_group_grid = QGroupBox(_('Options'), self)
        self.l.addWidget(options_group_grid, self.l.rowCount(), 0, 2, 2)
        options_group_grid_layout = QGridLayout()
        options_group_grid.setLayout(options_group_grid_layout)

        # chose DOWNLOAD LIMIT
        max_label = QLabel('Maximum title/author search matches to evaluate (1 = fastest):', self)
        max_label.setToolTip('More slows things down. Your choice anyway.\n'
                             'Maybe no point doing otherwise...')
        options_group_grid_layout.addWidget(max_label, 0, 0, 1, 1)
        self.max_downloads_spin = QtWidgets.QSpinBox(self)
        self.max_downloads_spin.setMinimum(1)
        self.max_downloads_spin.setMaximum(20)
        self.max_downloads_spin.setProperty('value', c.get(KEY_MAX_DOWNLOADS, DEFAULT_STORE_VALUES[KEY_MAX_DOWNLOADS]))
        options_group_grid_layout.addWidget(self.max_downloads_spin, 0, 1, 1, 1)

        # Genre tag prefix
        tagGenre_label = QLabel('Use prefix for genre tags:', self)
        tagGenre_label.setToolTip('It might help to keep your tag collection clean')
        options_group_grid_layout.addWidget(tagGenre_label, 1, 0, 1, 1)
        self.tagGenre = QLineEdit(self)
        self.tagGenre.setText(c.get(KEY_TAG_GENRE, DEFAULT_STORE_VALUES[KEY_TAG_GENRE]))
        self.tagGenre.setPlaceholderText("e.g. 'g_'")
        options_group_grid_layout.addWidget(self.tagGenre, 1, 1, 1, 1)
        # Topic tag prefix (controlled vocabulary)
        tagTopic1_label = QLabel('Use prefix for topic tags from controlled vocabulary:', self)
        tagTopic1_label.setToolTip('It might help to keep your tag collection clean')
        options_group_grid_layout.addWidget(tagTopic1_label, 2, 0, 1, 1)
        self.tagTopic1 = QLineEdit(self)
        self.tagTopic1.setText(c.get(KEY_TAG_TOPIC1, DEFAULT_STORE_VALUES[KEY_TAG_TOPIC1]))
        self.tagTopic1.setPlaceholderText("e.g. 't1_'")
        options_group_grid_layout.addWidget(self.tagTopic1, 2, 1, 1, 1)
        # Topic tag prefix (free)
        tagTopic2_label = QLabel('Use prefix for freely assigned topic tags:', self)
        tagTopic2_label.setToolTip('It might help to keep your tag collection clean')
        options_group_grid_layout.addWidget(tagTopic2_label, 3, 0, 1, 1)
        self.tagTopic2 = QLineEdit(self)
        self.tagTopic2.setText(c.get(KEY_TAG_TOPIC2, DEFAULT_STORE_VALUES[KEY_TAG_TOPIC2]))
        self.tagTopic2.setPlaceholderText("e.g. 't2_'")
        options_group_grid_layout.addWidget(self.tagTopic2, 3, 1, 1, 1)

        # THIRD grid
        info_group_box = QGroupBox('Info, Errors & Requests', self)
        self.l.addWidget(info_group_box, self.l.rowCount(), 0, 1, 1)
        info_group_box_layout = QGridLayout()
        info_group_box.setLayout(info_group_box_layout)

        # Help and stuff
        txt_plugin=QLabel('At the forum of Mobileread.com, section Calibre->Plugins, there is a special <a href="www.mobileread.com/forums/showthread.php?p=2748641">thread</a> for SRU!')
        txt_plugin.setOpenExternalLinks(True)
        txt_sru=QLabel('Find out about <a href="http://www.loc.gov/standards/sru/index.html">SRU standard</a> over at the Library of Congress.', self)
        txt_sru.setOpenExternalLinks(True)
        txt_gbv=QLabel('Find out about the <a href="http://www.gbv.de/wikis/cls/SRU">implementation of SRU at the GVB</a> (german).', self)
        txt_gbv.setOpenExternalLinks(True)
        txt_marc=QLabel('Look up <a href="http://www.loc.gov/marc/bibliographic/">MARC field specifications</a>.', self)
        txt_marc.setOpenExternalLinks(True)
        txt_note=QLabel('Note: This Plugin requests MarcXML. Alternatively Dublin Core (DC) could be requested, which is easier to parse, but offers less information.', self)

        info_group_box_layout.addWidget(txt_plugin)
        info_group_box_layout.addWidget(txt_sru)
        info_group_box_layout.addWidget(txt_gbv)
        info_group_box_layout.addWidget(txt_marc)
        info_group_box_layout.addWidget(txt_note)

        #reset to defaults
        reset_defaults_button = QtWidgets.QToolButton(self)
        reset_defaults_button.setToolTip('Resets fields to plugin default settings. You have to save afterwards.')
        reset_defaults_button.setText('Reset settings to default')
        reset_defaults_button.clicked.connect(self.reset_to_defaults)
        info_group_box_layout.addWidget(reset_defaults_button)


    '''
    Save settings
    NOTE: Don't forget to change reset_to_defaults too
    '''
    def commit(self):
        DefaultConfigWidget.commit(self)
        new_prefs = {}
        # Fetch from Gui-"Spin"
        new_prefs[KEY_MAX_DOWNLOADS] = str(self.max_downloads_spin.value())
        # Fetch from Source Select drop down
        source_index = int(self.sourceSelectCombo.selected_key())

        # Set source values as new default
        new_prefs[KEY_SELECTED_SOURCE] = str(source_index)
        new_prefs[KEY_SOURCE_NAME] = KEY_SOURCE[source_index][0]
        new_prefs[KEY_SOURCE_SRU] = KEY_SOURCE[source_index][2]
        new_prefs[KEY_SOURCE_WEB] = KEY_SOURCE[source_index][3]
        new_prefs[KEY_SOURCE_PICA] = bool(KEY_SOURCE[source_index][5])

        # Set tag values as new default
        new_prefs[KEY_TAG_GENRE]  = str(self.tagGenre.text())
        new_prefs[KEY_TAG_TOPIC1] = str(self.tagTopic1.text())
        new_prefs[KEY_TAG_TOPIC2] = str(self.tagTopic2.text())

        #save
        plugin_prefs[STORE_NAME] = new_prefs


    '''
    Restore default settings
    '''
    def reset_to_defaults(self):
        if not question_dialog(self, _('Are you sure?'), '<p>Are you sure you want to reset to the plugin default settings?', show_copy_button=False):
            return
        plugin_prefs[STORE_NAME] = DEFAULT_STORE_VALUES
        c = plugin_prefs[STORE_NAME]
        # Max titles to download
        self.max_downloads_spin.setProperty('value', int(c.get(KEY_SELECTED_SOURCE, DEFAULT_STORE_VALUES[KEY_SELECTED_SOURCE])))
        # Source to use
        self.sourceSelectCombo = KeyValueComboBox(self, CHOOSE_SOURCE, c.get(KEY_SELECTED_SOURCE, DEFAULT_STORE_VALUES[KEY_SELECTED_SOURCE]))
        # Tag prefixes
        self.tagGenre.setText(c.get(KEY_TAG_GENRE, DEFAULT_STORE_VALUES[KEY_TAG_GENRE]))
        self.tagTopic1.setText(c.get(KEY_TAG_TOPIC1, DEFAULT_STORE_VALUES[KEY_TAG_TOPIC1]))
        self.tagTopic2.setText(c.get(KEY_TAG_TOPIC2, DEFAULT_STORE_VALUES[KEY_TAG_TOPIC2]))
        
