import os
import regex
import copy

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.base import OEB_DOCS, OEB_STYLES
from calibre.gui2.tweak_book import editors, editor_name

from calibre_plugins.editor_chains.common_utils import reverse_lookup

from qt.core import (
    QApplication, Qt, QVBoxLayout, QHBoxLayout, QWidget, QComboBox,
    QFont, QLineEdit, QLabel
)

def get_nav(container):
    for item in container.opf_xpath(f'//opf:manifest/opf:item[@href and @id="nav"]'):
        return item.attrib['href']

def scope_names(chain, where, exclude_nav=True, include_nonsearchable=False):
    container = chain.current_container
    searchable_names = []
    text_files = []
    style_files = []
    all_files = []
    for name, media_type in container.mime_map.items():
        if media_type in OEB_DOCS:
            text_files.append(name)
        elif media_type in OEB_STYLES:
            style_files.append(name)
        elif os.path.splitext(name.lower())[1].lstrip('.') in ('txt', 'xml', 'opf', 'ncx'):
            searchable_names.append(name)
        all_files.append(name)
    searchable_names += text_files + style_files

    filenames = []
    if not isinstance(where, str):
        where, extra = where
    if where == 'current':
        name = editor_name(chain.gui.central.current_editor)
        if not name or container.mime_map[name] not in OEB_DOCS:
            if DEBUG:
                prints(_('No file open for editing or the current file is not an (x)html file.'))
        else:
            filenames = [name]
    elif where == 'text':
        filenames = list(container.manifest_items_of_type(OEB_DOCS))
        if exclude_nav:
            nav = get_nav(container)
            if nav:
                if nav in filenames:
                    filenames.remove(nav)
    elif where == 'styles':
        filenames = list(container.manifest_items_of_type(OEB_STYLES))
    #elif where in {'selected', 'open'}:
        #if include_nonsearchable:
            #filenames = list(chain.gui.file_list.file_list.selected_names)
        #else:
            #sn = chain.gui.file_list.searchable_names
            #filenames = list(sn[where].keys())
    elif where == 'selected':
        filenames = list(chain.gui.file_list.file_list.selected_names)
        if not include_nonsearchable:
            filenames = list(set(filenames).intersection(set(searchable_names)))
    elif where == 'open':
        filenames = list(name for name in editors)
        if not include_nonsearchable:
            filenames = list(set(filenames).intersection(set(searchable_names)))
    #elif where == 'selected-text':
        ## marked text
        #current_editor = chain.gui.central.current_editor
        #name = editor_name(chain.gui.central.current_editor)
        #filenames = [name]
##
    elif where == 'regex':
        filenames = []
        include_patterns = [x.strip() for x in extra.get('include', '').split(',')]
        names_to_search_in = searchable_names
        if include_nonsearchable:
            names_to_search_in = all_files
        for pat in include_patterns:
            for name in names_to_search_in:
                if regex.search(pat, name):
                    filenames.append(name)

        exclude_string = extra.get('exclude', '').strip()
        if exclude_string:
            exclude_patterns = [x.strip() for x in exclude_string.split(',')]
            for pat in exclude_patterns:
                for name in copy.copy(filenames):
                    if regex.search(pat, name):
                        try:
                            filenames.remove(name)
                        except:
                            pass

    elif where == 'text-regex':
        filenames = []
        include_pattern = extra.get('include', '')
        if include_pattern:
            for name in searchable_names:
                data = container.open(name, 'r').read()
                if regex.search(include_pattern, data):
                    filenames.append(name)
        else:
            filenames = searchable_names

        exclude_pattern = extra.get('exclude', '').strip()
        if exclude_pattern:
            for name in copy.copy(filenames):
                data = container.open(name, 'r').read()
                if regex.search(exclude_pattern, data):
                    try:
                        filenames.remove(name)
                    except:
                        pass

    return filenames

def validate_scope(where):
    if not isinstance(where, str):
        where, extra = where
    if where == 'regex':
        for k in ['exclude','include']:
            patterns = [x.strip() for x in extra.get(k, '').split(',')]
            for pat in patterns:
                try:
                    regex.search(pat, 'random text')
                except regex._regex_core.error:
                    return _('Invalid regex'), _(f'Pattern ({pat}) is not a valid regex')
    if where == 'text-regex':
        for k in ['exclude','include']:
            pattern = extra.get(k, '')
            try:
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
    return True

def scope_is_headless(where):
    if where in ['current', 'selected', 'open', 'selected-text']:
        return _('Scope Error'), _(f'Scope ({where}) cannot run in headless mode.')
    return True

scope_dict = {
    'current': _('Current file'),
    'text': _('All text files'),
    'styles': _('All style files'),
    'selected': _('Selected files'),
    'open': _('Open files'),
    'selected-text': _('Marked text'),
    'regex': _('Filenames'),
    'text-regex': _('Text search')
}


class RegExFilter(QWidget):
    def __init__(self, keyword='include', tooltip=_('Comma separated list of filenames.')):
        QWidget.__init__(self)
        l = QHBoxLayout()
        self.setLayout(l)
        lbl = QLabel(_(f'{keyword.title()} filters'))
        self.ledit = QLineEdit()
        #self.ledit.setToolTip(tooltip)
        l.addWidget(lbl)
        l.addWidget(self.ledit)

    def set_tooltip(self, tooltip):
        self.ledit.setToolTip(tooltip)

    def text(self):
        return self.ledit.text()

    def setText(self, val):
        return self.ledit.setText(val)

class WhereBox(QComboBox):

    def __init__(self, parent, emphasize=False, default='text', hide=['selected-text'],
                 show_tooltip=True):
        QComboBox.__init__(self)
        vals = [v for v in scope_dict.values() if reverse_lookup(scope_dict, v) not in hide]
        self.addItems(vals)
        if show_tooltip:
            self.setToolTip('<style>dd {margin-bottom: 1.5ex}</style>' + _(
                '''
                Which files to perform this action on:
                <dl>
                <dt><b>Current file</b></dt>
                <dd>Search only inside the currently opened file</dd>
                <dt><b>All text files</b></dt>
                <dd>Search in all text (HTML) files</dd>
                <dt><b>All style files</b></dt>
                <dd>Search in all style (CSS) files</dd>
                <dt><b>Selected files</b></dt>
                <dd>Search in the files currently selected in the File browser</dd>
                <dt><b>Open files</b></dt>
                <dd>Search in the files currently open in the editor</dd>
                <dt><b>Marked text</b></dt>
                <dd>Search only within the marked text in the currently opened file. You can mark text using the Search menu.</dd>
                <dt><b>Filename regex</b></dt>
                <dd>Comma separated list of filenames to include or exclude (regex).</dd>
                <dt><b>Text file regex</b></dt>
                <dd>Match text files containing the supplied regex</dd>
                </dl>'''))
        self.emphasize = emphasize
        self.ofont = QFont(self.font())
        if emphasize:
            f = self.emph_font = QFont(self.ofont)
            f.setBold(True), f.setItalic(True)
            self.setFont(f)
        self.setCurrentText(scope_dict[default])

    @property
    def where(self):
        return reverse_lookup(scope_dict, self.currentText())

    @where.setter
    def where(self, val):
        self.setCurrentText(scope_dict[val])

    def showPopup(self):
        # We do it like this so that the popup uses a normal font
        if self.emphasize:
            self.setFont(self.ofont)
        QComboBox.showPopup(self)

    def hidePopup(self):
        if self.emphasize:
            self.setFont(self.emph_font)
        QComboBox.hidePopup(self)


class ScopeWidget(QWidget):

    def __init__(self, parent, plugin_action, default='text',
                 orientation='vertical', hide=['selected-text'],
                 show_tooltip=True, headless=False):
        if headless:
            for w in ['current', 'selected', 'open']:
                if not w in hide:
                    hide.append(w)
        self.plugin_action = plugin_action
        self.orientation = orientation
        self.default = default
        self.hide = hide
        self.show_tooltip = show_tooltip
        QWidget.__init__(self)
        self.setup_ui()

    def setup_ui(self):
        if self.orientation == 'horizontal':
            l = QHBoxLayout()
        else:
            l = QVBoxLayout()
        self.setLayout(l)

        self.where_box = WhereBox(self, default=self.default, hide=self.hide, show_tooltip=self.show_tooltip)
        l.addWidget(self.where_box)
        self.include_filter = RegExFilter(keyword='include')
        l.addWidget(self.include_filter)
        self.include_filter.setVisible(False)
        self.exclude_filter = RegExFilter(keyword='exclude')
        l.addWidget(self.exclude_filter)
        self.exclude_filter.setVisible(False)
        self.where_box.currentTextChanged.connect(self._on_current_text_changed)
        
    def _on_current_text_changed(self, text):
        if text in [scope_dict[x] for x in ['regex']]:
            self.include_filter.set_tooltip(_('Comma separated list of filenames to include'))
            self.exclude_filter.set_tooltip(_('Comma separated list of filenames to exclude'))
        elif text in [scope_dict[x] for x in ['text-regex']]:
            self.include_filter.set_tooltip(_('Include text files containing the supplied regex'))
            self.exclude_filter.set_tooltip(_('Exclude text files containing the supplied regex'))
        else:
            tooltip = ''
        self.include_filter.setVisible(text in [scope_dict[x] for x in ['regex','text-regex']])
        self.exclude_filter.setVisible(text in [scope_dict[x] for x in ['regex','text-regex']])

    @property
    def where(self):
        val = reverse_lookup(scope_dict, self.where_box.currentText())
        if val in ['regex','text-regex']:
            val = (val, {'include': self.include_filter.text(), 'exclude': self.exclude_filter.text()})
        return val

    @where.setter
    def where(self, val):
        extra = ''
        if isinstance(val, str):
            text = val
        else:
            text, extra = val
            self.include_filter.setText(extra.get('include', ''))
            self.exclude_filter.setText(extra.get('exclude', ''))
        self.where_box.setCurrentText(scope_dict[text])
