#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

#    The following is based on code from calibre GPL3 Copyright: Kovid Goyal <kovid at kovidgoyal.net>:
#    - compile_code
#    - Function > Module
#    - functions > modules
#    - remove_functions > remove_modules
#    - FunctionEditor > ModuleEditor

import re, io, sys
import inspect
from collections import OrderedDict

from qt.core import (pyqtSignal, QGridLayout, QVBoxLayout, QHBoxLayout, QPlainTextEdit,
                     QLabel, QFontMetrics, QSize, Qt, QApplication, QIcon, QPushButton,
                     QListWidget, QListWidgetItem, QDialogButtonBox, QAbstractItemView)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.complete2 import EditWithComplete
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2.tweak_book.widgets import Dialog
from calibre.gui2.tweak_book.editor.text import TextEdit
from calibre.utils.config import JSONConfig

from calibre_plugins.editor_chains.common_utils import get_icon, has_non_inherited_attribute

try:
    load_translations()
except NameError:
    prints("EditorChains::modules.py - exception when loading translations")

module_sources = JSONConfig('plugins/editor_chains/editor-chains-user-modules')

def compile_code(src, name='<string>'):
    if not isinstance(src, str):
        match = re.search(br'coding[:=]\s*([-\w.]+)', src[:200])
        enc = match.group(1).decode('utf-8') if match else 'utf-8'
        src = src.decode(enc)
    if not src or not src.strip():
        src = ''
    # Python complains if there is a coding declaration in a unicode string
    src = re.sub(r'^#.*coding\s*[:=]\s*([-\w.]+)', '#', src, flags=re.MULTILINE)
    # Translate newlines to \n
    src = io.StringIO(src, newline=None).getvalue()
    code = compile(src, name, 'exec')

    namespace = {}
    exec(code, namespace)
    return namespace

class Module(object):

    def __init__(self, name, source):
        self._source = source
        self.name = name
        self.namespace = compile_code(source, name)

    @property
    def source(self):
        return self._source

    def __hash__(self):
        return hash(self.name)

    def __eq__(self, other):
        return self.name == getattr(other, 'name', None)

    def __ne__(self, other):
        return not self.__eq__(other)


_modules = None


def modules(refresh=False):
    global _modules
    if _modules is None or refresh:
        ans = _modules = {}
        for name, source in module_sources.items():
            try:
                m = Module(name, source)
            except Exception as e:
                if DEBUG:
                    prints(f'Editor Chains: Module ({name}) contains errors, not loaded')
                continue
            ans[m.name] = m
    return _modules


def remove_module(name, gui_parent=None):
    mods = modules()
    if not name:
        return False
    if name not in mods:
        error_dialog(gui_parent, _('No such module'), _(
            'There is no module named %s') % name, show=True)
        return False
    if name not in module_sources:
        error_dialog(gui_parent, _('Cannot remove builtin module'), _(
            'The module %s is a module module, it cannot be removed.') % name, show=True)
    del module_sources[name]
    modules(refresh=True)
    return True

class ModuleEditor(Dialog):

    CLASS_TEMPLATE = "\nfrom calibre_plugins.editor_chains.actions.base import EditorAction\n\nclass MyAction(EditorAction):\n\n    # replace with the name of your action\n    name = 'My Action'\n\n    def run(self, chain, settings, *args, **kwargs):\n        pass\n\n"

    def __init__(self, mod_name='', parent=None):
        self._mod_name = mod_name
        Dialog.__init__(self, _('Create/edit a module'), 'editor-chains-edit-mods', parent=parent)

    def setup_ui(self):
        self.l = l = QVBoxLayout(self)
        self.h = h = QHBoxLayout()
        l.addLayout(h)

        self.la1 = la = QLabel(_('M&odule name:'))
        h.addWidget(la)
        self.fb = fb = EditWithComplete(self)
        la.setBuddy(fb)
        h.addWidget(fb, stretch=10)

        self.la3 = la = QLabel(_('&Code:'))
        self.source_code = TextEdit(self)
        self.source_code.load_text('', 'python')
        la.setBuddy(self.source_code)
        l.addWidget(la), l.addWidget(self.source_code)

        if self._mod_name:
            self.fb.setText(self._mod_name)
            self.source_code.setPlainText(module_sources.get(self._mod_name) or self.CLASS_TEMPLATE)
        else:
            self.source_code.setPlainText(self.CLASS_TEMPLATE)

        self.la2 = la = QLabel()
        la.setOpenExternalLinks(True)
        l.addWidget(la)

        l.addWidget(self.bb)

    def sizeHint(self):
        fm = QFontMetrics(self.font())
        return QSize(fm.averageCharWidth() * 120, 600)

    @property
    def mod_name(self):
        return self.fb.text().strip()

    @property
    def source(self):
        return self.source_code.toPlainText()

    def accept(self):
        if not self.mod_name:
            return error_dialog(self, _('Must specify name'), _(
                'You must specify a name for this module.'), show=True)
        source = self.source
        try:
            mod = compile_code(source, self.mod_name)
        except Exception as err:
            return error_dialog(self, _('Invalid Python code'), _(
                'The code you created is not valid Python code, with error: %s') % err, show=True)
        module_sources[self.mod_name] = source
        modules(refresh=True)

        Dialog.accept(self)

class ManageModulesDialog(Dialog): 
    def __init__(self, parent, plugin_action):
        self.plugin_action = plugin_action
        Dialog.__init__(self, _('Manage modules'), 'editor-chains-manage-modules', parent)

    def setup_ui(self):
        layout = QGridLayout()
        self.setLayout(layout)
        
        self.modules_list = QListWidget(self)
        self.modules_list.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.modules_list.setAlternatingRowColors(True)
        layout.addWidget(self.modules_list, 0, 0, 1, 1)

        actl = QVBoxLayout()
        layout.addLayout(actl, 0, 1, 1, 1)

        self.create_button = QPushButton(get_icon('document-new.png'), _('Create'), self)
        self.create_button.clicked.connect(self._create_module)
        self.edit_button = QPushButton(get_icon('edit-undo.png'), _('Edit'), self)
        self.edit_button.clicked.connect(self._edit_selected_module)
        self.delete_button = QPushButton(get_icon('trash.png'), _('Delete'), self)
        self.delete_button.clicked.connect(self._delete_module)

        actl.addWidget(self.create_button)
        actl.addWidget(self.edit_button)
        actl.addWidget(self.delete_button)

        actl.addStretch(1)

        # delete Dialog inserted self.bb
        del self.bb

        self.button_box = QDialogButtonBox(QDialogButtonBox.Close)
        self.button_box.rejected.connect(self.reject)
        layout.addWidget(self.button_box, 1, 0, 1, 2)

        self._populate_list(select_index=0)
        self.modules_list.itemSelectionChanged.connect(self._selection_changed)
        self.modules_list.itemDoubleClicked.connect(self._edit_module)
        
        self._selection_changed()

    def _selection_changed(self):
        items_count = len(self.modules_list.selectedItems())
        self.delete_button.setEnabled(items_count > 0)
        self.edit_button.setEnabled(items_count == 1)

    def _populate_list(self, select_index=None):
        self.modules_list.clear()
        skeys = sorted(list(module_sources.keys()))
        
        for mod_name in skeys:
            item = QListWidgetItem(mod_name, self.modules_list)
            item.setIcon(get_icon('edit-paste.png'))

        count = len(module_sources)
        if (select_index in range(count)) and count:
            self.modules_list.setCurrentRow(select_index)

    def _delete_module(self):        
        selected_items = self.modules_list.selectedItems()
        current_row =  self.modules_list.currentRow()
        message = _(f'<p>Are you sure you want to delete the {len(selected_items)} selected module(s)?</p>')
        if not confirm(message, 'confirm_delete_modules', self):
            return
        for item in reversed(selected_items):
            mod_name = item.text()
            row = self.modules_list.row(item)
            self.modules_list.takeItem(row)
            remove_module(mod_name)
        
        count = len(module_sources)
        if (current_row in range(count)) and count:
            self.modules_list.setCurrentRow(current_row)
        self.plugin_action.on_modules_update()

    def _edit_module(self, item):
        selected_name = item.text()
        
        d = ModuleEditor(selected_name, self)
        if d.exec_() == d.Accepted:
            if d.mod_name != selected_name:
                remove_module(selected_name)
            index = sorted(module_sources).index(d.mod_name)
            self._populate_list(select_index=index)
            self.plugin_action.on_modules_update()

    def _edit_selected_module(self):
        selected_item = self.modules_list.selectedItems()[0]
        self._edit_module(selected_item)

    def _create_module(self):        
        d = ModuleEditor('', self)
        if d.exec_() == d.Accepted:
            index = sorted(module_sources).index(d.mod_name)
            self._populate_list(select_index=index)
            self.plugin_action.on_modules_update()

    def reject(self):
        Dialog.reject(self)

class UserModules(object):
    def __init__(self):
        self._modules = modules()

    def get_objects(self, module_filters=[], type_filters=[]):
        all_objects = []
        for module_name, module in self._modules.items():
            if module_filters and not self.apply_filters(module_name, module_filters):
                continue
            compiled_module = module.namespace
            for obj in compiled_module.values():
                if type_filters and not isinstance(obj, tuple(type_filters)):
                    continue

                # exclude classes and instances which have below attributes set to True
                # do not propagete the exclusion to subclasses or instances of subclasses.
                exclude = False
                for attribute in ['_is_builtin', 'exclude_from_modules']:
                    if has_non_inherited_attribute(obj, attribute):
                        if getattr(obj, attribute):
                            exclude = True
                            break
                if exclude:
                    continue      
                all_objects.append(obj)
        return all_objects

    def get_classes(self, module_filters=[], class_filters=[]):
        all_classes = []
        all_objects = self.get_objects(module_filters=module_filters)
        for obj in all_objects:
            if not inspect.isclass(obj):
                continue
            if class_filters and not issubclass(obj, tuple(class_filters)):
                continue
            all_classes.append(obj)
        return all_classes

    def apply_filter(self, obj, filter_):
        if callable(filter_):
            try:
                return filter_(obj)
            except Exception as e:
                if DEBUG:
                    prints(f'Editor Chains: Error applying callable filter: {e}')
                return False
        elif isinstance(filter_, type(re.compile('Hello'))):
            return bool(filter_.match(obj))
        else:
            return obj == filter_

    def apply_filters(self, obj, filters):
        for filter_ in filters:
            if self.apply_filter(obj, filter_):
                return True
        return False

    @property
    def module_names(self):
        return self._modules.keys()

    def get_namespace(self, module_name):
        module = self._modules.get(module_name)
        if module:
            return module.namespace
