#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from collections import defaultdict
from functools import partial

from qt.core import Qt, QApplication, QAction, QMenu, QToolButton, QUrl, pyqtSignal, QTimer, QIcon

# The base class that all tools must inherit from
from calibre.gui2.tweak_book.plugin import Tool

from calibre import prints
from calibre.constants import iswindows, isosx, DEBUG
from calibre.gui2 import open_url, error_dialog
from calibre.utils.config import config_dir
from calibre.utils.date import now
from calibre.utils.config import tweaks

import calibre_plugins.editor_chains.config as cfg
from calibre_plugins.editor_chains.action import EditorChainsPluginAction
from calibre_plugins.editor_chains.common_utils import (set_plugin_icon_resources, get_icon)
from calibre_plugins.editor_chains.modules import ManageModulesDialog, UserModules
from calibre_plugins.editor_chains.gui import IconCache
from calibre_plugins.editor_chains.gui.chains_dialog import ChainsDialog

try:
    load_translations()
except NameError:
    prints("EditorChains::main.py - exception when loading translations")

UNIQUE_NAME_PREFIX = 'editor-chains_'

try:
    load_translations()
except NameError:
    prints("EditorChains::main.py - exception when loading translations")

def topmost_menu(submenu):
    menu = submenu.parent()
    if not isinstance(menu, QMenu):
        return submenu
    else:
        while isinstance(menu.parent(), QMenu):
            menu = menu.parent()
    return menu

class EditChainsTool(Tool):

    #: Set this to a unique name it will be used as a key
    name = 'editor-chains'

    #: If True the user can choose to place this tool in the plugins toolbar
    allowed_in_toolbar = True

    #: If True the user can choose to place this tool in the plugins menu
    allowed_in_menu = True

    def create_action(self, for_toolbar=True):
        # Create an action, this will be added to the plugins toolbar and
        # the plugins menu
        self.genesis(for_toolbar=for_toolbar)
        ac = QAction(get_icons('images/editor_chains.png'), 'Editor Chains', self.gui)  # noqa
        if not for_toolbar:
            # Register a keyboard shortcut for this toolbar action. We only
            # register it for the action created for the menu, not the toolbar,
            # to avoid a double trigger
            #self.register_shortcut(ac, 'editor-chains')
            pass
        if for_toolbar:
            ac.setMenu(self.toolbar_menu)
        else:
            ac.setMenu(self.menu)
        return ac

    def genesis(self, for_toolbar=True):
        # This method will be run twice (once for dropdown menu and once for toolbar),
        # so make sure not to override attributes from previous run.
        if not hasattr(self, 'plugin_action'):           
            self.resources_dir = os.path.join(config_dir, 'resources/images')
            if iswindows:
                self.resources_dir = os.path.normpath(self.resources_dir)
            self.icon_cache = IconCache(self)
            self.plugin_action = EditorChainsPluginAction(gui=self.gui, boss=self.boss,
                                                          tool=self,
                                                          resources_dir=self.resources_dir,
                                                          icon_cache=self.icon_cache)
            #
            # these variables are used to block events when chains and events are running
            self.chainStack = []
            # counter for how many times a chain has run
            self.chain_iterations = defaultdict(lambda: 0)

            self.plugin_action.all_menus = []

        if for_toolbar:
            menu = self.toolbar_menu = QMenu(self.gui)
            menu.for_toolbar = True
        else:
            menu = self.menu = QMenu(self.gui)
            menu.for_toolbar = False

        self.plugin_action.all_menus.append(menu)

        # Setup hooks so that we only enable the relevant submenus for available formats for the selection.
        menu.aboutToShow.connect(partial(self.about_to_show_menu, menu))
        menu.aboutToHide.connect(partial(self.about_to_hide_menu, menu))
        
        menu.menu_actions = []
        self.rebuild_menus(menu)

    def rebuild_menus(self, menu):
        c = cfg.plugin_prefs[cfg.STORE_MENUS_NAME]
        chains_config = c.get(cfg.KEY_MENUS, [])

        menu.clear()
        menu.chain_menu = []
        sub_menus = {}

        if not menu.for_toolbar:
            for action in menu.menu_actions:
                self.gui.keyboard.unregister_shortcut(UNIQUE_NAME_PREFIX+action.unique_name)
                # starting in calibre 2.10.0, actions are registers at
                # the top gui level for OSX' benefit.
                self.gui.removeAction(action)
            menu.menu_actions = []

        for chain_config in chains_config:
            active = chain_config['active']
            if active:
                menu_text = chain_config['menuText']
                sub_menu_text = chain_config['subMenu']
                image_name = chain_config['image']
                self.create_menu_item_ex(menu, sub_menus, menu_text, sub_menu_text,
                                         image_name, chain_config)

        menu.addSeparator()
        
        self.create_menu_action_unique(menu, _('&Manage Modules')+'...', 'manage-modules', 'code.png',
                                  triggered=self.manage_modules)        
        
        self.create_menu_action_unique(menu, _('&Add/Modify Chains')+'...', 'add-chains', 'images/editor_chains.png',
                                  triggered=self.add_chains)
        
#        menu.addSeparator()
#        self.create_menu_action_unique(menu, _('&Customize plugin')+'...', 'editor-chains-show-config', 'config.png',
#                                  shortcut=False, triggered=self.show_configuration)
        
        # Hide submenus that contain certain keywords
        for path, submenu in sub_menus.items():
            name = path.split(':::')[-1].lower().strip()
            if name in ['invisible']:
                submenu.menuAction().setVisible(False)
       
        self.gui.keyboard.finalize()

    def register_shortcut(self, qaction, unique_name, default_keys=(), short_text=None, description=None, group=_('Plugins'), **extra_data):
        '''
        Register a keyboard shortcut that will trigger the specified ``qaction``. This keyboard shortcut
        will become automatically customizable by the user in the Keyboard shortcuts section of the editor preferences.

        :param qaction: A QAction object, it will be triggered when the
            configured key combination is pressed by the user.
        :param unique_name: A unique name for this shortcut/action. It will be
            used internally, it must not be shared by any other actions in this
            plugin.
        :param default_keys: A list of the default keyboard shortcuts. If not
            specified no default shortcuts will be set. If the shortcuts specified
            here conflict with either builtin shortcuts or shortcuts from user
            configuration/other plugins, they will be ignored. In that case, users
            will have to configure the shortcuts manually via Preferences. For example:
            ``default_keys=('Ctrl+J', 'F9')``.
        :param short_text: An optional short description of this action. If not
            specified the text from the QAction will be used.
        :param description: An optional longer description of this action, it
            will be used in the preferences entry for this shortcut.
        '''
        short_text = short_text or str(qaction.text()).replace('&&', '\0').replace('&', '').replace('\0', '&')
        self.gui.keyboard.register_shortcut(
            self.name + '_' + unique_name, short_text, default_keys=default_keys, action=qaction,
            description=description or '', group=group)

    def create_menu_action_unique(self, menu, text, unique_name, icon,
                                 shortcut=False, triggered=None, is_checked=None):
        ac = QAction(text, self.gui)
        if icon:
            ac.setIcon(get_icon(icon))
        if triggered is not None:
            ac.triggered.connect(triggered)
        if is_checked is not None:
            ac.setCheckable(True)
            if is_checked:
                ac.setChecked(True)
        if not menu.for_toolbar:
            if shortcut:
                self.register_shortcut(ac, unique_name, default_keys=(shortcut,), group=_('Editor Chains'))
            else:
                self.register_shortcut(ac, unique_name, group=_('Editor Chains'))
        topmost = topmost_menu(menu)
        menu.addAction(ac)
        ac.unique_name = unique_name
        topmost.menu_actions.append(ac)
        return ac   

    def create_menu_item_ex(self, m, sub_menus, menu_text, sub_menu_text,
                            image_name, chain_config):
        parent_menu = m
        if sub_menu_text:
            parent_menu = self.create_sub_menus(parent_menu, sub_menus, sub_menu_text, image_name)

        if not menu_text:
            ac = parent_menu.addSeparator()
        else:
            ac = self.create_chain_menu_action(chain_config, menu_text, parent_menu, menu_text, image_name)
        return ac

    def create_sub_menus(self, parent_menu, sub_menus, sub_menu_text, image_name):
        # Create the sub-menu(s) if it does not exist
        # Multiple sub-menus can be nested by using ':::'. e.g sub_menu_text == 'level1:::level2:::level3'
        tokens = sub_menu_text.split(':::')
        for idx in range(len(tokens)):
            sub_menu_path = ':::'.join(tokens[:idx+1])
            if sub_menu_path not in sub_menus:
                icon = self.get_sub_menu_icon(sub_menu_path, image_name)
                sm = parent_menu.addMenu(QIcon(icon), tokens[idx])
                sm.for_toolbar = parent_menu.for_toolbar
                sub_menus[sub_menu_path] = sm
            # Now set our menu variable so the parent menu item will be the sub-menu
            parent_menu = sub_menus[sub_menu_path]
        return parent_menu

    def get_sub_menu_icon(self, sub_menu_path, first_action_icon):
        if tweaks.get('editor_chains_default_sub_menu_icon', 'first_action') == 'first_action':
            return first_action_icon
        return None

    def create_chain_menu_action(self, chain_config, unique_name, parent_menu, menu_text, image_name):
        ac = self.create_menu_action_unique(parent_menu, menu_text, unique_name,image_name,
                       triggered=partial(self.plugin_action.run_chain, chain_config, True, True))
        # Maintain our list of menus by chain references so we can easily enable/disable menus when user right-clicks.
        topmost = topmost_menu(parent_menu)
        topmost.menu_actions.append(ac)
        topmost.chain_menu.append((chain_config, ac))
        return ac

    def about_to_show_menu(self, menu):
        chain_menu = menu.chain_menu
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            for chain_config, menu_action in chain_menu:
                is_enabled = self.is_menu_enabled(chain_config)
                menu_action.setEnabled(is_enabled)
        finally:
            QApplication.restoreOverrideCursor()

    def about_to_hide_menu(self, menu):
        # When hiding menus we must re-enable all selections in case a shortcut key for the
        # action gets pressed after moving to a new row.
        self.set_enabled_for_all_menu_actions(True, menu)

    def set_enabled_for_all_menu_actions(self, is_enabled, menu):
        chain_menu = menu.chain_menu
        for chain_config, menu_action in chain_menu:
            menu_action.setEnabled(is_enabled)

    def is_menu_enabled(self, chain_config):
        '''
        Determine whether menu item for the chain is enabled or not
        '''
        return True

    def manage_modules(self):
        d = ManageModulesDialog(self.gui, self.plugin_action)
        d.exec_()

    def add_chains(self):
        d = ChainsDialog(self.plugin_action)
        d.exec_()
