#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, sys
import shutil
import argparse

from qt.core import QApplication

from calibre import prints
from calibre.constants import DEBUG
from calibre.ptempfile import better_mktemp

from calibre_plugins.editor_chains.action import EditorChainsPluginAction
from calibre_plugins.editor_chains.chains import Chain
import calibre_plugins.editor_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("EditorChains::headless.py - exception when loading translations")

editor_chains_action = EditorChainsPluginAction()

class EditorWrapper(object):
    def __init__(self):
        self.plugin_action = editor_chains_action

    def run_editor_chain(
        self,
        chain_config,
        book_path,
        book_id=None,
        chain_vars={},
        debug_string=None
    ):
        chain = Chain(
            self.plugin_action,
            chain_config,
            tool=None,
            gui=None,
            boss=None,
            book_path=book_path,
            chain_vars=chain_vars,
            book_id=book_id
        )
        if debug_string and DEBUG:
            prints(f'{debug_string}')
        return chain.run()

    def is_chain_compatible(self, chain_config):
        all_actions = self.plugin_action.actions
        chain_name = chain_config['menuText']
        chain_settings = chain_config['chain_settings']
        chain_links = chain_settings.get('chain_links',[])
        if len(chain_links) == 0:
            return (
                _('No actions'),
                _('Chain must have at least one action')
            )
        for chain_link in chain_links:
            action_name = chain_link['action_name']
            if action_name in all_actions.keys():
                action = all_actions.get(action_name)
                action_settings = chain_link.get('action_settings')
                if not action_settings:
                    action_settings = action.default_settings()
                try:
                    ok = action.is_headless(action_settings)
                except:
                    ok = (
                        _('Headless Error'),
                        _(f'Cannot run action ({action_name}) in headless mode')
                    )
                    import traceback
                    print(traceback.format_exc())
                if ok is False:
                    ok = (
                          _('Headless Error'),
                          _(f'Cannot run action ({action_name}) in headless mode')
                    )
                if ok is not True:
                    return ok
            else:
                return (
                    _('Missing action'),
                    _(f'Cannot find action: {action_name}')
                )
        return True

    def get_compatible_chains(self):
        from importlib import reload
        reload(cfg)
        compatible_chains = []
        all_actions = self.plugin_action.actions
        for chain_config in cfg.get_chains_config():
            if self.is_chain_compatible(chain_config) is True:
                chain_name = chain_config['menuText']
                compatible_chains.append(chain_name)
        return compatible_chains

    def get_chain_supported_formats(self, chain_config):
        all_actions = self.plugin_action.actions
        supported_fmts = set()
        chain_settings = chain_config['chain_settings']
        chain_links = chain_settings.get('chain_links',[])
        for idx, chain_link in enumerate(chain_links):
            action_name = chain_link['action_name']
            if action_name in all_actions.keys():
                action = all_actions[action_name]
                action_settings = chain_link.get('action_settings')
                if not action_settings:
                    action_settings = action.default_settings()
                ac_fmts = action.supported_formats(action_settings)
                if idx == 0:
                    supported_fmts = set(ac_fmts)
                else:
                    supported_fmts = supported_fmts.intersection(set(ac_fmts))
                if not supported_fmts:
                    return supported_fmts
        return supported_fmts

    def validate_editor_chain(self, chain_config):
        if not chain_config:
            return (
                _('Cannot run'),
                _('Chain cannot be found. '
                  'Most probably it has been modified or deleted')
            )
        compatible = self.is_chain_compatible(chain_config)
        if compatible is not True:
            return compatible
        chain = Chain(self.plugin_action, chain_config, tool=None, gui=None, boss=None)
        is_valid = chain.validate()
        return is_valid


editor_wrapper = EditorWrapper()

#===============

def run_chain(chain_name, input_path, output_path):
    suffix = os.path.splitext(input_path)[-1].lower()
    temp_output = better_mktemp(suffix=suffix)
    chain_config = cfg.get_chain_config(chain_name)
    is_valid = editor_wrapper.validate_editor_chain(chain_config)
    if is_valid is not True:
        print(_(f'Error: Chain ({chain_name}) cannot be run from command line.'))
        return

    fmt = os.path.splitext(input_path)[-1].lstrip('.').lower()
    supported_fmts = editor_wrapper.get_chain_supported_formats(chain_config)
    if not fmt in supported_fmts:
        print(_(f'Error: format ({fmt}) is not supported for chain: {chain_name}'))
        return

    if not os.access(input_path, os.R_OK):
        print(_(f'Error: No read permission for input file: {input_file}'))
        return

    if os.path.exists(output_path):
        if not os.access(output_path, os.W_OK):
            print(_(f'Error: No write permission for output path: {output_path}'))
            return
    else:
        output_directory = os.path.dirname(os.path.abspath(output_path))
        if not os.access(output_directory, os.W_OK):
            print(_(f'Error: No write permission for output directory: {output_directory}'))
            return

    if input_path == output_path:
        print(_(f'Error: ouput_path cannot be the same as input_path'))
        return

    print(f'\nRunning chain ({chain_name}) for book: {input_path}\n=============================================================\n')
    shutil.copy(input_path, temp_output)
    editor_wrapper.run_editor_chain(chain_config, temp_output)
    shutil.copy(temp_output, output_path)

def cmdline_run(args):
    parser = argparse.ArgumentParser(
        description="Run an editor chain from the command line"
    )

    parser.add_argument(
        "chain_name",
        help="name of the chain you want to run"
    )

    parser.add_argument(
        "input_path",
        help="path to the input file"
    )

    parser.add_argument(
        "output_path",
        help="path to the output file"
    )

    args = parser.parse_args(args)
    try:
        run_chain(args.chain_name, args.input_path, args.output_path)
    except:
        import traceback
        print(traceback.format_exc())
        os.remove(args.output_path)

if __name__ == '__main__':
    _app = QApplication([])
    args = sys.argv
    args.remove("Editor Chains")
