#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from collections import OrderedDict
from functools import partial
import itertools, operator
import copy
import inspect

from qt.core import (QApplication, Qt, QTableView, QAbstractItemView, QModelIndex,
                     QItemSelection, QMenu, QDialog, QAction, QInputDialog)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import question_dialog, error_dialog, info_dialog, open_local_file
from calibre.utils.config import JSONConfig
from calibre.utils.zipfile import ZipFile
from calibre.utils.config import tweaks

import calibre_plugins.editor_chains.config as cfg
from calibre_plugins.editor_chains.gui.delegates import ComboDelegate, ButtonDelegate, TreeComboDelegate
from calibre_plugins.editor_chains.gui.models import UP, DOWN
from calibre_plugins.editor_chains.gui import SettingsWidgetDialog
from calibre_plugins.editor_chains.common_utils import ViewLog, get_icon, safe_name
from calibre_plugins.editor_chains.export_import import export_chains, pick_archive_name_to_export, chains_config_from_archive


try:
    load_translations()
except NameError:
    prints("EditorChains::gui/views.py - exception when loading translations")

class TableView(QTableView):

    def __init__(self, parent):
        QTableView.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.setMouseTracking(True)
        self.setProperty('highlight_current_item', 150)
        self.verticalHeader().setDefaultSectionSize(30)
        self.verticalHeader().setVisible(True)
        self.verticalHeader().setSectionsMovable(True)
        self.horizontalHeader().setSectionsMovable(True)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        self.horizontalHeader().setStretchLastSection(True)
        #self.column_header = HeaderView(Qt.Orientation.Horizontal, self)
        self.column_header = self.horizontalHeader()

    def _set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def set_current_row(self, row=0, select=True, for_sync=False):
        if row > -1 and row < self.model().rowCount(QModelIndex()):
            h = self.horizontalHeader()
            logical_indices = list(range(h.count()))
            logical_indices = [x for x in logical_indices if not
                    h.isSectionHidden(x)]
            pairs = [(x, h.visualIndex(x)) for x in logical_indices if
                    h.visualIndex(x) > -1]
            if not pairs:
                pairs = [(0, 0)]
            pairs.sort(key=lambda x: x[1])
            i = pairs[0][0]
            index = self.model().index(row, i)
            if for_sync:
                sm = self.selectionModel()
                sm.setCurrentIndex(index, sm.NoUpdate)
            else:
                self.setCurrentIndex(index)
                if select:
                    sm = self.selectionModel()
                    sm.select(index, sm.SelectionFlag.ClearAndSelect|sm.Rows)

    def select_rows(self, rows, change_current=True, scroll=True):
        rows = {x.row() if hasattr(x, 'row') else x for x in
            rows}
        rows = list(sorted(rows))
        if rows:
            row = rows[0]
            if change_current:
                self.set_current_row(row, select=False)
            if scroll:
                self.scroll_to_row(row)
        sm = self.selectionModel()
        sel = QItemSelection()
        m = self.model()
        max_col = m.columnCount(QModelIndex()) - 1
        # Create a range based selector for each set of contiguous rows
        # as supplying selectors for each individual row causes very poor
        # performance if a large number of rows has to be selected.
        for k, g in itertools.groupby(enumerate(rows), lambda i_x:i_x[0]-i_x[1]):
            group = list(map(operator.itemgetter(1), g))
            sel.merge(QItemSelection(m.index(min(group), 0),
                m.index(max(group), max_col)), sm.SelectionFlag.Select)
        sm.select(sel, sm.SelectionFlag.ClearAndSelect)
        return rows

    def scroll_to_row(self, row):
        if row > -1:
            h = self.horizontalHeader()
            for i in range(h.count()):
                if not h.isSectionHidden(i) and h.sectionViewportPosition(i) >= 0:
                    self.scrollTo(self.model().index(row, i), self.PositionAtCenter)
                    break

    def add_row(self):
        self.setFocus()
        # We will insert a blank row below the currently selected row
        row = self.selectionModel().currentIndex().row() + 1
        self.model().insertRow(row)
        self.scroll_to_row(row)

    def delete_rows(self):
        self.setFocus()
        selrows = self.selectionModel().selectedRows()
        selrows = sorted(selrows, key=lambda x: x.row())
        if len(selrows) == 0:
            return
        message = _('<p>Are you sure you want to delete this action?')
        if len(selrows) > 1:
            message = _(f'<p>Are you sure you want to delete the selected {len(selrows)} actions?')
        if not question_dialog(self, _('Are you sure?'), message, show_copy_button=False):
            return
        first_sel_row = selrows[0].row()
        for selrow in reversed(selrows):
            self.model().removeRow(selrow.row())
        if first_sel_row < self.model().rowCount(QModelIndex()):
            self.setCurrentIndex(self.model().index(first_sel_row, 0))
            self.scroll_to_row(first_sel_row)
        elif self.model().rowCount(QModelIndex()) > 0:
            self.setCurrentIndex(self.model().index(first_sel_row - 1, 0))
            self.scroll_to_row(first_sel_row - 1)

    def move_rows(self, direction=DOWN):
        self.setFocus()
        sm = self.selectionModel()
        selrows = sm.selectedRows()
        current_index = sm.currentIndex()
        if len(selrows) == 0:
            return
        m = self.model()
        # make sure there is room to move UP/DOWN
        if direction == DOWN:
            boundary_selrow = selrows[-1].row()
            if boundary_selrow >= m.rowCount(QModelIndex()) - 1:
                return
        elif direction == UP:
            boundary_selrow = selrows[0].row()
            if boundary_selrow <= 0:
                return
        
        rows = [row.row() for row in selrows]
        m.move_rows(rows, direction=direction)
        
        # reset selections and scroll
        rows = [row+direction for row in rows]
        scroll_to_row = boundary_selrow + direction
        self.select_rows(rows)
        #self.set_current_row(scroll_to_row)
        self.scroll_to_row(scroll_to_row)

    def show_column_header_context_menu(self, pos):
        model = self.model()
        idx = self.column_header.logicalIndexAt(pos)
        col = None
        if idx > -1 and idx < len(model.col_map):
            col = model.col_map[idx]
            name = str(model.headerData(idx, Qt.Orientation.Horizontal, Qt.ItemDataRole.DisplayRole) or '')
            self.column_header_context_menu = self.create_header_context_menu(col, name)
            self.column_header_context_menu.popup(self.column_header.mapToGlobal(pos))

    def create_header_context_menu(self, col, name):
        ans = QMenu(self)
        model = self.model()
        handler = partial(self.column_header_context_handler, column=col)
        if col in model.optional_cols:
            ans.addAction(_('Hide column %s') % name, partial(handler, action='hide'))

        hidden_cols = {model.col_map[i]: i for i in range(self.column_header.count())
                       if self.column_header.isSectionHidden(i)}

        hidden_cols = {k:v for k,v in hidden_cols.items() if k in model.optional_cols}

        ans.addSeparator()
        if hidden_cols:
            m = ans.addMenu(_('Show column'))
            hcols = [(hcol, str(self.model().headerData(hidx, Qt.Orientation.Horizontal, Qt.ItemDataRole.DisplayRole) or ''))
                     for hcol, hidx in hidden_cols.items()]
            hcols.sort()
            for hcol, hname in hcols:
                m.addAction(hname, partial(handler, action='show', column=hcol))
        return ans


    def column_header_context_handler(self, action=None, column=None):
        if not action or not column:
            return
        try:
            idx = self.col_map.index(column)
        except:
            return
        h = self.column_header

        if action == 'hide':
            if h.hiddenSectionCount() >= h.count():
                return error_dialog(self, _('Cannot hide all columns'), _(
                    'You must not hide all columns'), show=True)
            h.setSectionHidden(idx, True)
        elif action == 'show':
            h.setSectionHidden(idx, False)
            if h.sectionSize(idx) < 3:
                sz = h.sectionSizeHint(idx)
                h.resizeSection(idx, sz)

    def get_state(self):
        h = self.column_header
        cm = self.model().col_map
        state = {}
        state['hidden_columns'] = [cm[i] for i in range(h.count())
                if h.isSectionHidden(i)]
        state['column_positions'] = {}
        state['column_sizes'] = {}
        for i in range(h.count()):
            name = cm[i]
            state['column_positions'][name] = h.visualIndex(i)
            state['column_sizes'][name] = h.sectionSize(i)
        return state

    def apply_state(self, state):
        h = self.column_header
        cmap = {}
        hidden = state.get('hidden_columns', [])
        for i, c in enumerate(self.model().col_map):
            cmap[c] = i
            h.setSectionHidden(i, c in hidden)

        positions = state.get('column_positions', {})
        pmap = {}
        for col, pos in positions.items():
            if col in cmap:
                pmap[pos] = col
        for pos in sorted(pmap.keys()):
            col = pmap[pos]
            idx = cmap[col]
            current_pos = h.visualIndex(idx)
            if current_pos != pos:
                h.moveSection(current_pos, pos)

        sizes = state.get('column_sizes', {})
        for col, size in sizes.items():
            if col in cmap:
                sz = sizes[col]
                if sz < 3:
                    sz = h.sectionSizeHint(cmap[col])
                h.resizeSection(cmap[col], sz)

        for i in range(h.count()):
            if not h.isSectionHidden(i) and h.sectionSize(i) < 3:
                sz = h.sectionSizeHint(i)
                h.resizeSection(i, sz)

class ActionsTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.doubleClicked.connect(self._on_double_clicked)
        self.column_header.setSectionsClickable(True)
        self.column_header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.column_header.customContextMenuRequested.connect(partial(self.show_column_header_context_menu))
        #self.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        #self.setContextMenuPolicy(Qt.ContextMenuPolicy.DefaultContextMenu)
        #self.customContextMenuRequested.connect(self.show_context_menu)

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        #self.actions_delegate = ComboDelegate(self, self.plugin_action.actions.keys())
        self.actions_delegate = TreeComboDelegate(self, self.action_tree())
        self.setItemDelegateForColumn(self.col_map.index('action_name'), self.actions_delegate)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('action_settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def action_tree(self):
        '''
        Build dictionary containing action tree where:
        1. Builtin actions are top level nodes
        2. User Action are inside a node called User Actions
        3. Actions imported from another plugins are under node
           for each plugin that has action. And all those are 
           under a note called 'Other Plugins Actions'
        This is a dictionary of nested dictionaries. Even actions
        are nested dictionaries in the form of 'action_name': {}
        '''
        for action in self.plugin_action.user_actions.values():
            setattr(action, '_display_tree', ['User Actions'] + getattr(action, 'display_tree', []))

        for action in self.plugin_action.imported_actions.values():
            source_plugin = getattr(action, '_source_plugin')
            tree_path = ['Imported From Other Plugins'] + getattr(action, 'display_tree', [])
            if source_plugin:
                tree_path.insert(1, source_plugin)
            setattr(action, '_display_tree', tree_path)

        d = OrderedDict()
        for name, action in self.plugin_action.actions.items():
            parent = d
            tree_path = getattr(action, '_display_tree', []) + [name]
            for node in tree_path:
                if not parent.get(node):
                    parent[node] = {}
                parent = parent[node]
        return d            

    def copy_row(self):
        self.setFocus()
        m = self.model()
        sm = self.selectionModel()
        index = sm.currentIndex()
        new_row = copy.deepcopy(m.chain_links[index.row()])
        # We will insert the new row below the currently selected row
        m.chain_links.insert(index.row()+1, new_row)
        m.layoutChanged.emit()
        self.select_rows([index.row()+1])

    #def contextMenuEvent(self, event):
        #self.show_context_menu(event)

    #def show_context_menu(self, event):
        #model = self.model()
        #self.context_menu = m = self.create_context_menu()
        #m.popup(event.globalPos())

    def _on_button_clicked(self, index):
        m = self.model()
        col_name = m.col_map[index.column()]
        if col_name == 'action_settings':
            chain_link = m.chain_links[index.row()]
            action_name = chain_link['action_name']
            if not action_name:
                # user clicking on setting before choosing action
                error_dialog(
                    self,
                    _('No Action Selected'),
                    _('You must choose an action first'),
                    show=True
                )
                return
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action_settings:
                action_settings = action.default_settings()
            config_widget = action.config_widget()
            if config_widget:
                name = f'ActionsChain::{action_name}'
                title = f'{action_name}'
                if issubclass(config_widget, QDialog):
                    # config_widget is a dialog
                    d = config_widget(self, self.plugin_action, action, name, title)
                else:
                    # config_widget is a qwidget
                    d = SettingsWidgetDialog(name, self, self.plugin_action, config_widget, action, m.chain_name, m.chains_config, title)
                # inject copy of chain data into the settings dialog, for internal use only
                d._chain_link = copy.deepcopy(chain_link)
                if action_settings:
                    d.load_settings(action_settings)
                if d.exec_() == d.Accepted:
                    chain_link['action_settings'] = d.settings
                    # reset any previous error if present
                    chain_link['errors'] = ''
                    m.dataChanged.emit(index, index)

    def _on_double_clicked(self, index):
        m = self.model()
        col_name = m.col_map[index.column()]
        if col_name == 'errors':
            chain_link = m.chain_links[index.row()]
            details = chain_link.get('errors', '')
            self._view_error_details(details)

    def _view_error_details(self, details):
        ViewLog(_('Errors details'), details, self)

class MenusTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.icon_cache = self.plugin_action.icon_cache
        self.column_header.setSectionsClickable(True)
        self.horizontalHeader().setStretchLastSection(False)
        self.column_header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.column_header.customContextMenuRequested.connect(partial(self.show_column_header_context_menu))
        self.customContextMenuRequested.connect(self.show_context_menu)
#        self.doubleClicked.connect(self._on_double_clicked)

    def set_model(self, _model):
        self.setModel(_model)
        _model.error.connect(lambda *args: error_dialog(self, *args, show=True))
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def append_data(self, chains_config):
        added = []
        failed = []
        m = self.model()
        names = m.get_names()
        for chain_config in chains_config:
            chain_name = chain_config['menuText']
            if chain_config['menuText'] in names:
                failed.append(chain_name)
                if DEBUG:
                    prints(f'Editor Chains: cannot import chain ({chain_name}). Name already exists')
            else:
                m.chains_config.append(chain_config)
                added.append(chain_name)
        m.layoutChanged.emit()
        return added, failed

    def get_selected_data(self):
        chains_config = []
        m = self.model()
        for row in self.selectionModel().selectedRows():
            chains_config.append(m.chains_config[row.row()])
        return chains_config

    def copy_row(self):
        self.setFocus()
        m = self.model()
        sm = self.selectionModel()
        index = sm.currentIndex()
        new_row = copy.deepcopy(m.chains_config[index.row()])
        # change name
        if new_row['menuText']:
            all_names = m.get_names()
            new_name = new_row['menuText'] + ' (copy)'
            new_name = safe_name(new_name, all_names)
        else:
            new_name = new_row['menuText']
        new_row['menuText'] = new_name
        # We will insert a blank row below the currently selected row
        m.chains_config.insert(index.row()+1, new_row)
        m.layoutChanged.emit()
        self.select_rows([index.row()+1])

    def contextMenuEvent(self, event):
        self.show_context_menu(event)

    def show_context_menu(self, event):
        model = self.model()
        self.context_menu = m = self.create_context_menu()
        m.popup(event.globalPos())

    def create_context_menu(self):
        m = QMenu(self)

        act_add_image = QAction(get_icon('images/image_add.png'), _('&Add image...'), m)
        act_add_image.triggered.connect(self.plugin_action.icon_cache.pick_icon)
        m.addAction(act_add_image)
        act_open = QAction(get_icon('document_open.png'), _('&Open images folder'), m)
        act_open.triggered.connect(partial(self.open_images_folder, self.plugin_action.resources_dir))
        m.addAction(act_open)
        m.addSeparator()
        act_import = QAction(get_icon('images/import.png'), _('&Import...'), m)
        act_import.triggered.connect(self.import_menus)
        m.addAction(act_import)
        act_export = QAction(get_icon('images/export.png'), _('&Export...'), m)
        act_export.triggered.connect(self.export_chains)
        m.addAction(act_export)
        m.addSeparator()
        
        return m

    def _on_button_clicked(self, index):
        m = self.model()
        chain_config = m.chains_config[index.row()]
        col_name = m.col_map[index.column()]
        if col_name == 'settings':
            from calibre_plugins.editor_chains.gui.actions_dialog import ActionsDialog
            chain_name = chain_config['menuText']
            if chain_name:
                d = ActionsDialog(self.plugin_action.gui, self.plugin_action, copy.deepcopy(m.chains_config), chain_name)
                if d.exec_() == d.Accepted:
                    chain_config['chain_settings'] = d.chain_settings
                    chain_config['image'] = d.icon
                    m.dataChanged.emit(m.index(index.row(), 0), index)

    def open_images_folder(self, path):
        if not os.path.exists(path):
            if not question_dialog(self, _('Are you sure?'), '<p>'+
                    _('Folder does not yet exist. Do you want to create it?<br>%s') % path,
                    show_copy_button=False):
                return
            os.makedirs(path)
        open_local_file(path)

    def import_menus(self):
        chains_config = chains_config_from_archive(self.plugin_action.gui, self.plugin_action, add_resources=True)
        if not chains_config:
            return
        # Now insert the menus into the table
        added, failed = self.append_data(chains_config)
        det = _('%d menu items added') % len(added)
        if len(failed) > 0:
            det += _(', failed to add %d menu items') % len(failed)
        info_dialog(self, _('Import completed'), det,
                    show=True, show_copy_button=False)

    def export_chains(self):
        chains_config = self.get_selected_data()
        export_chains(self.plugin_action, self.plugin_action.gui, chains_config)

    def keyPressEvent(self, ev):
        m = self.model()
        sm = self.selectionModel()
        selrows = sm.selectedRows()
        col = self.col_map.index('subMenu')
        if ev.key() == Qt.Key.Key_F2 and (sm.currentIndex().column() == col) and len(selrows) > 1:
            new_name, ok = QInputDialog.getText(self, _('New submenu:'),
                                                _('New submenu:'), text='')
            new_name = str(new_name).strip()
            if ok:
                for selrow in selrows:
                    row = selrow.row()
                    index = m.index(row, col, QModelIndex())
                    m.setData(index, new_name, Qt.EditRole)
            return
        return TableView.keyPressEvent(self, ev)

