#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QAbstractTableModel, QModelIndex,
                     QIcon, QImage, QBrush, pyqtSignal, QStyle)

from calibre import prints
from calibre.gui2 import error_dialog
from calibre.constants import DEBUG

from calibre_plugins.editor_chains.common_utils import get_icon, get_pixmap, call_method
from calibre_plugins.editor_chains.chains import Chain
import calibre_plugins.editor_chains.config as cfg

from calibre.constants import numeric_version
if numeric_version >= (5,99,0):
    QT_CHECKED = Qt.CheckState.Checked.value
    QT_UNCHECKED = Qt.CheckState.Unchecked.value
else:
    QT_CHECKED = Qt.Checked
    QT_UNCHECKED = Qt.Unchecked

try:
    load_translations()
except NameError:
    prints("ActionsChain::gui/models.py - exception when loading translations")

DOWN    = 1
UP      = -1

class ActionsModel(QAbstractTableModel):

    def __init__(self, plugin_action, chains_config, chain_name):
        # We are passing the chains_config and chain_name and extracting the chain_links from them.
        # This is done because some actions like "Chain Caller" need to have access to the chain_name
        # and to chains_config.
        QAbstractTableModel.__init__(self)
        self.chains_config = chains_config
        self.chain_name = chain_name
        self.chain_config = cfg.get_chain_config(chain_name, chains_config=chains_config)
        self.chain_links = self.chain_config.get('chain_settings', {}).get('chain_links', [])
        self.plugin_action = plugin_action
        self.col_map = ['action_name','action_settings','comment','errors']
        self.editable_columns = ['action_name','comment']
        self.optional_cols = ['comment']
        self.hidden_cols = []
        self.col_min_width = {
            'action_name': 300,
            'action_settings': 50,
            'comment': 200,
            'errors': 250
        }
        all_headers = [_('Action'),_('Settings'),_('Comment'),_('Errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.chain_links)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.chain_links):
            return None
        chain_link = self.chain_links[row]
        col_name = self.col_map[col]
        value = chain_link.get(col_name, '')
        error = chain_link.get('errors', '')
        
        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['action_settings']:
                pass
            elif col_name == 'errors':
                if error:
                    return _('Validation Error. Double click for details')
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return QIcon(get_icon('dialog_error.png'))
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

            elif col_name == 'comment':
                tooltip = chain_link.get('comment')
                return tooltip

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        chain_link = self.chain_links[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'action_name':
                old_name = chain_link.get('action_name', '')
                chain_link['action_name'] = val
                if val != old_name:
                    # reset settings as they are not valid for this new action
                    chain_link['action_settings'] = {}
                    # reset any previous errors
                    chain_link['errors'] = ''
            elif col_name == 'settings':
                pass
            elif col_name in ['comment']:
                chain_link[col_name] = val
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            chain_link = self.chain_links[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def button_state(self, index):
        visible = False
        enabled = False
        row, col = index.row(), index.column()
        chain_link = self.chain_links[row]
        action_name = chain_link.get('action_name')
        if action_name:
            visible = True
            action = self.plugin_action.actions.get(action_name)
            config_widget = None
            if action:
                config_widget = action.config_widget()
            if config_widget:
                enabled = True
        return visible, enabled
        

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            chain_link = {}
            chain_link['action_name'] = ''
            chain_link['action_settings'] = {}
            chain_link['comment'] = ''
            self.chain_links.insert(row + i, chain_link)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.chain_links.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.chain_links.pop(row)
            self.chain_links.insert(row+direction, pop)
        self.layoutChanged.emit()

    def validate_chain_link(self, row, chain_link):
        col = self.col_map.index('errors')
        action_name = chain_link['action_name']
        if action_name in self.plugin_action.actions.keys():
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action_settings:
                action_settings = action.default_settings()
            is_action_valid = action.validate(action_settings)
            if is_action_valid is not True:
                msg, details = is_action_valid
                chain_link['errors'] = details
                index = self.index(row, col, QModelIndex())
                self.dataChanged.emit(index, index)
                return is_action_valid
        else:
            details = _(f'Action ({action_name}) is not currently available')
            chain_link['errors'] = details
            index = self.index(row, col, QModelIndex())
            self.dataChanged.emit(index, index)
            return (_('Action unavailable'), details)
        return True

    def validate(self):
        is_chain_valid = True
        for row, chain_link in enumerate(self.chain_links):
            is_action_valid = self.validate_chain_link(row, chain_link)
            if is_action_valid is not True:
                is_chain_valid = False
        return is_chain_valid

class MenusModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, chains_config, plugin_action):
        QAbstractTableModel.__init__(self)
        self.chains_config = chains_config
        self.plugin_action = plugin_action
        self.col_map = ['active', 'menuText', 'subMenu', 'settings']
        self.editable_columns = ['menuText']
        self.hidden_cols = []
        self.optional_cols = ['conditions']
        self.col_min_width = {
            'active': 10,
            'subMenu': 50,
            'menuText': 200,
            'settings': 30
        }
        all_headers = ['', _('Title'), _('Submenu'), _('Settings')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.chains_config)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.chains_config):
            return None
        chain_config = self.chains_config[row]
        col_name = self.col_map[col]
        value = chain_config.get(col_name, '')
        error = chain_config.get('errors', '')

        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['active','settings','conditions']:
                pass
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'menuText':
                icon_name = chain_config.get('image', '')
                if icon_name:
                    return QIcon(get_icon(icon_name))
            elif col_name == 'errors':
                if error:
                    return QIcon(get_icon('dialog_error.png'))
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                is_checked = chain_config[col_name]
                state = QT_CHECKED if is_checked else QT_UNCHECKED
                return state

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        chain_config = self.chains_config[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name in ['active','settings']:
                pass
            # make sure the name is unique
            elif col_name == 'menuText':
                old_name = self.data(index, Qt.ItemDataRole.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if val in names:
                    msg = _('Name must be unique')
                    details = _(f'Name ({val}) is already used by another menu')
                    self.error.emit(msg, details)
                else:
                    chain_config[col_name] = val
            else:
                chain_config[col_name] = val
            done = True

        elif role == Qt.ItemDataRole.CheckStateRole:
            if col_name == 'active':
                state = True if (value == QT_CHECKED) else False
                chain_config[col_name] = state
            return True
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            chain_config = self.chains_config[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
            elif col_name == 'subMenu':
                menu_col = self.col_map.index('menuText')
                menu_index = self.index(index.row(), menu_col, QModelIndex())
#                if self.data(menu_index, Qt.ItemDataRole.DisplayRole):
#                    flags |= Qt.ItemFlag.ItemIsEditable
                # Even if no menu name is given, submenu should be editable to enable adding separators
                # to submenus
                flags |= Qt.ItemFlag.ItemIsEditable
            if col_name == 'active':
                flags |=  Qt.ItemFlag.ItemIsUserCheckable
        return flags

    def button_state(self, index):
        visible = False
        enabled = True
        row, col = index.row(), index.column()
        chain_config = self.chains_config[row]
        chain_name = chain_config.get('menuText')
        if chain_name:
            visible = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            chain_config = {}
            chain_config['active'] = True
            chain_config['menuText'] = ''
            chain_config['subMenu'] = ''
            chain_config['image'] = ''
            chain_config['chain_settings'] = {}
            self.chains_config.insert(row + i, chain_config)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.chains_config.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.chains_config.pop(row)
            self.chains_config.insert(row+direction, pop)
        self.layoutChanged.emit()

    def get_names(self):
        names = []
        col = self.col_map.index('menuText')
        for row in range(self.rowCount(QModelIndex())):
            index = self.index(row, col, QModelIndex())
            name = self.data(index, Qt.ItemDataRole.DisplayRole)
            # empty name belong to separators, dont include
            if name:
                names.append(name)
        return names

