#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import copy

from qt.core import (Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, QMenu,
                     QToolButton, QAction, QIcon, QSizePolicy, QInputDialog,
                     QSpacerItem)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, info_dialog, question_dialog
from calibre.gui2.actions import menu_action_unique_name
from calibre.gui2.tweak_book.widgets import Dialog

from calibre_plugins.editor_chains.common_utils import get_icon, KeyboardConfigDialog
from calibre_plugins.editor_chains.modules import ManageModulesDialog
from calibre_plugins.editor_chains.gui.views import MenusTable
from calibre_plugins.editor_chains.gui.models import MenusModel, UP, DOWN
from calibre_plugins.editor_chains.export_import import pick_archive_name_to_export

import calibre_plugins.editor_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("EditorChains::gui/chains_dialog.py - exception when loading translations")

class ChainsDialog(Dialog):

    def __init__(self, plugin_action):
        self.plugin_action = plugin_action
        Dialog.__init__(self, _('Chains Dialog'), 'editor-chains-chains-dialog', self.plugin_action.gui)

    def setup_ui(self):
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        c = cfg.plugin_prefs[cfg.STORE_MENUS_NAME]
        chains_config = c.get(cfg.KEY_MENUS, [])

        self.resources_dir = self.plugin_action.resources_dir

        heading_layout = QHBoxLayout()
        layout.addLayout(heading_layout)
        heading_label = QLabel(_('&Select and configure the menu items to display:'), self)
        heading_layout.addWidget(heading_label)
        # Add hyperlink to a help url at the right.
        help_label = QLabel(_('<a href="https://www.mobileread.com/forums/showpost.php?p=4174583&postcount=1">Help</a>'), self)
        help_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        help_label.setOpenExternalLinks(True)
        help_label.setAlignment(Qt.AlignRight)
        heading_layout.addWidget(help_label)

        # Add a horizontal layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Create a table the user can edit the data values in
        self._table = MenusTable(self)
        heading_label.setBuddy(self._table)
        table_layout.addWidget(self._table)
        
        # Create table model
        _model = MenusModel(copy.deepcopy(chains_config), self.plugin_action)
        self._table.set_model(_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)

        # restore table state
        state = cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_CHAINS_TABLE_STATE]
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        table_layout.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        button_layout.addWidget(move_up_button)
        spacerItem = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add menu item row'))
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        copy_button = self.copy_button = QToolButton(self)
        copy_button.setToolTip(_('Duplicate chain'))
        copy_button.setIcon(QIcon(I('edit-copy.png')))
        button_layout.addWidget(copy_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete menu item row'))
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)
        spacerItem3 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)
        copy_button.clicked.connect(self._table.copy_row)

        hl = QHBoxLayout()
        layout.addLayout(hl)

        keyboard_shortcuts_button = QPushButton(_('Keyboard shortcuts...'), self)
        keyboard_shortcuts_button.setToolTip(_(
                    _('Edit the keyboard shortcuts associated with this plugin')))
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        hl.addWidget(keyboard_shortcuts_button)
        hl.addStretch(1)
                
        manage_modules_button = QPushButton(_('Manage modules...'), self)
        manage_modules_button.setToolTip(_(
                    _('Manage user custom modules')))
        manage_modules_button.clicked.connect(self.manage_modules)
        hl.addWidget(manage_modules_button)

        layout.addWidget(self.bb)
        
        # Build a list of all the active unique names
        self.orig_unique_active_menus = self.get_active_unique_names(chains_config)

        self._on_table_selection_change()

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.copy_button.setEnabled(selection_count == 1)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # save table state
        cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_CHAINS_TABLE_STATE] = self._table.get_state()

    def save_settings(self):
        self.save_table_state()
        
        open_menus = {}
        open_menus[cfg.KEY_MENUS] = self._table.model().chains_config
        cfg.plugin_prefs[cfg.STORE_MENUS_NAME] = open_menus

        # For each menu that was visible but now is not, we need to unregister any
        # keyboard shortcut associated with that action.
        menus_changed = False
        kb = self.plugin_action.gui.keyboard
        new_unique_active_menus = self.get_active_unique_names(open_menus[cfg.KEY_MENUS])
        for raw_unique_name in list(self.orig_unique_active_menus.keys()):
            if raw_unique_name not in new_unique_active_menus:
                unique_name = 'editor-chains_'+raw_unique_name
                if unique_name in kb.shortcuts:
                    kb.unregister_shortcut(unique_name)
                    menus_changed = True
        if menus_changed:
            self.plugin_action.gui.keyboard.finalize()
        self.orig_unique_active_menus = new_unique_active_menus
        
        for menu in self.plugin_action.all_menus:
            self.plugin_action.tool.rebuild_menus(menu)

    def get_active_unique_names(self, chains_config):
        active_unique_names = {}
        for chain_config in chains_config:
            if chain_config['active']:
                unique_name = chain_config['menuText']
                active_unique_names[unique_name] = chain_config['menuText']
        return active_unique_names

    def edit_shortcuts(self):
        self.save_settings()
#        # Force the menus to be rebuilt immediately, so we have all our actions registered
#        for menu in self.plugin_action.all_menus:
#            self.plugin_action.tool.rebuild_menus(menu)
        d = KeyboardConfigDialog(self.plugin_action.gui, _('Editor Chains'))
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()

    def manage_modules(self):
        d = ManageModulesDialog(self, self.plugin_action)
        d.exec_()

    def accept(self):
        self.save_settings()
        Dialog.accept(self)

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)
