#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import os, shutil

from qt.core import (QApplication, Qt, QWidget, QHBoxLayout, QVBoxLayout, QGridLayout, QComboBox,
                     QLabel, QDialog, QDialogButtonBox, QLineEdit, QRadioButton, QPushButton,
                     QGroupBox, QAbstractItemView, QTreeView, QStandardItem, QStandardItemModel,
                     QFrame, QModelIndex, QEvent, QColor, QFont, QSize, QIcon, QPixmap,
                     QScrollArea, QSizePolicy, QCheckBox, QSpinBox)

from calibre import prints
from calibre.constants import DEBUG, iswindows, isosx
from calibre.utils.config import config_dir
from calibre.gui2 import error_dialog, question_dialog, choose_files
from calibre.gui2.tweak_book.widgets import Dialog

import calibre_plugins.editor_chains.config as cfg
from calibre_plugins.editor_chains.common_utils import get_icon, NoWheelComboBox
#from calibre_plugins.editor_chains.scopes.fallback import FallbackScope

try:
    load_translations()
except NameError:
    prints("EditorChains::gui/__init__.py - exception when loading translations")

class SettingsWidgetDialog(Dialog):

    def __init__(self, name, parent, plugin_action, widget_cls, action, chain_name='', chains_config={}, title=_('Settings')):
        self.plugin_action = plugin_action
        self.widget_cls = widget_cls
        self.action = action
        self.chain_name = chain_name
        self.chains_config = chains_config
        Dialog.__init__(self, title, name, parent)

    def setup_ui(self):
        try:
            # Signature of widget_cls was changed to allow passing extra parameters
            self.widget = self.widget_cls(self.plugin_action, self.chain_name, self.chains_config)
        except Exception as e:
            if DEBUG:
                prints('Editor Chains: SettingsWidgetDialog: Falling back to old constructor call')
            # Some actions still use the old signature
            self.widget = self.widget_cls(self.plugin_action)
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)
        l.addWidget(self.bb)

    def load_settings(self, settings):
        self.widget.load_settings(settings)

    def save_settings(self):
        return self.widget.save_settings()

    def validate(self, settings):
        if hasattr(self.widget, 'validate'):
            return self.widget.validate(settings)
        else:
            return self.action.validate(settings)
    
    def accept(self):
        self.settings = self.save_settings()
        # Validate settings
        #is_valid = self.action.validate(self.settings)
        is_valid = self.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(self, msg, details, show=True)
            return
        if not self.plugin_action.gui:
            # We are running headless. Make sure this action can run in headleass mode.
            is_headless = self.action.is_headless(self.settings)
            if is_headless is not True:
                msg, details = is_headless
                error_dialog(self, msg, details, show=True)
                return
        Dialog.accept(self)

COMBO_IMAGE_ADD = _('Add New Image...')

class IconCache(object):

    def __init__(self, parent):
        self.parent = parent
        self.gui = parent.gui
        self.resources_dir = parent.resources_dir
        self.create_icon_map(self.resources_dir)
        self._names = sorted(self.map.keys())

    def create_icon_map(self, resources_dir):
        self.map = {}

        if os.path.exists(resources_dir):
            # Get the names of any .png images in this directory
            for f in os.listdir(resources_dir):
                if f.lower().endswith('.png'):
                    icon_name = os.path.basename(f)
                    self.map[icon_name] = get_icon(icon_name)

    @property
    def names(self):
        icon_names = sorted(self.map.keys())
        # Add a blank item at the beginning of the list, and a blank then special 'Add' item at end
        return icon_names

    def pick_icon(self):
        images = choose_files(None, _('menu icon dialog'), _('Select a .png file for the menu icon'),
                             filters=[('PNG Image Files', ['png'])], all_files=False, select_only_single_file=True)
        if not images:
            return
        icon = images[0]
        if not icon.lower().endswith('.png'):
            return error_dialog(self, 'Cannot select image',
                    'Source image must be a .png file.', show=True)

        self.add_icon(icon)
        return os.path.basename(icon)

    def add_icon(self, icon):
        icon_name = os.path.basename(icon)
        dest_path = os.path.join(self.resources_dir, icon_name)
        if not icon:
            return error_dialog(self, _('Cannot import image'),
                    _('You must specify a source file.'), show=True)
        if not icon.lower().endswith('.png'):
            return error_dialog(self, 'Cannot import image',
                    _('Source image must be a .png file.'), show=True)
        if not os.path.exists(icon):
            return error_dialog(self, 'Cannot import image',
                    _('Source image does not exist!'), show=True)
        shutil.copyfile(icon, dest_path)
        self.map[icon_name] = get_icon(icon)
        self._names.append(icon_name)
        self._names.sort()

    def get_icon(self, icon, default=None):
        if not default:
            default = get_icon(icon)
        return self.map.get(icon, default)

class ImageComboBox(NoWheelComboBox):

    def __init__(self, parent, icon_cache):
        NoWheelComboBox.__init__(self, parent)
        self.icon_cache = icon_cache
        self.populate_combo()
        self.activated.connect(self.image_combo_index_changed)

    def populate_combo(self):
        self.clear()
        for i, icon_name in enumerate(self.icon_cache.names):
            self.insertItem(i, self.icon_cache.get_icon(icon_name, icon_name), icon_name)
        self.insertSeparator(0)
        self.insertItem(0, COMBO_IMAGE_ADD)
        self.insertItem(0, '')
        self.setCurrentIndex(-1)

    def image_combo_index_changed(self):
        if self.currentText() == COMBO_IMAGE_ADD:
            # Special item in the combo for choosing a new image to add to Calibre
            res = self.icon_cache.pick_icon()
            if res:
                # new icon was added to the image map, insert and change to it.
                self.update_icon(res)
            else:
                # The user cancelled the add image dialog, revert to previous icon
                prevIndex = self.itemData(0)
                if prevIndex is None:
                    prevIndex = -1
                self.blockSignals(True)
                self.setCurrentIndex(prevIndex)
                self.blockSignals(False)

    def update_icon(self, icon):
        self.blockSignals(True)
        self.populate_combo()
        self.set_current_text(icon)
        self.blockSignals(False)

    def current_text(self):
        return self.currentText()

    def set_current_text(self, text):
        idx = self.findText(text)
        self.setCurrentIndex(idx)
        # Store the current index as item data in index 0 in case user choose open image dialog and cancel
        self.setItemData(0, idx)

class TreeComboBox(QComboBox):
    def __init__(self, *args):
        super().__init__(*args)

        self.__skip_next_hide = False

        self.tree_view = tree_view = QTreeView(self)
        tree_view.setFrameShape(QFrame.NoFrame)
        tree_view.setEditTriggers(tree_view.NoEditTriggers)
        tree_view.setAlternatingRowColors(True)
        tree_view.setSelectionBehavior(tree_view.SelectRows)
        tree_view.setWordWrap(True)
        tree_view.setAllColumnsShowFocus(True)
        tree_view.setHeaderHidden(True)
        self.setView(tree_view)

        self.view().viewport().installEventFilter(self)

    def build_tree(self, data):
        
        self.tree_model = tree_model = QStandardItemModel()
        root_node = tree_model.invisibleRootItem()

        self.setModel(tree_model)

        self.populate_items(data, root_node, 1)

    def populate_items(self, data, parent_node, level, parent_non_selectable=True):
        for k, v in data.items():
            node = QStandardItem(k)
            parent_node.appendRow([node])
            if v:
                if parent_non_selectable:
                    node.setSelectable(False)
                self.populate_items(v, node, level+1)

    def showPopup(self):
        self.setRootModelIndex(QModelIndex())
        super().showPopup()

    def hidePopup(self):
        self.setRootModelIndex(self.view().currentIndex().parent())
        self.setCurrentIndex(self.view().currentIndex().row())
        if self.__skip_next_hide:
            self.__skip_next_hide = False
        else:
            super().hidePopup()

    def selectIndex(self, index):
        self.setRootModelIndex(index.parent())
        self.setCurrentIndex(index.row())

    def eventFilter(self, object, event):
        if event.type() == QEvent.MouseButtonPress and object is self.view().viewport():
            index = self.view().indexAt(event.pos())
            self.__skip_next_hide = not self.view().visualRect(index).contains(event.pos())
        return False

class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action

        layout = QVBoxLayout(self)
        self.setLayout(layout)

#        timer_group_box = QGroupBox('Timer event interval in seconds')
#        timer_group_box_l = QVBoxLayout()
#        timer_group_box.setLayout(timer_group_box_l)
#        layout.addWidget(timer_group_box)
#        self.timer_spin = QSpinBox()
#        self.timer_spin.setMaximum(100000)
#        self.timer_spin.setMinimum(60)
#        self.timer_spin.setValue(cfg.plugin_prefs[cfg.TIMER_EVENT_TIMEOUT])
#        timer_group_box_l.addWidget(self.timer_spin)
        
        chains_button = QPushButton(_('Add/Modify Chains'))
        layout.addWidget(chains_button)
        chains_button.clicked.connect(self.plugin_action.add_chains)

        events_button = QPushButton(_('&Event Manager')+'...')
        layout.addWidget(events_button)
        events_button.clicked.connect(self.plugin_action.event_manager)

        modules_button = QPushButton(_('&Manage Modules')+'...')
        layout.addWidget(modules_button)
        modules_button.clicked.connect(self.plugin_action.manage_modules)

        keyboard_button = QPushButton(_('Keyboard shortcuts...'), self)
        keyboard_button.setToolTip(_(
                    _('Edit the keyboard shortcuts associated with this plugin')))
        layout.addWidget(keyboard_button)
        keyboard_button.clicked.connect(self.plugin_action.edit_shortcuts)

        layout.addStretch(1)

    def save_settings(self):
        #cfg.plugin_prefs[cfg.TIMER_EVENT_TIMEOUT] = self.timer_spin.value()
        pass
