#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2023, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os

from qt.core import QFileDialog

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, info_dialog, choose_files, FileDialog
from calibre.utils.config import JSONConfig
from calibre.utils.zipfile import ZipFile
import calibre_plugins.editor_chains.config as cfg


def pick_archive_name_to_export(gui):
    fd = FileDialog(
        name='zip archive dialog',
        title=_('Save archive as'),
        filters=[('Settings Files', ['zip'])],
        parent=gui,
        add_all_files_filter=False,
        mode=QFileDialog.FileMode.AnyFile
    )
    fd.setParent(None)
    if not fd.accepted:
        return None
    f = fd.get_files()[0]
    if not f.lower().endswith('.zip'):
        f += '.zip'
    return f

def pick_archive_to_import(gui):
    archives = choose_files(
        gui,
        'zip archive dialog',
        _('Select a chain file archive to import'),
        filters=[('Settings Files', ['zip'])],
        all_files=False,
        select_only_single_file=True
    )
    if not archives:
        return
    f = archives[0]
    return f

def chain_image_resources(chain_config):
    images = set()
    chain_icon = chain_config.get('image')
    images.add(chain_icon)
    chain_condition_icon = chain_config.get('chain_settings', {}).get('condition_settings', {}).get('icon')
    images.add(chain_condition_icon)
    for chain_link in chain_config.get('chain_settings', {}).get('chain_links', []):
        chain_link_condition_icon = chain_condition_icon = chain_link.get('condition_settings', {}).get('icon')
        images.add(chain_condition_icon)
    try:
        images.remove(None)
    except:
        pass
    return images


def export_chains(plugin_action, gui, chains_config):
    if len(chains_config) == 0:
        return error_dialog(
            gui,
            _('Cannot export'),
            _('No chains selected to export.'),
            show=True
        )
    archive_path = pick_archive_name_to_export(gui)
    if not archive_path:
        return
    # Build our unique list of images that need to be exported
    all_image_names = {}
    for chain_config in chains_config:
        chain_images = chain_image_resources(chain_config)
        for image_name in chain_images:
            if image_name and image_name not in all_image_names:
                image_path = os.path.join(
                    plugin_action.resources_dir,
                    image_name)
                if os.path.exists(image_path):
                    all_image_names[image_name] = image_path
    # Write our menu items out to a json file
    if not os.path.exists(plugin_action.resources_dir):
        os.makedirs(plugin_action.resources_dir)
    archive_config = JSONConfig('resources/images/editor_chains')
    export_chains = {}
    export_chains[cfg.KEY_MENUS] = chains_config
    archive_config.set(cfg.STORE_MENUS_NAME, export_chains)
    archive_config.set(cfg.KEY_SCHEMA_VERSION, cfg.DEFAULT_SCHEMA_VERSION)
    json_path = os.path.join(
        plugin_action.resources_dir,
        'editor_chains.json')

    try:
        # Create the zip file archive
        with ZipFile(archive_path, 'w') as archive_zip:
            archive_zip.write(json_path, os.path.basename(json_path))
            # Add any images referred to in those menu items that are local resources
            for image_name, image_path in list(all_image_names.items()):
                archive_zip.write(image_path, os.path.basename(image_path))
        info_dialog(
            gui,
            _('Export completed'),
            _(f'{len(chains_config)} chains exported to<br>{archive_path}'),
            show=True, show_copy_button=False
        )
    finally:
        if os.path.exists(json_path):
            os.remove(json_path)

def chains_config_from_archive(gui, plugin_action, archive_path=None, add_resources=False,
                        overwrite=False):
    if not archive_path:
        archive_path = pick_archive_to_import(gui)
    if not archive_path:
        return
    with ZipFile(archive_path, 'r') as zf:
        contents = zf.namelist()
        if 'editor_chains.json' not in contents:
            return error_dialog(
                gui,
                _('Import Failed'),
                _('This is not a valid Editor Chains export archive'),
                show=True
            )
        if not add_resources:
            resources = ['editor_chains.json']
        else:
            resources = contents
        if not os.path.exists(plugin_action.resources_dir):
            os.makedirs(plugin_action.resources_dir)
        for resource in resources:
            fs = os.path.join(plugin_action.resources_dir, resource)
            if os.path.exists(fs) and not overwrite:
                continue
            with open(fs,'wb') as f:
                f.write(zf.read(resource))
    json_path = os.path.join(plugin_action.resources_dir,'editor_chains.json')
    try:
        # Read the .JSON file.
        archive_config = JSONConfig('resources/images/editor_chains')
        cfg.migrate_plugin_prefs_if_required(archive_config, commit=False)
        chains_config = archive_config.get(cfg.STORE_MENUS_NAME).get(cfg.KEY_MENUS)
    finally:
        if os.path.exists(json_path):
            os.remove(json_path)
    return chains_config
