#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import re
import copy
import regex

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.base import etree
from calibre.ebooks.html_transform_rules import parse_start_tag, parse_html_snippet, clone

try:
    load_translations()
except NameError:
    prints("EditorChains::etree.py - exception when loading translations")

#def get_text(elem, include_tail=False):
    #t = elem.text or ''
    #for ch in elem:
        #t += get_text(ch, include_tail=True)
    #if len(elem) > 0:
        #t += elem[-1].tail or ''
    #if include_tail:
        #t += elem.tail or ''
    #return t

def get_text(elem):
    return ''.join(elem.itertext())

def get_tail(element, ignore_whitespace=False):
    tail = element.tail or ''
    if ignore_whitespace:
        tail = tail.strip()
    return tail

def append_text(element, text):
    tail = get_tail(element, ignore_whitespace=False)
    element.tail = tail + text

def prepend_text(element, text):
    tail = get_tail(element, ignore_whitespace=False)
    element.tail = text + tail

def get_string(element, ignore_whitespace=False):
    string = element.text or ''
    if ignore_whitespace:
        string = string.strip()
    return string

def remove_duplicate_classes(element):
    classes = element.attrib.get("class", "").split()
    new_classes = []
    for cls in classes:
        if not cls in new_classes:
            new_classes.append(cls)
    element.attrib["class"] = " ".join(new_classes)

def extract(element, include_tail=False):
    parent = element.getparent()
    if parent is not None:
        if not include_tail:
            if element.tail:
                previous = element.getprevious()
                if previous is not None:
                    append_text(previous, get_tail(element))
                else:
                    parent.text = get_string(parent) + get_tail(element)

        idx = parent.index(element)
        extracted = copy.deepcopy(parent[idx])
        del parent[idx]
        return extracted

def append_snippet(container, before_tag, tag, before_tail=False):
    p = tag.getparent()
    idx = p.index(tag)
    if not before_tag:
        idx += 1
        if before_tail:
            orig_tail = tag.tail
            tag.tail = None
            if len(container) > 0:
                container[-1].tail = (container[-1].tail or '') + (orig_tail or '')
            else:
                container.text = (container.text or '') + (orig_tail or '')
    if container.text:
        if idx:
            c = p[idx-1]
            c.tail = (c.tail or '') + container.text
        else:
            p.text = (p.text or '') + container.text
    for child in reversed(container):
        c = clone(child, tag)
        p.insert(idx, c)
    return True

def insert_after(tag, element_to_insert, before_tail=True):
    container = parse_html_snippet(element_to_insert)
    return append_snippet(container, False, tag, before_tail=before_tail)

def insert_before(tag, element_to_insert):
    container = parse_html_snippet(element_to_insert)
    return append_snippet(container, True, tag)

def wrap(elements_to_wrap, wrapper, insert_between=None, include_last_element_tail=False):
    start_tag = regex.sub(r'</.[^>]*>', r'', wrapper)
    data = parse_start_tag(start_tag)
    elem = elements_to_wrap[0].makeelement(data['tag'])
    for k, v in data['attrib']:
        elem.set(k, v)
    if not include_last_element_tail:
        elem.tail = elements_to_wrap[-1].tail
        elements_to_wrap[-1].tail = None
    p = elements_to_wrap[0].getparent()
    idx = p.index(elements_to_wrap[0])
    p.insert(idx, elem)
    for e in elements_to_wrap:
        elem.append(e)
        if (insert_between is not None) and (e is not elements_to_wrap[-1]):
            container = parse_html_snippet(insert_between)
            append_snippet(container, False, e)
    return elem

def replace_with(tag, replacement, keep_tail=True):
    '''
    Unlike BeautifulSoup, this only works for elements not text or tail
    '''
    if keep_tail:
        replacement += tag.tail or ''
    container = parse_html_snippet(replacement)
    l = [x for x in container]
    parent = tag.getparent()
    idx  = parent.index(tag)
    append_snippet(container, True, tag, before_tail=True)
    parent.remove(tag)
    return l
