#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import defaultdict
from functools import partial
from itertools import count
from operator import itemgetter
import re

from css_parser.css import CSSStyleSheet, CSSRule, Property

from css_selectors import Select, INAPPROPRIATE_PSEUDO_CLASSES, SelectorError
from calibre import as_unicode
from calibre.ebooks.css_transform_rules import all_properties
from calibre.ebooks.oeb.base import OEB_DOCS, OEB_STYLES, XHTML, css_text
from calibre.ebooks.oeb.stylizer import media_ok
from polyglot.builtins import iteritems, itervalues, unicode_type


from calibre.ebooks.oeb.polish.cascade import (html_css_stylesheet, iterrules, StyleDeclaration, specificity,
                                              normalize_style_declaration, resolve_declarations, Specificity,
                                              resolve_pseudo_declarations, resolve_property)

def get_sheet_styles_maps(container, name, select=None,
                    sheet_callback=None, include_inline_styles=True):
    root = container.parsed(name)
    select = select or Select(root, ignore_inappropriate_pseudo_classes=True)
    style_map = defaultdict(list)
    pseudo_style_map = defaultdict(list)
    rule_index_counter = count()
    pseudo_pat = re.compile(':{1,2}(%s)' % ('|'.join(INAPPROPRIATE_PSEUDO_CLASSES)), re.I)

    def process_sheet(sheet, sheet_name):
        if sheet_callback is not None:
            sheet_callback(sheet, sheet_name)
        for rule, sheet_name, rule_index in iterrules(container, sheet_name, rules=sheet, rule_index_counter=rule_index_counter, rule_type='STYLE_RULE'):
            for selector in rule.selectorList:
                text = selector.selectorText
                try:
                    matches = tuple(select(text))
                except SelectorError as err:
                    container.log.error(f'Ignoring CSS rule with invalid selector: {text!r} ({as_unicode(err)})')
                    continue
                m = pseudo_pat.search(text)
                style = normalize_style_declaration(rule.style, sheet_name)
                if m is None:
                    for elem in matches:
                        style_map[elem].append(StyleDeclaration(specificity(rule_index, selector), style, None))
                else:
                    for elem in matches:
                        pseudo_style_map[elem].append(StyleDeclaration(specificity(rule_index, selector), style, m.group(1)))

    process_sheet(html_css_stylesheet(container), 'user-agent.css')

    for elem in root.iterdescendants(XHTML('style'), XHTML('link')):
        if elem.tag.lower().endswith('style'):
            if not elem.text:
                continue
            sheet = container.parse_css(elem.text)
            sheet_name = name
        else:
            if (elem.get('type') or 'text/css').lower() not in OEB_STYLES or \
                    (elem.get('rel') or 'stylesheet').lower() != 'stylesheet' or \
                    not media_ok(elem.get('media')):
                continue
            href = elem.get('href')
            if not href:
                continue
            sheet_name = container.href_to_name(href, name)
            if not container.has_name(sheet_name):
                continue
            sheet = container.parsed(sheet_name)
            if not isinstance(sheet, CSSStyleSheet):
                continue
        process_sheet(sheet, sheet_name)

    if include_inline_styles:
        for elem in root.xpath('//*[@style]'):
            text = elem.get('style')
            if text:
                style = container.parse_css(text, is_declaration=True)
                style_map[elem].append(StyleDeclaration(Specificity(1, 0, 0, 0, 0), normalize_style_declaration(style, name), None))

    for l in (style_map, pseudo_style_map):
        for x in itervalues(l):
            x.sort(key=itemgetter(0), reverse=True)

    style_map = {elem:resolve_declarations(x) for elem, x in iteritems(style_map)}
    pseudo_style_map = {elem:resolve_pseudo_declarations(x) for elem, x in iteritems(pseudo_style_map)}

    return style_map, pseudo_style_map


def get_style_maps(container, include_inline_styles=True):
    style_map = defaultdict(list)
    pseudo_style_map = defaultdict(list)

    for name, media_type in container.mime_map.items():
        if media_type in OEB_DOCS:
            sheet_style_map, sheet_pseudo_style_map = get_sheet_styles_maps(container, name, include_inline_styles=include_inline_styles)
            style_map.update(sheet_style_map)
            pseudo_style_map.update(sheet_pseudo_style_map)
    return style_map, pseudo_style_map
