#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from calibre.utils.config import JSONConfig

KEY_SCHEMA_VERSION = 'schemaVersion'
DEFAULT_SCHEMA_VERSION = 1.1
STORE_MENUS_NAME = 'EditorChainMenus'
KEY_MENUS = 'Menus'
KEY_GPREFS = 'gprefs'
KEY_CHAINS_TABLE_STATE = 'chainsTableState'
KEY_ACTIONS_TABLE_STATE = 'actionsTableState'

DEFAULT_VALUES = {
    STORE_MENUS_NAME: {
        KEY_MENUS: []
    },
    KEY_GPREFS: {
        KEY_CHAINS_TABLE_STATE: None,
        KEY_ACTIONS_TABLE_STATE: None
    }
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Editor Chains')

## Set defaults
#plugin_prefs.defaults[STORE_MENUS_NAME] = DEFAULT_VALUES

PLUGIN_ICONS = [
    'editor_chains.png',
    'image_add.png',
    'import.png',
    'export.png',
    'multiple.png'
]

def get_missing_values_from_defaults(default_settings, settings):
    '''add keys present in default_settings and absent in setting'''
    for k, default_value in default_settings.items():
        try:
            setting_value = settings[k]
            if isinstance(default_value, dict):
                get_missing_values_from_defaults(default_value, setting_value)
        except KeyError:
            settings[k] = copy.deepcopy(default_value)

def migrate_plugin_prefs_if_required(plugin_prefs, commit=True):
    schema_version = plugin_prefs.get(KEY_SCHEMA_VERSION, 0)
    if schema_version == DEFAULT_SCHEMA_VERSION:
        return

    # We have changes to be made - mark schema as updated
    plugin_prefs[KEY_SCHEMA_VERSION] = DEFAULT_SCHEMA_VERSION

    # Any migration code in future will exist in here.
    if schema_version < 1.0:
        pass

    if schema_version < 1.1:

        def rename_process_scope(dictionary):
            # rename dictionary['scope'] > dictionary['where'] {{{
            changed = False
            scope = dictionary.get('where')
            if not scope:
                scope = dictionary.get('scope')
                if scope:
                    del dictionary['scope']
                    dictionary['where'] = scope
                    scope = dictionary.get('where')
                    changed = True
            # }}}
            if scope and not isinstance(scope, str):
                where, extra = scope
                if where == 'include':
                    dictionary['where'] = ['regex', {'include': extra, 'exclude': ''}]
                    changed = True
                if where == 'exclude':
                    dictionary['where'] = ['regex', {'include': '', 'exclude': extra}]
                    changed = True
            return changed

        c = plugin_prefs[STORE_MENUS_NAME]
        chains = c.get(KEY_MENUS, [])
        for chain in chains:
            chain_name = chain.get('menuText','')
            chain_settings = chain.get('chain_settings', {})
            chain_links = chain_settings.get('chain_links', [])
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                action_settings = chain_link.get('action_settings', {})
                if action_name.lower() == 'add templates':
                    # Rename the action
                    chain_link['action_name'] = 'Add Files'
                    #
                    if action_settings.get('rename') is not None:
                        if not action_settings.get('if_name_exists'):
                            action_settings['if_name_exists'] = 'rename'
                            del action_settings['rename']
                    action_settings['is_file'] = True
                    chain_link['action_settings'] = action_settings
                #
                changed = rename_process_scope(action_settings)
                if changed:
                    chain_link['action_settings'] = action_settings
                if action_name.lower() == 'search & replace':
                    searches = action_settings.get('searches')
                    if searches:
                        searches_changed = False
                        for search in searches:
                            changed = rename_process_scope(search)
                            if changed:
                                searches_changed = True
                    if searches_changed:
                        chain_link['action_settings'] = action_settings

    # Update: add defaults for new keys {
    get_missing_values_from_defaults(DEFAULT_VALUES, plugin_prefs)
    if commit:
        plugin_prefs.commit()

if not plugin_prefs:
    get_missing_values_from_defaults(DEFAULT_VALUES, plugin_prefs)
migrate_plugin_prefs_if_required(plugin_prefs)

def get_chains_config(exclude_separators=True):
    c = plugin_prefs[STORE_MENUS_NAME]
    chains = c.get(KEY_MENUS, [])
    if exclude_separators:
        chains = [chain for chain in chains if chain.get('menuText')]
    return chains

def get_chain_config(chain_name, chains_config=[]):
    if not chains_config:
        chains_config = get_chains_config()
    for chain_config in chains_config:
        name = chain_config['menuText']
        if name == chain_name:
            return chain_config
    return {}
    
