#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
import copy
import time
from collections import defaultdict
from uuid import uuid4

from qt.core import QApplication, Qt, QModelIndex

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.utils.date import now, format_date
from calibre.ebooks.oeb.polish.container import get_container
from calibre.gui2.tweak_book import current_container
from calibre.gui2.tweak_book.polish import show_report
#from calibre.gui2.tweak_book.save import save_container

import calibre_plugins.editor_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("EditorChains::chains.py - exception when loading translations")

class Chain(object):

    class UserInterrupt(Exception):
        pass

    def __init__(
        self,
        plugin_action,
        chain_config,
        tool,
        gui=None,
        boss=None,
        book_path=None,
        qt_processEvents=True,
        show_progress=True,
        add_savepoint=True,
        container=None,
        chain_vars={},
        is_chain_caller=False,
        book_id=None,
        report=''
    ):
        self.plugin_action = plugin_action
        self.qt_processEvents = qt_processEvents
        self.gui = gui
        self.boss = boss
        self.tool = tool
        self.book_path = book_path
        self.show_progress = show_progress
        self.add_savepoint = add_savepoint
        self.book_id = book_id
        # copy as to not overwrite mutable chain data
        self.chain_config = copy.deepcopy(chain_config)
        self.chain_name = self.chain_config['menuText']
        self.chain_links = self.chain_config['chain_settings'].get('chain_links')
        self.start_time = now()
        self.step_length = 1
        self.current_step = -1
        self.last_step = None
        self.pbar = None
        self.container = container
        self.is_chain_caller = is_chain_caller
        self.report = report
        self.uuid = uuid4()

        self._initialize_vars(chain_vars=chain_vars)

    def validate(self):
        all_actions = self.plugin_action.actions
        msg = _('Chain Error')
        details = _('Validating the chain settings before running '
                    'failed. You can see the detailed errors by '
                    'opening the chain in the config dialog')
        try:
            chain_name = self.chain_config['menuText']
            chain_settings = self.chain_config['chain_settings']
            chain_links = chain_settings.get('chain_links',[])
            if len(chain_links) == 0:
                return (
                    _('Empty chain'),
                    _('A chain must contain at least one action')
                )
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                if action_name in all_actions.keys():
                    action = all_actions[action_name]
                    action_settings = chain_link.get('action_settings')
                    if not action_settings:
                        action_settings = action.default_settings()
                    is_valid = action.validate(action_settings)
                    if is_valid is not True:
                        if DEBUG:
                            prints(f'Editor Chains: chain ({chain_name}): action ({action_name}) settings are not valid')
                        return msg, details
                    # NOTE: When trying to edit chains from action chains,
                    # the action will do its own checking of supported formats in
                    # based on book record in calibre library
                    if self.gui and self.current_container:
                        ac_fmts = action.supported_formats(action_settings)
                        book_type = self.current_container.book_type
                        if book_type not in ac_fmts:
                            print(f'Editor Chains: chain ({chain_name}): action ({action_name}) does not support current book format: {book_type}')
                            return _('Format not supported by chain'), _('Some of the actions of this chain does not support the current book format')
                else:
                    if DEBUG:
                        print(f'Editor Chains: chain ({chain_name}): action ({action_name}) is not currently available')
                    return msg, details
        except Exception as e:
            if DEBUG:
                import traceback
                print(f'Editor Chains: Exception when checking chain:\nchain: {self.chain_config}\nexception: {traceback.format_exc()}')
                raise e
        return True

    def _run_loop(self):
        count = len(self.chain_links)
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:

            #
            # chain_start_time has to be reset here in case someone initializes this object once, 
            # and and runs the loop mulitple times
            self.chain_start_time = now()            
            # initiate previous action start time, will be reset after each action
            self.previous_action_start_time = self.chain_start_time
            
            self.set_chain_vars({'_chain_start_time': format_date(self.chain_start_time, 'iso')})

            # Add to stack. Non empty stack will block event manager
            self.plugin_action.chainStack.append(self.chain_name)
            #
            
            if DEBUG:
                prints(f'Editor Chains: Starting chain: {self.chain_name}')

            chain_iteration = getattr(self, 'chain_iteration', 1)

            try:
                container = self.current_container
                container.book_id = self.book_id
                if self.gui:
                    if not self.boss.ensure_book():
                        return

                    # Ensure any in progress editing the user is doing is present in the container
                    self.boss.commit_all_editors_to_container()
                    
                    if self.add_savepoint:
                        self.boss.add_savepoint(f'Before: Editor Chains ({self.chain_name} [{chain_iteration}])')
            except FileNotFoundError as e:
                if DEBUG:
                    prints(f'Editor Chains: {self.book_path} cannot be found. Aborting ...')
                    return
            except PermissionError as e:
                if DEBUG:
                    prints(f'Editor Chains: {self.book_path} is not writable file. Aborting ...')
                    return

            while True:
                self.last_step = self.current_step
                self.current_step += self.step_length if self.step_length > 0 else 1
                # now that we read the step lenght that might have been modified by the previous action, restore it back
                self.step_length = 1
                if self.current_step > len(self.chain_links) - 1:
                    break
                #
                chain_link = self.chain_links[self.current_step]
                # store start time for each chain link, used by some actions to change selections based on last_modified date.
                chain_link['start_time'] = now()
                action_name = chain_link['action_name']
                action = self.plugin_action.actions[action_name]
                action_settings = chain_link.get('action_settings')
                if not action_settings:
                    action_settings = action.default_settings()

                self.update_vars_for_action(chain_link)                
                # test conditions for running action
                #do_run = self.check_action_conditions(chain_link.get('condition_settings', {}))
                do_run = True
                #
                try:
                    if do_run:
                        if DEBUG:
                            prints(f'Editor Chains: {self.chain_name}: starting action No. {self.current_step + 1}: {action_name}')
                        action.run(self, action_settings)
                    else:
                        if DEBUG:
                            prints(f'Editor Chains: {self.chain_name}: Conditions are not met for action No. {self.current_step + 1} ({action_name})')

                    #
                    # This is set false when called from library view double clicked event as it
                    # makes the active cell enter into edit mode.
                    if self.qt_processEvents:
                        QApplication.processEvents()
                except self.UserInterrupt:
                    if DEBUG:
                        prints(f'Editor Chains: Aborting chain ({self.chain_name})')
                    if self.gui:
                        self.boss.rewind_savepoint()
                    break
                except Exception as e:
                    if self.gui:
                        self.boss.rewind_savepoint()
                    if DEBUG:
                        prints(f'Editor Chains: action "{action_name}" terminated with exception: {e}')
                    import traceback
                    traceback.print_exc()
                    raise e
                finally:
                    # reset previous action start time
                    self.previous_action_start_time = chain_link['start_time']

            self.refresh_gui()
            if self.boss:
                self.boss.apply_container_update_to_gui()
            else:
                # Headless mode
                container.commit()

            end = now()
            if DEBUG:
                prints(f'Editor Chains: chain ({self.chain_name}) finished in: {end - self.chain_start_time}')
            if self.report and self.settings.get('show_report', False) and not self.is_chain_caller:
                if self.gui:
                    show_report(bool(self.report), self.boss.current_metadata.title, self.report, self.gui, self.show_current_diff)
                else:
                    print(self.report)
        finally:
            QApplication.restoreOverrideCursor()
            self.plugin_action.chainStack.pop(-1)
            # re-initialize chain vars for an possible next runs(in case the
            # chain oject initialized by extrenal action or plugin and deleted)
            self._initialize_vars()
            # persist the chain_iteration var, we defered to this point to avoid
            # test initializations adding to iterations
            self.plugin_action.chain_iterations[self.chain_name] = chain_iteration


    def create_container(self): 
        if not self.book_path:
            raise FileNotFoundError
        if not os.access(self.book_path, os.W_OK):
            raise PermissionError
        #container = get_container(self.book_path, tweak_mode=False)
        container = get_container(self.book_path)
        return container

    @property
    def current_container(self):
        if self.gui:
            container = self.tool.current_container
            return container
        else:
            if not self.container:
                self.container =  self.create_container()
            return self.container


    def _initialize_vars(self, chain_vars={}):
        #
        self.chain_vars = chain_vars
        # chain iteration is saved, will be presisted only if the chain runs
        self.chain_iteration = self.plugin_action.chain_iterations[self.chain_name] + 1
        _book_vars = chain_vars.get('_ac_chain_vars', {}).get('_book_vars', defaultdict(dict))
        vars_dict = {
            '_chain_name': self.chain_name,
            '_chain_iteration': str(self.chain_iteration),
            '_book_vars': _book_vars
        }
        self.set_chain_vars(vars_dict)

    def set_chain_vars(self, vars_dict):
        for k, v in vars_dict.items():
            if k.startswith('_') or ( isinstance(k, (str)) and isinstance(v, (str)) ):
                self.chain_vars[k] = v
            else:
                if DEBUG:
                    prints(f'Editor Chains: variable ({k}:{v}) cannot be set, variables must be of string type')

    def update_vars_for_action(self, chain_link):
        vars_dict = {
            '_action_name': chain_link['action_name'],
            '_action_comment': chain_link.get('comment', ''),
            '_action_index': str(self.current_step),
            '_previous_action_start_time': format_date(self.previous_action_start_time, 'iso')
        }
        self.set_chain_vars(vars_dict)
        self.chain_vars['_action_vars'] = {}

    @property
    def action_vars(self):
        return self.chain_vars['_action_vars']

    def get_book_var(self, var_name, default=None):
        return self.chain_vars['_book_vars'].get(self.book_id, {}).get(var_name, default)

    def set_book_var(self, var_name, var_value):
        self.chain_vars['_book_vars'][self.book_id][var_name] = var_value

    def refresh_gui(self):
        pass

    def print_error(self, msg, details, show=True):
        if self.gui:
                error_dialog(
                    self.gui,
                    msg,
                    details,
                    show=True
                )
        else:
            if DEBUG:
                prints(f'Editor Chains: {msg}: {details}')          

    def add_to_report(self, data):
        if data:
            action_name = self.action_vars["_name"]
            action_idx = self.action_vars["_action_index"]
            prefix = f'{self.chain_name}: {action_name}: {action_idx}'
            prefix += f'\n{len(prefix) * "="}\n'
            self.report += '\n\n' + prefix + data

    def run(
        self,
        check_conditions=True,
        print_errors=False
    ):
        # check chains for errors
        is_chain_valid = self.validate()
        if is_chain_valid is not True:
            msg, details = is_chain_valid
            if print_errors:
                return self.print_error(
                    msg,
                    details,
                    show=True
                )
            else:
                if DEBUG:
                    print(f'Editor Chains: {msg}: {details}')
                    return

        self._run_loop()
