#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from collections import OrderedDict, defaultdict
import copy

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout, QStackedLayout,
                     QLabel, QGroupBox, QToolButton, QPushButton, QScrollArea, QComboBox, QFrame,
                     QDialog, QDialogButtonBox, QTableWidget, QAbstractItemView, QCheckBox,
                     QIcon, QSizePolicy, pyqtSignal, QSize, QSpinBox, QRadioButton, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.widgets import Dialog
from calibre.gui2.convert.xpath_wizard import Wizard

from calibre_plugins.editor_chains.common_utils import get_icon, get_pixmap, ViewLog, reverse_lookup
from calibre_plugins.editor_chains.actions.tag_actions.filters.group import GroupWidget
from calibre_plugins.editor_chains.scope import ScopeWidget

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/gui/__init__.py - exception when loading translations")


SCOPE_TOOLTIP = '<style>dd {margin-bottom: 1.5ex}</style>' + _(
'''
Which files to perform this action on:
<dl>
<dt><b>Current file</b></dt>
<dd>Search only inside the currently opened file</dd>
<dt><b>All text files</b></dt>
<dd>Search in all text (HTML) files</dd>
<dt><b>Selected files</b></dt>
<dd>Search in the files currently selected in the File browser</dd>
<dt><b>Open files</b></dt>
<dd>Search in the files currently open in the editor</dd>
<dt><b>Include filenames</b></dt>
<dd>Comma separated list of filenames to include (regex).</dd>
<dt><b>Exclude filenames</b></dt>
<dd>Comma separated list of filenames to exclude (regex).</dd>
</dl>''')


TAG_SELECTION_TOOLTIP = '<style>dd {margin-bottom: 1.5ex}</style>' + _(
'''
<dl>
<dt><b>I. Individual Tag(s)</b></dt>
<dd>The default option where you select single tag(s) based on criteria, and apply action<br/>
on them individually</dd>
<dt><b>II. Range(s) of Tags</b></dt>
<dd>Here you can choose range(s) of tags, where you will be able to apply actions on<br/>
them collectively. e.g wrap them in another tag. To be able to select a range, you need<br/>
to set criteria of two tags:</dd>
<ol>
<ol>
<li><b>First tag in the range</b></li>
<li><b>Last tag in the range</b></li>
</ol>
Also there two optional settings to abort the range, even if you are able to match<br/>
the first and last tags:
<ol>
<li><b>(Optional) Abort:</b> Here you specify criteria for tag that if found between the first and last <br/>
tags, the range broken and not selected</dd>
</li>
<li><b>(Optional) Max Steps:</b> The maximum number of tags between the first and last tag. If actual <br/>
number in a range is higher than the value specified here, the range will not be selected.</li>
</ol>
</ol>
</dl>

<dl>
<dt><b>Info on Tag Criteria</b></dt>
<dd>Opens a dialog for you to select criteria by which to select tags. It allows for <br/>
boolean or/and criteria, or even nested groups of or/any (by choosing "match all"/"match any")<br/>
from the dropdown menu.
Here is a list of the filters by which you can match a tag:
<ol>
<li>Tag Name: Allows for regular expression or exact matching of tag name.</li>
<li>Attribute: Allows for regular expression or exact matching of different attributes.<br/>
For multivalue attributes like classes, you can also match a single class by choosing<br/>
the contain option from the dropdown.</li>
<li>Text: This can apply to the text of tag itself. Or the text immediately before or<br/>
after i.e. The tail of the previous or next sibling if any</li>
<li>Related Tag: Allows you choose a tag by criteria of its relative tag. e.g. a tag<br/>
that has a child tag with certain class name.</li>
<li>Style Property: Allows you choose a tag based on its style e.g. font-size, color .... etc</li>
<li>Code: If you know how to code in python, you can refine your criteria by using this filter.</li>
</ol>
</dd>
</dl>''')

ACTIONS_TOOLTIP = '<style>dd {margin-bottom: 1.5ex}</style>' + _(
'''
Perform one of the following actions on the selected tags/ranges:
<ol>
<li><b>Modify Tag:</b> Allows you modify a tag in a variety of way:</li>
<ul>
<li>Modify tag name</li>
<li>Delete classes or other attributes</li>
<li>Add or append new classes</li>
<li>Add new attributes</li>
</ul>
<li><b>Delete:</b> Delete tag(s) and its contents</li>
<li><b>Wrap:</b> Wrap the tag(s) in a new tag</li>
<li><b>Unwrap:</b> Delete only the tag but keeping its text</li>
<li><b>Insert Before:</b> Insert a text or tag before</li>
<li><b>Insert After:</b> Insert a text or tag after</li>
<li><b>Search and Replace:</b> Perform a search and replace on the on the whole<br/>
element (tag and text). This supports regex and calibre's replace function</li>
<li><b>Code: </b>If you can program in python, you can use this to apply your custom action</li>
</ol>
''')

class XpathLineEditor(QLineEdit):

    '''
    Extend the context menu of a QLineEdit to include more actions.
    '''

    def __init__(self, parent):
        QLineEdit.__init__(self, parent)
        self.mi   = None
        self.setClearButtonEnabled(True)

    def contextMenuEvent(self, event):
        menu = self.createStandardContextMenu()
        menu.addSeparator()

        action_clear_field = menu.addAction(_('Remove any text from the box'))
        action_clear_field.triggered.connect(self.clear_field)
        action_open_editor = menu.addAction(_('Open Xpath Wizard'))
        action_open_editor.triggered.connect(self.open_editor)
        menu.exec(event.globalPos())

    def clear_field(self):
        self.setText('')

    def open_editor(self):
        wiz = Wizard(self)
        if wiz.exec() == wiz.Accepted:
            self.setText(wiz.xpath)

class FiltersDialog(Dialog):
    def __init__(self, parent, plugin_action, filters_config, title='Filters Dialog'):
        self.plugin_action = plugin_action
        self.filters = self.plugin_action.actions['Tag Actions'].filters
        self.filters_config = filters_config
        self.title = title
        Dialog.__init__(self, title, 'editor-chains-filters-dialog', parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)
        
        self.group = GroupWidget(self, self.plugin_action, self.filters, identifier=self.title)
        self.group.load_settings(self.filters_config)
        l.addWidget(self.group)
        
        l.addWidget(self.bb)
        self.setMinimumSize(500, 500)

    def reject(self):
        Dialog.reject(self)

    def accept(self):
        group_filter = self.plugin_action.actions['Tag Actions'].filters['Group']
        settings = self.group.save_settings()
        #all_valid = group_filter.validate(settings)
        #if all_valid is True:
            #self.filters_config = settings
            #Dialog.accept(self)
        #else:
            #return error_dialog(self, *all_valid, show=True)
        self.filters_config = settings
        Dialog.accept(self)

class ActionSettingsDialog(Dialog):
    def __init__(self, parent, plugin_action, action, action_config={}):
        self.action_config = action_config
        self.plugin_action = plugin_action
        self.action = action
        name = f'editor-chains-action-settings-dialog-{action.name}'
        Dialog.__init__(self, 'Action Settings Dialog', name, parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)

        config_widget_cls = self.action.config_widget()
        config_widget = config_widget_cls(self.plugin_action, self.action)
        l.addWidget(config_widget)
        config_widget.load_settings(self.action_config)  
        self.config_widget = config_widget
        
        l.addWidget(self.bb)

    def reject(self):
        Dialog.reject(self)

    def accept(self):
        new_settings = self.config_widget.save_settings()
        is_valid = self.action.validate(new_settings)
        if is_valid is True:
            self.action_config = new_settings
            Dialog.accept(self)
        else:
            return error_dialog(self, *is_valid, show=True)

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.tag_actions = plugin_action.actions['Tag Actions']
        self.filters = self.tag_actions.filters
        self.actions = self.tag_actions.actions
        self.action_settings_map = {}
        self.filters_config = {}
        self.last_filters_config = {}
        self.abort_filters_config = {}
        self.limit_nodes = OrderedDict()
        self.limit_nodes['all'] = _('Apply to all nodes')
        self.limit_nodes['xpath'] = _('Apply to nodes in xpath')
        self.limit_nodes['css'] = _('Apply to nodes matching CSS Selector')
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        main_label = QLabel(_('Using this action, you can choose tags or range of tags based on several '
                              'criteria, and then apply an action to them. You can learn more about each '
                              'option below by hovering over the widget for detailed tooltips'
        ))
        main_label.setWordWrap(True)
        l.addWidget(main_label)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        l.addStretch(1)
        lbl1 = QLabel(_('<b>Step 1:</b> Choose the HTML files'))
        lbl1.setWordWrap(True)
        l.addWidget(lbl1)
        scope_group_box = QGroupBox()
        #scope_group_box.setToolTip(SCOPE_TOOLTIP)
        l.addWidget(scope_group_box)
        scope_group_box_l = QHBoxLayout()
        scope_group_box.setLayout(scope_group_box_l)
        self.scope_widget = ScopeWidget(self, self.plugin_action,
                hide=['selected-text','styles'], show_tooltip=False,
                headless=self.plugin_action.gui is None)
        scope_group_box_l.addWidget(self.scope_widget)

        l.addStretch(1)
        lbl2 = QLabel(_('<b>Step 2:</b> Restrict the nodes that will be searched'))
        lbl2.setWordWrap(True)
        l.addWidget(lbl2)
        limits_gb = QGroupBox()
        limits_l = QHBoxLayout()
        limits_gb.setLayout(limits_l)
        l.addWidget(limits_gb)
        self.limit_box = QComboBox(self)
        self.limit_box.activated.connect(self.toggle_limit_box)
        self.limit_box.addItems(self.limit_nodes.values())
        self.limit_box.setCurrentText(self.limit_nodes['all'])
        limits_l.addWidget(self.limit_box)
        self.limits_sl = QStackedLayout()
        limits_l.addLayout(self.limits_sl, 0)
        self.empty_w_2 = QWidget()
        self.limits_sl.addWidget(self.empty_w_2)
        self.xpath_ledit = XpathLineEditor(self)
        self.limits_sl.addWidget(self.xpath_ledit)
        self.css_ledit = QLineEdit(self)
        self.limits_sl.addWidget(self.css_ledit)
        self.toggle_limit_box()

        l.addStretch(1)
        lbl3 = QLabel(_('<b>Step 3</b>: Select tags/range of tags from nodes trees specified above'))
        lbl3.setWordWrap(True)
        l.addWidget(lbl3)
        tag_group_box = QGroupBox()
        l.addWidget(tag_group_box)
        tag_group_box_l = QVBoxLayout()
        tag_group_box.setLayout(tag_group_box_l)
        tag_group_box.setToolTip(TAG_SELECTION_TOOLTIP)

        hl1 = QHBoxLayout()
        tag_group_box_l.addLayout(hl1)
        
        self.single_opt = QRadioButton('Select individual tags')
        self.single_opt.setChecked(True)
        hl1.addWidget(self.single_opt)

        self.range_opt = QRadioButton('Select ranges of tags')
        hl1.addWidget(self.range_opt)
        
        for opt in [self.single_opt, self.range_opt]:
            opt.toggled.connect(self._on_tag_opt_toggled)

        hl2 = QHBoxLayout()
        tag_group_box_l.addLayout(hl2)
        self.first_tag_button = QPushButton(_('Tag Criteria'))
        #tag_group_box_l.addWidget(self.first_tag_button)
        hl2.addWidget(self.first_tag_button)
        self.first_tag_button.clicked.connect(self._on_first_tag_button_clicked)

        self.range_group_widget = range_group_widget = QWidget()
        self.range_group_widget.setToolTip(_('Match a range of consecutive tags (siblings). You must also set '
                                    'criteria for last tag in range by pressing the button below'))

        last_tag_button = self.last_tag_button = QPushButton(_('Last Tag'))
        hl2.addWidget(last_tag_button)
        last_tag_button.setToolTip(_('Set criteria for last tag in the range.'))
        last_tag_button.clicked.connect(self._on_last_tag_button_clicked)

        vl1 = QVBoxLayout()
        tag_group_box_l.addLayout(vl1)
        abort_tag_button = self.abort_tag_button = QPushButton(_('Abort Range Criteria'))
        vl1.addWidget(abort_tag_button, 1)
        abort_tag_button.setToolTip(_('If tag with criteria set here is found, search for range will be aborted'))
        abort_tag_button.clicked.connect(self._on_abort_tag_button_clicked)

        hl2 = QHBoxLayout()
        vl1.addLayout(hl2)
        max_steps_label = self.max_steps_label = QLabel(_('Max steps'))
        hl2.addWidget(max_steps_label)
        self.max_steps_spin = QSpinBox()
        self.max_steps_spin.setMaximum(10000)
        self.max_steps_spin.setMinimum(0)
        self.max_steps_spin.setSingleStep(1)
        self.max_steps_spin.setValue(0)
        self.max_steps_spin.setToolTip('Maximum number of sibling tags after which the search '
                                       'for last tag in range should be aborted.'
                                       'choose 0 for unlimited steps.')
        hl2.addWidget(self.max_steps_spin, 1)

        tag_group_box_l.addWidget(self.range_group_widget)
        self._on_tag_opt_toggled()

        l.addStretch(1)
        lbl4 = QLabel(_('<b>Step 4:</b> Apply action on selected Tags/Ranges'))
        lbl4.setWordWrap(True)
        l.addWidget(lbl4)
        action_group_box = QGroupBox()
        l.addWidget(action_group_box)
        action_group_box_l = QHBoxLayout()
        action_group_box.setLayout(action_group_box_l)
        
        self.action_combo = QComboBox()
        self.action_combo.addItems(self.actions.keys())
        self.action_combo.setCurrentIndex(-1)
        self.action_combo.currentTextChanged.connect(self._on_action_combo_changed)
        action_group_box_l.addWidget(self.action_combo)
        
        self.action_settings_button = QToolButton()
        self.action_settings_button.setIcon(get_icon('gear.png'))
        self.action_settings_button.clicked.connect(self._on_action_settings_button_clicked)
        action_group_box_l.addWidget(self.action_settings_button)

        self.setMinimumSize(600,600)
        #l.addStretch(1)
        
        self._on_action_combo_changed(self.action_combo.currentText())

    def _on_first_tag_button_clicked(self):
        d = FiltersDialog(self, self.plugin_action, self.filters_config, title=_('Tag criteria'))
        if d.exec_() == d.Accepted:
            self.filters_config = d.filters_config

    def _on_last_tag_button_clicked(self):
        d = FiltersDialog(self, self.plugin_action, self.last_filters_config, title=_('Last tag criteria'))
        if d.exec_() == d.Accepted:
            self.last_filters_config = d.filters_config

    def _on_abort_tag_button_clicked(self):
        d = FiltersDialog(self, self.plugin_action, self.abort_filters_config, title=_('Abort criteria'))
        if d.exec_() == d.Accepted:
            self.abort_filters_config = d.filters_config

    def _on_action_settings_button_clicked(self):
        action_name = self.action_combo.currentText()
        if action_name:
            action = self.actions[action_name]
            action_config_widget = action.config_widget()
            if action_config_widget:
                action_settings = self.action_settings_map.get(action_name, {})
                if issubclass(action_config_widget, QDialog):
                    # config widget is a dialog
                    d = action_config_widget(self, self.plugin_action, action, action_settings)
                else:
                    # config_widget is a QWidget
                    d = ActionSettingsDialog(self, self.plugin_action, action, action_settings)
                if d.exec_() == d.Accepted:
                    self.action_settings_map[action_name] = d.action_config

    def _on_action_combo_changed(self, text):
        action_config_widget = False
        if text:
            action = self.actions[text]
            action_config_widget = action.config_widget()
        self.action_settings_button.setEnabled(bool(text) and bool(action_config_widget))

    def toggle_limit_box(self):
        self.limits_sl.setCurrentIndex(self.limit_box.currentIndex())

    def _on_tag_opt_toggled(self, *args):
        is_range = self.range_opt.isChecked()
        self.range_group_widget.setVisible(is_range)
        self.last_tag_button.setVisible(is_range)
        self.abort_tag_button.setVisible(is_range)
        self.max_steps_label.setVisible(is_range)
        self.max_steps_spin.setVisible(is_range)
        if is_range:
            self.first_tag_button.setToolTip(_('Set criteria for first tag in the range.'))
            self.first_tag_button.setText(_('First Tag Criteria'))
        else:
            self.first_tag_button.setToolTip(_('Set tag criteria'))
            self.first_tag_button.setText(_('Tag Criteria'))

    def load_settings(self, settings):
        if settings:
            action_name = settings['action_name']
            self.action_combo.setCurrentText(action_name)
            self.action_settings_map[action_name] = settings['action_settings']
            apply_filters_opt = settings.get('apply_filters_opt', 'all')
            if apply_filters_opt == 'xpath':
                self.limit_box.setCurrentText(self.limit_nodes['xpath'])
                self.xpath_ledit.setText(settings.get('xpath', ''))
            elif apply_filters_opt == 'css':
                self.limit_box.setCurrentText(self.limit_nodes['css'])
                self.css_ledit.setText(settings.get('css', ''))
            elif apply_filters_opt == 'all':
                self.limit_box.setCurrentText(self.limit_nodes['all'])
            self.toggle_limit_box()
            tag_selection = settings['tag_selection']
            if tag_selection == 'range':
                self.range_opt.setChecked(True)
            elif tag_selection == 'individual':
                self.single_opt.setChecked(True)
            self.filters_config = settings['filters_config']
            self.last_filters_config = settings['last_filters_config']
            self.abort_filters_config = settings['abort_filters_config']
            self.max_steps_spin.setValue(settings.get('max_steps', 0))
            self.scope_widget.where = settings.get('where', 'text')

    def save_settings(self):
        settings = {}
        action_name = self.action_combo.currentText()
        settings['action_name'] = action_name
        settings['action_settings'] = self.action_settings_map.get(action_name, {})
        apply_limit_opt = reverse_lookup(self.limit_nodes, self.limit_box.currentText())
        if apply_limit_opt == 'all':
            settings['apply_filters_opt'] = 'all'
        elif apply_limit_opt == 'xpath':
            settings['apply_filters_opt'] = 'xpath'
            settings['xpath'] = self.xpath_ledit.text().strip()
        elif apply_limit_opt == 'css':
            settings['apply_filters_opt'] = 'css'
            settings['css'] = self.css_ledit.text().strip()
        if self.single_opt.isChecked():
            settings['tag_selection'] = 'individual'
        elif self.range_opt.isChecked():
            settings['tag_selection'] = 'range'
        settings['max_steps'] = self.max_steps_spin.value()
        settings['filters_config'] = self.filters_config
        settings['last_filters_config'] = self.last_filters_config
        settings['abort_filters_config'] = self.abort_filters_config
        settings['where'] = self.scope_widget.where
        return settings
