#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree

from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter
from calibre_plugins.editor_chains.common_utils import reverse_lookup
from calibre_plugins.editor_chains.etree import get_text

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/text.py - exception when loading translations")

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        self.text_combo = QComboBox()
        self.text_combo.addItems(['text', 'text before', 'text after'])
        l.addWidget(self.text_combo, 0, 0, 1, 1)
        
        self.operator_combo = QComboBox()
        self.operator_combo.addItems(['matches', 'does not match', 'is null', 'not null'])
        l.addWidget(self.operator_combo, 0, 1, 1, 1)
        self.operator_combo.activated.connect(self._on_operator_changed)
        
        self.value_edit = QLineEdit()
        l.addWidget(self.value_edit, 0, 2, 1, 1)
        
        self.regex_chk = QCheckBox(_('Regex'))
        self.regex_chk.setToolTip(_('<b>Important</b>: This filter passes the DOTALL flag to regex'))
        l.addWidget(self.regex_chk, 0, 3, 1, 1)

        self._on_operator_changed()

    def _on_operator_changed(self, *args):
        operator = self.operator_combo.currentText()
        self.value_edit.setVisible(operator not in ['is null', 'not null'])
        self.regex_chk.setVisible(operator not in ['is null', 'not null'])

    def load_settings(self, settings):
        if settings:
            self.text_combo.setCurrentText(settings['text_type'])
            self.operator_combo.setCurrentText(settings['operator'])
            self.value_edit.setText(settings['value'])
            self.regex_chk.setChecked(settings['regex'])
            self._on_operator_changed()

    def save_settings(self):
        settings = {}
        settings['text_type'] = self.text_combo.currentText()
        settings['operator'] = self.operator_combo.currentText()
        settings['value'] = self.value_edit.text()
        settings['regex'] = self.regex_chk.isChecked()
        return settings


class TextFilter(ElementFilter):

    name = 'Tag Text'
    
    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        target_text = settings['text_type']
        if target_text == 'text':
            text = get_text(element)
        elif target_text == 'text before':
            text = ''
            previous = element.getprevious()
            if previous is not None:
                text = previous.tail or ''
            else:
                parent = element.getparent()
                if parent is not None:
                    text = parent.text or ''
        elif target_text == 'text after':
            text = element.tail or ''

#        # Convert text to unicode (from bytes) to allow proper comparisons,
#        # and avoid errors from regex moule
#        text = str(text, encoding="utf-8")
        
        if operator == 'matches':
            if is_regex:
                return bool(regex.search(value, text, flags=regex.DOTALL | regex.M))
            else:
                return value == text
                
        elif operator == 'does not match':
            if is_regex:
                return not bool(regex.search(value, text, flags=regex.DOTALL | regex.M))
            else:
                return value != text
                
        elif operator == 'is null':
            return text == ''

        elif operator == 'not null':
            return text != ''

    def validate(self, settings):
        if not settings:
            return (_('Text Filter Error'), _('You must configure this filter'))
        operator = settings.get('operator')
        if not operator:
            return (_('Text Filter Error'), _('You must choose an operator'))
        if not settings.get('value'):
            if operator not in [ 'is null', 'not null']:
                return (_('Text Filter Error'), _('You must specify a tag text'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget
