#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree, prefixname, qname, isqname

from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter
from calibre_plugins.editor_chains.common_utils import reverse_lookup

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/tag_name.py - exception when loading translations")

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

#        self.label = QLabel(_('Tag Name'))
#        l.addWidget(self.label, 0, 0, 1, 1)
        
        self.combo = QComboBox()
        self.combo.addItems(['matches', 'does not match'])
        l.addWidget(self.combo, 0, 1, 1, 1)
        
        self.value_edit = QLineEdit()
        l.addWidget(self.value_edit, 0, 2, 1, 1)
        
        self.regex_chk = QCheckBox(_('Regex'))
        l.addWidget(self.regex_chk, 0, 3, 1, 1)

    def load_settings(self, settings):
        if settings:
            self.combo.setCurrentText(settings['operator'])
            self.value_edit.setText(settings['value'])
            self.regex_chk.setChecked(settings['regex'])

    def save_settings(self):
        settings = {}
        settings['operator'] = self.combo.currentText()
        settings['value'] = self.value_edit.text()
        settings['regex'] = self.regex_chk.isChecked()
        return settings


class TagNameFilter(ElementFilter):

    name = 'Tag Name'
    
    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        element_name = element.tag
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        ns_prefix = ':' in value and not isqname(value)
        if ns_prefix:
            element_name = prefixname(element_name, {v: k for k, v in XPNSMAP.items()})
        elif isqname(value):
            pass
        else:
            element_name = barename(element_name)
        
        if is_regex:
            is_match = bool(regex.search(value, element_name))
            if operator == 'matches':
                return is_match
            elif operator == 'does not match':
                return not is_match
        else:
            is_match = value == element_name
            if operator == 'matches':
                return is_match
            elif operator == 'does not match':
                return not is_match

    def validate(self, settings):
        if not settings:
            return (_('Tag Name Filter Error'), _('You must configure this filter'))
        if not settings.get('operator'):
            return (_('Tag Name Filter Error'), _('You must choose an operator'))
        if not settings.get('value'):
            return (_('Tag Name Filter Error'), _('You must specify a tag name'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget     
