#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex
import operator
from collections import OrderedDict

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.widgets import Dialog
from calibre.ebooks.oeb.stylizer import FONT_SIZE_NAMES
from calibre.ebooks.css_transform_rules import unit_convert
from calibre.ebooks.oeb.base import XPNSMAP, barename
from calibre.ebooks.oeb.normalize_css import normalizers
from calibre.ebooks.oeb.polish.cascade import resolve_property, Values
from css_parser.css import PropertyValue, Property

from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter
from calibre_plugins.editor_chains.actions.expand_styles import get_profile
from calibre_plugins.editor_chains.common_utils import get_icon

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/style_property.py - exception when loading translations")

DEFAULT_PROFILE = 'default'

NUMERICAL_OPERATORS = OrderedDict()
for x, y in (
    ('=', operator.eq),
    ('>', operator.gt),
    ('>=', operator.ge),
    ('<', operator.lt),
    ('<=', operator.le),
    ('!=', operator.ne) ):
    NUMERICAL_OPERATORS[x] = y

OPERATORS = [
    'matches',
    'does not match',
    'is null',
    'not null',
] + list(NUMERICAL_OPERATORS.keys())

SUPPORTED_DIMENSIONS = [
    'px',
    'in',
    'pt',
    'pc',
    'mm',
    'cm',
    'rem',
    'q'
]

#def expand_style_map(style_map):
    #for elem, styles in style_map.items():
        #for prop_name, prop_value in styles.items():
            #normalizer = normalizers.get(prop_name, None)
            #if normalizer:
                #for name, val in normalizer(name, prop_value).items():
                    #priority = ''
                    #if prop_value.is_important: priority = 'important'
                    #styles[name] = Values(val, prop_value.sheet_name, priority)
    #return style_map

def convert_dimension(style_map, elem, name, numerical_value, dimension, target_dimension, body_font_size=12):
    if dimension == 'em':
        font_size = get_font_size(elem, style_map)
        if not font_size:
            return None
        pv = font_size.propertyValue[0]
        if pv.type == 'IDENT':
            return None
        elif pv.type == 'DIMENSION':
            if not pv.dimension in SUPPORTED_DIMENSIONS:
                return None
            elif pv.dimension == 'pt':
                font_size_in_pt = pv.value
            else:
                font_size_in_pt = unit_convert(pv.value, pv.dimension, body_font_size=body_font_size)
        if name == 'font-size':
            value_in_pt = font_size_in_pt
        else:
            value_in_pt = font_size_in_pt * numerical_value
    elif dimension == 'rem':
        value_in_pt = unit_convert(numerical_value, target_unit, body_font_size=body_font_size)
    elif dimension not in SUPPORTED_DIMENSIONS:
        # Cannot convert unsupported dimension, conversion returns None
        return None
    else:
        value_in_pt = unit_convert(numerical_value, dimension)
    value_in_target_dimension = unit_convert_from_pt(value_in_pt, target_dimension)
    return value_in_target_dimension

def unit_convert_from_pt(value, unit, dpi=96.0, body_font_size=12):
    result = None
    if unit == 'px':
        result = value * dpi / 72.0
    elif unit == 'in':
        result = value / 72.0
    elif unit == 'pt':
        result = value
    elif unit == 'pc':
        result = value / 12.0
    elif unit == 'mm':
        result = value / 2.8346456693
    elif unit == 'cm':
        result = value / 28.346456693
    elif unit == 'rem':
        result = value / body_font_size
    elif unit == 'q':
        result = value / 0.708661417325
    return result

def get_body_font_size_in_pt(elem, style_map, body_font_size=12):
    body = elem.xpath('//h:body', namespaces=XPNSMAP)[0]
    size = resolve_property(style_map, body, 'font-size')
    prop = Property(name='font-size', value=size.cssText)
    prop = normalize_fontsize_property(prop, profile_name='default')
    pv = prop.propertyValue[0]
    if pv.type == 'DIMENSION':
        # normalizing keywords like medium will result in rem values
        if pv.dimension in ['rem','em']:
            return pv.value * body_font_size
        elif pv.dimension in SUPPORTED_DIMENSIONS:
            size_in_pt = unit_convert(pv.value, pv.dimension)
            return size_in_pt
        else:
            # in case of unsupported dimension revert to default value
            return body_font_size
    else:
        return body_font_size

def normalize_fontsize_property(prop, profile_name='default'):
    profile = get_profile(profile_name)
    if prop.value == 'normal':
        prop.value = 'medium'
    if prop.value == 'smallest':
        prop.value = 'xx-small'
    if prop.value in FONT_SIZE_NAMES:
        prop = Property('font-size', f"{profile.fnames[prop.value] / float(profile.fbase):.1f}rem")
    return prop

def get_font_size(elem, style_map, profile_name='default', body_font_size=12):
    # This return a Values object, use .cssText instead of .value
    size_value = resolve_property(style_map, elem, 'font-size')
    size = Property('font-size', size_value.cssText)
    size = normalize_fontsize_property(size, profile_name=profile_name)
    if size.propertyValue[0].type == 'IDENT':
        if barename(elem.tag) == 'body':
            return body_font_size
        else:
            return None
    elif size.propertyValue[0].type == 'DIMENSION':
        if size.propertyValue[0].dimension in ['em','%']:
            parent_size = get_font_size(elem.getparent(), style_map, profile_name=profile_name, body_font_size=body_font_size)
            if not parent_size:
                return None
            parent_size = normalize_fontsize_property(parent_size, profile_name=profile_name)
            number = float(parent_size.propertyValue[0].value) * float(size.propertyValue[0].value)
            size = Property('font-size', f'{number}{parent_size.propertyValue[0].dimension}')
    return normalize_fontsize_property(size, profile_name=profile_name)

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self.convert_unit = False
        self.unit = ''
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        self.name_edit = QLineEdit()
        self.name_edit.setToolTip(_('Property name e.g font-size, margin-top, .... etc'))
        l.addWidget(self.name_edit, 0, 0, 1, 1)
        
        self.operator_combo = QComboBox()
        self.operator_combo.addItems(OPERATORS)
        self.operator_combo.activated.connect(self._on_operator_changed)
        l.addWidget(self.operator_combo, 0, 1, 1, 1)

        self.value_edit = QLineEdit()
        l.addWidget(self.value_edit, 0, 2, 1, 1)
        
        self.regex_chk = QCheckBox(_('Regex'))
        l.addWidget(self.regex_chk, 0, 3, 1, 1)

        self.unit_combo = QComboBox()
        self.unit_combo.setToolTip(_(
            'Convert Property from one dimension to another.\n'
            'Supported <b>Output Units</b> are: {}'
            '\nSame units are supported for <b>input</b>, in addtion to <b>em</b>'.format("\n".join(["    " + x for x in SUPPORTED_DIMENSIONS]))
        ))
        self.unit_combo.addItems([''] + SUPPORTED_DIMENSIONS)
        self.unit_combo.setCurrentText(self.unit)
        l.addWidget(self.unit_combo, 0, 4, 1, 1)

        self._on_operator_changed()

    def _on_operator_changed(self, *args):
        operator = self.operator_combo.currentText()
        self.unit_combo.setVisible(operator in NUMERICAL_OPERATORS)
        self.regex_chk.setVisible(operator in ['matches','does not match'])

    def load_settings(self, settings):
        if settings:
            self.operator_combo.setCurrentText(settings['operator'])
            self.value_edit.setText(settings['value'])
            self.name_edit.setText(settings['name'])
            self.regex_chk.setChecked(settings['regex'])
            self.unit_combo.setCurrentText(settings.get('unit', ''))
            self._on_operator_changed()

    def save_settings(self):
        settings = {}
        settings['operator'] = self.operator_combo.currentText()
        settings['name'] = self.name_edit.text()
        settings['value'] = self.value_edit.text()
        settings['regex'] = self.regex_chk.isChecked()
        settings['unit'] = self.unit_combo.currentText()
        return settings

class StylePropertyFilter(ElementFilter):

    name = 'Style Property'

    def get_body_font_size(self, elem, style_map, filename, context, default_value=12):
        val = getattr(context, 'body_font_size', {}).get(filename)
        if val:
            return val
        val = get_body_font_size_in_pt(elem, style_map, body_font_size=default_value)
        if not hasattr(context, 'body_font_size'):
            context.body_font_size = {}
        context.body_font_size[filename] = val
        return val

    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        style_map = chain.action_vars['style_map']
        filename = context.file_name
        name = settings['name']
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        convert_unit = settings['unit']

        property_value = resolve_property(style_map, element, name)
        if settings.get('no_inheritance_or_defaults', False):
            property_value = style_map.get(name)

        if not property_value: property_value = ''

        if property_value:
            if operator == 'is null':
                return False
            elif operator == 'not null':
                return True
        else:
            if operator == 'is null':
                return True
            else:
                return False

        # convert calibre's Values() to PropertyValue()
        property_value = PropertyValue(property_value.cssText)
        if operator in NUMERICAL_OPERATORS.keys():
            if len(property_value) == 1:
                property_value = property_value[0]
                if property_value.type == 'DIMENSION':
                    numerical_value = property_value.value
                    if convert_unit:
                        body_font_size = self.get_body_font_size(element, style_map, filename, context)
                        target_dimension = convert_unit
                        numerical_value = convert_dimension(style_map, element,
                                name, numerical_value, property_value.dimension,
                                target_dimension, body_font_size=body_font_size)
                    comp = NUMERICAL_OPERATORS[operator]
                    try:
                        return comp(numerical_value, float(value))
                    except TypeError:
                        return False
                else:
                    return False
            else:
                return False

        if operator == 'matches':
            if is_regex:
                return bool(regex.search(value, property_value.cssText))
            else:
                return value == property_value.cssText
        elif operator == 'does not match':
            if is_regex:
                return not bool(regex.search(value, property_value.cssText))
            else:
                return value != property_value.cssText

    def validate(self, settings):
        if not settings:
            return (_('Attribute Filter Error'), _('You must configure this filter'))
        operator = settings.get('operator')
        if not operator:
            return (_('Attribute Filter Error'), _('You must choose an operator'))
        if operator == 'no attributes':
            return True
        value = settings.get('value')
        if not value:
            if operator not in [ 'is null', 'not null']:
                return (_('Attribute Filter Error'), _('You must specify an attribute value'))
        if operator in NUMERICAL_OPERATORS:
            try:
                float(value)
            except:
                return _('Value Error'), _(f'Non numerical value ({value}) with numerical operator ({operator})')
        if not settings.get('name'):
            return (_('Attribute Filter Error'), _('You must specify an attribute name'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget  
