#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import inspect

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit, QToolButton, QRadioButton,
                     QSizePolicy)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree

from calibre_plugins.editor_chains.common_utils import get_icon, reverse_lookup
from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter
from calibre_plugins.editor_chains.code_editor import CodeEditor, compile_code
from calibre_plugins.editor_chains.actions.tag_actions.gui import FiltersDialog

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/relative.py - exception when loading translations")

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filters = self.plugin_action.actions['Tag Actions'].filters
        self.filter_ = filter_
        self.filters_config = {}
        self._init_controls()

    def _init_controls(self):

        l = self.l = QHBoxLayout()
        self.setLayout(l)

        self.relative_combo = QComboBox()
        self.relative_combo.addItems(['previous sibling', 'next sibling', 'parent', 'child'])
        l.addWidget(self.relative_combo)
        
        self.relative_combo.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Preferred)

        self.opt_combo = QComboBox()
        self.opt_combo.addItems(['is null', 'not null'])
        l.addWidget(self.opt_combo)

        self.opt_combo.currentTextChanged.connect(self._on_opt_combo_changed)
        self.opt_combo.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Preferred)
        
        self.tag_button = QToolButton()
        self.tag_button.setIcon(get_icon('gear.png'))
        self.tag_button.clicked.connect(self._tag_button_clicked)
        l.addWidget(self.tag_button)

        self._on_opt_combo_changed(self.opt_combo.currentText())

    def _tag_button_clicked(self):
        d = FiltersDialog(self, self.plugin_action, self.filters_config, title=_('Sibling Criteria'))
        if d.exec_() == d.Accepted:
            self.filters_config = d.filters_config

    def _on_opt_combo_changed(self, text):
        self.tag_button.setEnabled(text == 'not null')

    def load_settings(self, settings):
        if settings:
            self.relative_combo.setCurrentText(settings['relative'])
            self.opt_combo.setCurrentText(settings['opt'])
            self.filters_config = settings.get('filters_config', {})

    def save_settings(self):
        settings = {}
        settings['relative'] = self.relative_combo.currentText()
        settings['opt'] = self.opt_combo.currentText()
        settings['filters_config'] = self.filters_config
        return settings


class RelativeFilter(ElementFilter):

    name = 'Related Tags'
    
    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        opt = settings['opt']
        filters_config = settings.get('filters_config')
        relative = settings.get('relative')
        filters = self.plugin_action.actions['Tag Actions'].filters
        group_filter = filters['Group']
        if relative == 'parent':
            target_element = element.getparent()
        elif relative == 'next sibling':
            target_element = element.getnext()
        elif relative == 'previous sibling':
            target_element = element.getprevious()
        elif relative == 'child':
            target_element = list(element)

        if opt == 'is null':
            if target_element in [None, []]:
                return True
            else:
                return False
        elif opt == 'not null':
            if target_element in [None, []]:
                return False
            else:
                if filters_config:
                    if relative == 'child':
                        for child in target_element:
                            res = group_filter.evaluate(chain, child, filters_config, context)
                            if res is True:
                                return True
                        return False
                    else:
                        return group_filter.evaluate(chain, target_element, filters_config, context)
                else:
                    return True

    def validate(self, settings):
        if not settings:
            return (_('Related Tag Filter Error'), _('You must configure this filter'))
        if not settings.get('relative'):
            return (_('Related Tag Filter Error'), _('You must choose a tag relative'))
#        if settings['opt'] == 'not null':
#            if not settings.get('filters_config'):
#                return (_('Related Tag Filter Error'), _('Criteria for relative tag not specified'))
        return True

    def filter_widget(self):
        return FilterWidget
