#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from collections import OrderedDict, defaultdict
import copy
import builtins
import json

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QGroupBox, QToolButton, QPushButton, QScrollArea, QComboBox,
                     QSizePolicy, QSize, QMenu, QFont)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import XPNSMAP, etree

from calibre_plugins.editor_chains.common_utils import get_icon, get_pixmap, ViewLog, reverse_lookup
from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter     

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/group.py - exception when loading translations")

class ControlContainer(QGroupBox):
    
    def __init__(self, parent, plugin_action, filter_widget, filter_name, filters):
        self.filter_widget = filter_widget
        self.filter_name = filter_name
        try:
            self.filter_ = filters[filter_name]
        except:
            # Empty control for custom filters no longer available
            self.filter_ = None
        self.parent_group = parent
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):
        QGroupBox.__init__(self)
        
        self.setTitle(self.filter_name)
        
        l = QGridLayout()
        self.setLayout(l)

        l.addWidget(self.filter_widget, 1, 0, 2, 1)

        remove_label = QLabel('<a href="close">✕</a>')
        remove_label.setToolTip(_('Remove'))
        remove_label.linkActivated.connect(self._remove)
        l.addWidget(remove_label, 1, 1, 1, 1, Qt.AlignRight)

        if self.filter_name == 'Group':
            self.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.MinimumExpanding)
        else:
            self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Maximum)
            

    def _remove(self):
        group = self.parent_group
        self.setParent(None)
        self.deleteLater()

    def load_settings(self, filter_config):
        res = self.filter_widget.load_settings(filter_config)
        return res

    def validate(self):
        if self._filter:
            settings = self.filter_widget.save_settings()
            is_valid = self.filter_.validate(settings)
        else:
            is_valid = _('Filter unavailable'), _('No filter can be found with theis')
        return is_valid
    
    def save_settings(self):
        return self.filter_widget.save_settings()

class EmptyControl(QGroupBox):
    
    def __init__(self, parent, plugin_action, filter_):
        self.parent_group = parent
        self.filter = filter_
        if isinstance(filter_, ElementFilter):
            self.label_msg = filter_.name
            self.mode = 'error'
        else:
            self.label_msg = _('Error: No filter with the name ({}) can be found.').format(filter_)
            self.mode = 'non_configurable'
        self._init_controls()

    def _init_controls(self):
        QGroupBox.__init__(self)
        
        self.layout = l = QGridLayout()
        self.label = QLabel(self.label_msg)
        self.setLayout(l)

    def load_settings(self, filter_config):
        pass        

    def validate(self):
        if self.mode == 'error':
            return _('Filter not found'), _(_('Error: No filter with the name ({}) can be found.').format(self.filter_.name))
        else:
            return True
    
    def save_settings(self):
        pass

class GroupWidget(QGroupBox):
    
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        self.identifier = kwargs.get('identifier')
        self.allow_subgroups = True
        if isinstance(parent, GroupWidget):
            self.allow_subgroups = False
        self.parent = parent
        self.filter_ = filter_
        self.plugin_action = plugin_action
        self.filters = self.plugin_action.actions['Tag Actions'].filters
        self.match_dict = {
            'Match all': 'all',
            'Match any': 'any'
        }
        self._init_controls()

    def _init_controls(self):
        QGroupBox.__init__(self)
        l = QVBoxLayout()
        self.setLayout(l)
        
        hl1 = QHBoxLayout()
        clear_button = QToolButton(self)
        clear_button.setToolTip(_('Clear all filters'))
        clear_button.setIcon(get_icon('clear_left.png'))
        clear_button.clicked.connect(self.reset)
        hl1.addWidget(clear_button)
        hl1.addStretch(1)

        self.match_combo = QComboBox()
        for desc, match in self.match_dict.items():
            self.match_combo.addItem(desc, match)
        self.match_combo.setCurrentIndex(0)
        hl1.addWidget(self.match_combo)
        hl1.addStretch(1)

        add_button = QToolButton(self)
        add_button.setToolTip(_('Add filter(s) to narrow down tags to certain criteria.'))
        add_button.setIcon(get_icon('plus.png'))
        hl1.addWidget(add_button)
        l.addLayout(hl1)

        add_button.setPopupMode(QToolButton.MenuButtonPopup)
        m = QMenu()
        filter_names = list(self.filters.keys())
        if not self.allow_subgroups:
            filter_names.remove('Group')
        for filter_name in filter_names:
            m.addAction(_(filter_name), partial(self._add_control, {'name':filter_name, 'config': {}}))
        add_button.setMenu(m)

        w = QWidget(self)
        self.controls_layout = QVBoxLayout()
        self.controls_layout.setSizeConstraint(self.controls_layout.SetMinAndMaxSize)
        w.setLayout(self.controls_layout)
        
        scroll = QScrollArea()
        scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setWidgetResizable(True)
        scroll.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.MinimumExpanding)
        scroll.setObjectName('myscrollarea')
        scroll.setStyleSheet('#myscrollarea {background-color: transparent}')
        scroll.setWidget(w)
         
        l.addWidget(scroll)

    def _add_control(self, filter_config, validate=False):
        if filter_config:
            filter_name = filter_config['name']
            filter_ = self.filters.get(filter_name)
            if filter_ is None:
                filter_widget_cls = EmptyControl
                filter_ = filter_name
            else:
                filter_widget_cls = filter_.filter_widget()
            filter_widget = filter_widget_cls(self, self.plugin_action, filter_)
            control = ControlContainer(self, self.plugin_action, filter_widget, filter_name, self.filters)
            control.load_settings(filter_config.get('config', {}))
        self.controls_layout.addWidget(control)

    def reset(self):
        # remove controls in reverse order
        for idx in reversed(range(self.controls_layout.count())):
            control = self.controls_layout.itemAt(idx).widget()
            control.setParent(None)
            control.deleteLater()

    def load_settings(self, settings):
        match_opt = settings.get('config', {}).get('match_opt', reverse_lookup(self.match_dict, 'all'))
        text = reverse_lookup(self.match_dict, match_opt)
        self.match_combo.setCurrentText(text)
        filters_config = settings.get('config', {}).get('filters_config', {})
        for filter_config in filters_config:
            self._add_control(filter_config)

    def save_settings(self):
        match_opt = self.match_combo.itemData(self.match_combo.currentIndex())
        filters_config = []
        for idx in range(self.controls_layout.count()):
            control = self.controls_layout.itemAt(idx).widget()
            filter_config = {'name': control.filter_name, 'config': control.save_settings()}
            filters_config.append(filter_config)
        if filters_config:
            return {'name': 'Group', 'config': {'match_opt': match_opt, 'identifier': self.identifier,'filters_config': filters_config}}
        else:
            return {}


class GroupFilter(ElementFilter):

    name = 'Group'
    
    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        arg_list = []
        match_opt = settings['config']['match_opt']
        func = builtins.__dict__.get(match_opt)
        filters = self.plugin_action.actions['Tag Actions'].filters
        for filter_config in settings.get('config', {}).get('filters_config', {}):
            name = filter_config['name']
            filter_ = filters.get(name)
            if filter_:
                res = filter_.evaluate(chain, element, filter_config['config'], context)
                arg_list.append(res)
                #filter_config['result'] = res
        return func(arg_list)

    def validate(self, settings):
        if not settings:
            return (_('Group Filter Error'), _('You must configure this filter'))
        filters = self.plugin_action.actions['Tag Actions'].filters
        all_errors = []
        filters_config = settings.get('config', {}).get('filters_config', {})
        if not filters_config:
            return (_('Group Filter Error'), _('You must at least add one filter to the group'))
        for filter_config in filters_config:
            name = filter_config['name']
            filter_ = filters.get(name)
            if filter_:
                res = filter_.validate(filter_config['config'])
                if res is not True:
                    all_errors.append(': '.join(list(res)))
            else:
                all_errors.append(_(f'Filter Error: cannot find filter with the name ({name})'))
        if all_errors:
            return _('Group Filter Error'), '\n'.join(all_errors)
        return True

    def filter_widget(self):
        return GroupWidget
