#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, prefixname, etree, qname, isqname

from calibre_plugins.editor_chains.actions.tag_actions.filters.base import ElementFilter
from calibre_plugins.editor_chains.common_utils import reverse_lookup

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/attribute.py - exception when loading translations")

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)


        self.name_edit = QLineEdit()
        self.name_edit.setToolTip(_('Attribute name e.g class, identifier, href, .... etc'))
        l.addWidget(self.name_edit, 0, 0, 1, 1)
        
        self.combo = QComboBox()
        self.combo.addItems(['matches', 'does not match','contains','does not contain', 'is null', 'not null', 'no attributes'])
        l.addWidget(self.combo, 0, 1, 1, 1)
        self.combo.activated.connect(self._on_operator_changed)

        self.value_edit = QLineEdit()
        l.addWidget(self.value_edit, 0, 2, 1, 1)
        
        self.regex_chk = QCheckBox(_('Regex'))
        l.addWidget(self.regex_chk, 0, 3, 1, 1)
        self._on_operator_changed()

    def _on_operator_changed(self, *args):
        operator = self.combo.currentText()
        self.value_edit.setVisible(operator not in ['is null', 'not null', 'no attributes'])
        self.regex_chk.setVisible(operator not in ['is null', 'not null', 'no attributes'])
        self.name_edit.setVisible(operator not in ['no attributes'])

    def load_settings(self, settings):
        if settings:
            self.combo.setCurrentText(settings['operator'])
            self.value_edit.setText(settings['value'])
            self.name_edit.setText(settings['name'])
            self.regex_chk.setChecked(settings['regex'])
            self._on_operator_changed()

    def save_settings(self):
        settings = {}
        settings['operator'] = self.combo.currentText()
        settings['name'] = self.name_edit.text()
        settings['value'] = self.value_edit.text()
        settings['regex'] = self.regex_chk.isChecked()
        return settings


class AttributeFilter(ElementFilter):

    name = 'Attribute'
    
    def evaluate(self, chain, element, settings, context, *args, **kwargs):
        name = settings['name']
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        ns_prefix = ':' in name and not isqname(name)
        if ns_prefix:
            name = qname(name, XPNSMAP)
#        elif isqname(name):
#            pass
#        else:
#            name = qname('h:'+name, XPNSMAP)

        if operator == 'no attributes': 
            return element.attrib == {} 

        attribute_value = element.get(name, '')

        if operator == 'matches':
            if is_regex:
                return bool(regex.search(value, attribute_value))
            else:
                return value == attribute_value
        elif operator == 'does not match':
            if is_regex:
                return not bool(regex.search(value, attribute_value))
            else:
                return value != attribute_value 
        elif operator == 'is null':
            return bool(attribute_value) is False
        elif operator == 'not null':
            return bool(attribute_value) is True
        elif operator in ['contains', 'does not contain']:
            vlist = attribute_value.split()
            if operator == 'contains':
                flag = False
                for v in vlist:
                    if is_regex:
                        res = regex.search(value, v)
                        if res:
                            flag = True
                            break
                    else:
                        res = value == v
                        if res:
                            flag = True
                            break
            elif operator == 'does not contain':
                flag = True
                for v in vlist:
                    if is_regex:
                        res = regex.search(value, v)
                        if res:
                            flag = False
                            break
                    else:
                        res = value == v
                        if res:
                            flag = False
                            break
            return flag

    def validate(self, settings):
        if not settings:
            return (_('Attribute Filter Error'), _('You must configure this filter'))
        operator = settings.get('operator')
        if not operator:
            return (_('Attribute Filter Error'), _('You must choose an operator'))
        if operator == 'no attributes':
            return True
        if not settings.get('value'):
            if operator not in [ 'is null', 'not null']:
                return (_('Attribute Filter Error'), _('You must specify an attribute value'))
        if not settings.get('name'):
            return (_('Attribute Filter Error'), _('You must specify an attribute name'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget  
