#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QGroupBox, QLineEdit, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree

from calibre_plugins.editor_chains.actions.tag_actions.actions.base import ElementAction
from calibre_plugins.editor_chains.etree import wrap

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/actions/wrap.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        groupbox1 = QGroupBox(_('Wrapper Tag'))
        l.addWidget(groupbox1)
        groupbox1_l = QVBoxLayout()
        groupbox1.setLayout(groupbox1_l)      
        self.wrap_ledit = QLineEdit()
        self.wrap_ledit.setToolTip(_('Enter full tag used to wrap\n'
                                     'e.g. <div class="class1 class2" id="id1"></div>'))
        groupbox1_l.addWidget(self.wrap_ledit)

        groupbox2 = self.groupbox2 = QGroupBox(_('Insert tag or text between wrapped tags (Optional)'))
        l.addWidget(groupbox2)
        groupbox2_l = QVBoxLayout()
        groupbox2.setLayout(groupbox2_l)
        groupbox2.setCheckable(True)
        groupbox2.setChecked(False)
        self.insert_ledit = QLineEdit()
        self.insert_ledit.setToolTip(_('Enter an optional tag or text to insert between wrapped elements'))
        groupbox2_l.addWidget(self.insert_ledit)
        
        l.addStretch(1)
        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            self.wrap_ledit.setText(settings['wrapper'])
            insert = settings.get('insert')
            if insert:
                self.groupbox2.setChecked(True)
                self.insert_ledit.setText(insert)

    def save_settings(self):
        settings = {}
        settings['wrapper'] = self.wrap_ledit.text()
        if self.groupbox2.isChecked():
            settings['insert'] = self.insert_ledit.text()
        return settings

class WrapAction(ElementAction):

    name = 'Wrap'
    
    def run(self, chain, number, elements, settings, context, *args, **kwargs):
        if elements:
            wrap(elements, settings['wrapper'], settings.get('insert', None))


    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this filter'))
        try:
            wrapper = etree.fromstring(settings['wrapper'])
        except:
            return (_('Invalid tag'), _('Invalid wrapper'))
        insert = settings.get('insert')
        if insert:
            try:
                insert = etree.fromstring(element_to_insert)
            except:
                # element is text. Try to wrap it in a paragraph to see if it is a valid text
                try:
                    insert = etree.fromstring('<p>'+insert+'</p>')
                except:
                    return (_('Invalid text'), _('The text you entered to insert between wrapped tag is not valid.'))
        return True

    def config_widget(self):
        return ConfigWidget

