#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import re
from uuid import uuid4

from qt.core import QCheckBox

from calibre_plugins.editor_chains.etree import etree, insert_before, insert_after
from calibre_plugins.editor_chains.actions.search_replace import (
    replace_function_init_env, get_search_name, process_searches,
    tprefs, search_from_saved_search,
    Function, ConfigWidget as SRConfigWidget)
from calibre_plugins.editor_chains.actions.tag_actions.actions.base import ElementAction

def replace_all(container, uuid, searches, name):
    count = 0
    substring_locator = f'<div class="temp_delete">start_{uuid}</div>(.+?)<div class="temp_delete">end_{uuid}</div>'

    for p, repl in searches:
        repl_is_func = isinstance(repl, Function)
        if repl_is_func:
            replace_function_init_env(container, repl)
        if repl_is_func:
            repl.context_name = name
        raw = container.raw_data(name)
        res = reversed(list(re.finditer(substring_locator, raw, flags=re.DOTALL | re.M)))
        for m in res:
            prefix = raw[:m.start()]
            substring = raw[m.start():m.end()]
            substring = re.sub(substring_locator, r'\1', substring, flags=re.DOTALL | re.M)
            new_substring, num = p.subn(repl, substring)
            count += num
            suffix = raw[m.end():]
            raw = prefix + new_substring + suffix

        if repl_is_func:
            repl.end()
    return raw

class ConfigWidget(SRConfigWidget):
    def __init__(self, plugin_action, action):
        self.plugin_action = plugin_action
        self.action = action
        SRConfigWidget.__init__(self, plugin_action)
        self.search_widget.where_box.setVisible(False)
        self.saved_searches_widget.where_box.setVisible(False)

    def load_settings(self, settings):
        SRConfigWidget.load_settings(self, settings)

    def save_settings(self):
        settings = SRConfigWidget.save_settings(self)
        return settings

class SearchReplaceAction(ElementAction):

    name = 'Search And Replace'

    def get_searches(self, settings):
        opt = settings.get('opt', 'search_replace')
        if opt == 'search_replace':
            searches = settings['searches']
        elif opt == 'saved_searches':
            where = settings['where']
            saved_searches_names = settings['saved_searches']
            searches = [s for s in tprefs['saved_searches'] if s['name'] in saved_searches_names]
            searches = [search_from_saved_search(s, where=where) for s in searches]
        return searches

    def run(self, chain, number, elements, settings, context, *args, **kwargs):
        for element in elements:
            context.elements[element] = context.file_name

    def pre_run(self, chain, settings, context):
        context.uuid = str(uuid4())
        context.elements = {}

    def post_run(self, chain, settings, context):
        container = chain.current_container
        searches = self.get_searches(settings)
        # Remove any non-top elements to prevent overlapping matches {
        all_elements = set(context.elements.keys())
        top_elements = {}
        for element, name in context.elements.items():
            parent = element.getparent()
            if not parent in all_elements:
                top_elements[element] = name
        # }
        for element in top_elements.keys():
            insert_before(element, f'<div class="temp_delete">start_{context.uuid}</div>')
            insert_after(element, f'<div class="temp_delete">end_{context.uuid}</div>')
        for name in set(top_elements.values()):
            # first, read the changes done by self.run()
            container.dirty(name)
            for search in searches:
                raw = replace_all(container, context.uuid, process_searches(search), name)
            container.open(name, 'w').write(raw)
            #container.dirty(name)

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        searches = self.get_searches(settings)
        if settings.get('opt') == 'saved_searches':
            all_saved_searches = [s['name'] for s in tprefs['saved_searches']]
            saved_searches = settings['saved_searches']
            if not saved_searches:
                return _('No saved searches'), _('You must select at least one saved search')
            for s in saved_searches:
                if s not in all_saved_searches:
                    return _('Saved search unavailable'), _(f'Cannot find saved search: {s}')

        for search in searches:
            s = process_searches(search, validate=True)
            if s is not True:
                return s
            if not search.get('find'):
                return _('Missing find'), _(f'You must specify find term. (Search name: {get_search_name(search)})')
            #if not search.get('replace'):
                #return _('Missing replace'), _('You must specify replace term/function. (Search name: {})'.format(get_search_name(search)))
        return True

    def config_widget(self):
        return ConfigWidget
