#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import re

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QHBoxLayout,
                     QLineEdit, QGroupBox, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree, qname, isqname

from calibre_plugins.editor_chains.common_utils import reverse_lookup
from calibre_plugins.editor_chains.actions.tag_actions.actions.base import ElementAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/actions/modify_tag.py - exception when loading translations")

def validate_ns_prefix(string, nsmap=XPNSMAP):
    if not isqname(string):
        if ':' in string:
            ns_prefix, sep, name = string.rpartition(':')
            if not nsmap.get(ns_prefix):
                return False
    return True

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        name_group_box = QGroupBox(_('Change name of the tag'))
        l.addWidget(name_group_box)
        name_group_box_l = QVBoxLayout()
        name_group_box.setLayout(name_group_box_l)
        
        self.name_ledit = QLineEdit()
        name_group_box_l.addWidget(self.name_ledit)

        delete_attrs_groupbox = QGroupBox('Delete attributes')
        l.addWidget(delete_attrs_groupbox)
        delete_attrs_groupbox_l = QVBoxLayout() 
        delete_attrs_groupbox.setLayout(delete_attrs_groupbox_l)    
        self.attributes_delete_ledit = QLineEdit()
        self.attributes_delete_ledit.setToolTip(_('Comma separated list of attribute names to delete.\n'
                                                  'Use "*" to delete all attributes.'))
        delete_attrs_groupbox_l.addWidget(self.attributes_delete_ledit)

        remove_classes_groupbox = QGroupBox('Remove classes')
        l.addWidget(remove_classes_groupbox)
        remove_classes_groupbox_l = QVBoxLayout() 
        remove_classes_groupbox.setLayout(remove_classes_groupbox_l)    
        self.classes_remove_ledit = QLineEdit()
        self.classes_remove_ledit.setToolTip(_('Comma separated list of classes to remove.\n'
                                               'Use "*" to delete all attributes.'))
        remove_classes_groupbox_l.addWidget(self.classes_remove_ledit)

        add_attrs_groupbox = QGroupBox(_('Add tag attributes'))
        l.addWidget(add_attrs_groupbox)
        add_attrs_groupbox_l = QVBoxLayout()
        add_attrs_groupbox.setLayout(add_attrs_groupbox_l)
        
        self.attributes_ledit = QLineEdit()
        self.attributes_ledit.setToolTip(_('Enter attributes the same way you the are represented in tag\n'
                                           'e.g. class="cls1 cls2" id="id1"'))
        add_attrs_groupbox_l.addWidget(self.attributes_ledit)

        self.cls_append_chk = QCheckBox(_('Append to existing classes'))
        self.cls_append_chk.setChecked(False)
        add_attrs_groupbox_l.addWidget(self.cls_append_chk)
        
        l.addStretch(1)
        self.setMinimumSize(300,500)

    def load_settings(self, settings):
        if settings:
            self.name_ledit.setText(settings.get('new_name', ''))
            self.attributes_delete_ledit.setText(settings.get('delete_attrs', ''))
            self.classes_remove_ledit.setText(settings.get('remove_classes', ''))
            self.attributes_ledit.setText(settings.get('new_attrs', ''))
            self.cls_append_chk.setChecked(settings.get('append_to_classes', False))

    def save_settings(self):
        settings = {}
        settings['new_name'] = self.name_ledit.text()
        settings['delete_attrs'] = self.attributes_delete_ledit.text()
        settings['remove_classes'] = self.classes_remove_ledit.text()
        settings['new_attrs'] = self.attributes_ledit.text()
        settings['append_to_classes'] = self.cls_append_chk.isChecked()
        return settings

class ModifyAction(ElementAction):

    name = 'Modify Tag'
    
    def run(self, chain, number, elements, settings, context, *args, **kwargs):
        for element in elements:
            # Modify tag name
            new_name = settings.get('new_name')
            if new_name:
                new_name = qname(new_name, XPNSMAP)
                element.tag = new_name
            
            # Delete attributes
            delete_attrs = settings.get('delete_attrs')
            if delete_attrs:
                if delete_attrs == '*':
                    for k in element.attrib.keys(): del element.attrib[k]
                else:
                    for x in delete_attrs.split(','):
                        x = x.strip()
                        for attr_name in element.attrib.keys():
                            if ':' in x:
                               if attr_name.startswith('{'):
                                   ns, sep, name = attr_name.lstrip('{').rparition('}')
                                   name = f"{reverse_lookup(XPNSMAP, ns or XPNSMAP['h'])}:{name}"
                               else:
                                   name = 'h:'+name
                            else:
                                name = attr_name
                            if x == name:
                                del element.attrib[attr_name]

            # Delete classes
            remove_classes = settings.get('remove_classes')
            if remove_classes:
                if remove_classes == '*':
                    try: del element.attrib['class']
                    except: pass
                else:
                    classes = element.attrib.get('class', '').split()
                    for cls in remove_classes.split(','):
                        cls = cls.strip()
                        # Sometimes classes are duplicated, remove all instances of the class
                        while cls in classes:
                            classes.remove(cls)
                    element.attrib['class'] = ' '.join(classes)
                    if not classes:
                        try: del element.attrib['class']
                        except: pass

            # Add new attributes
            new_attrs = settings.get('new_attrs')
            if new_attrs:
                temp_attrib = self.parse_attrs_string(new_attrs)
                for k in temp_attrib.keys():
                    if k == 'class':
                        if settings.get('append_to_classes'):
                            classes = element.attrib.get('class', '').split()
                            for cls in temp_attrib.get('class', '').split():
                                if not cls in classes:
                                    classes.append(cls)
                            element.attrib['class'] = ' '.join(classes)
                        else:
                            element.attrib['class'] = temp_attrib['class']
                    else:
                        element.attrib[k] = temp_attrib[k]

    def parse_attrs_string(self, attrs_string):
        l = re.findall(r'([\w\:]+\=\"[^\"]+\")', attrs_string)
        dictionary = {}
        for pair in l:
            key, sep, val = pair.rpartition('=')
            q_name = qname(key, XPNSMAP)
            dictionary[q_name] = val.lstrip('"').rstrip('"')
        return dictionary

    def validate(self, settings):
        if not settings:
            return (_('Modify Tag Error'), _('You must configure this action'))
        new_name = settings.get('new_name')
        if new_name:
            if not validate_ns_prefix(new_name):
                return _('Modify Tag Error'), _(f'Invalid namespace prefix: {new_name}')          
        new_attrs = settings.get('new_attrs')
        if new_attrs:
            attrib = self.parse_attrs_string(new_attrs)
            for k in attrib.keys():
                if not validate_ns_prefix(k):
                    return _('Modify Tag Error'), _(f'Invalid namespace prefix: {k}')
        return True

    def config_widget(self):
        return ConfigWidget
