#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QGroupBox, QLineEdit, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.oeb.base import XHTML, XPNSMAP, barename, etree

from calibre_plugins.editor_chains.actions.tag_actions.actions.base import ElementAction
from calibre_plugins.editor_chains.etree import insert_after

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/actions/insert_after.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        self.ledit = QLineEdit()
        self.ledit.setToolTip(_('Enter tag or text to insert before matched element(s)'))
        l.addWidget(self.ledit)

        l.addStretch(1)
        self.setMinimumSize(300,100)

    def load_settings(self, settings):
        if settings:
            self.ledit.setText(settings['element_to_insert'])

    def save_settings(self):
        settings = {}
        settings['element_to_insert'] = self.ledit.text()
        return settings

class InsertAfterAction(ElementAction):

    name = 'Insert After'
    
    def run(self, chain, number, elements, settings, context, *args, **kwargs):
        for element in elements:
            insert_after(element, settings['element_to_insert'])


    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action'))
        element_to_insert = settings['element_to_insert']
        try:
            element_to_insert = etree.fromstring(element_to_insert)
        except:
            # element is text. Try to wrap it in a paragraph to see it is a valid text
            try:
                element_to_insert = etree.fromstring('<p>'+element_to_insert+'</p>')
            except:
                return (_('Invalid text'), _('The text you entered is not valid.'))
        return True

    def config_widget(self):
        return ConfigWidget

