#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.tag_actions.actions.base import ElementAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.modify_tag import ModifyAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.code import CodeAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.wrap import WrapAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.unwrap import UnwrapAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.delete import DeleteAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.insert_before import InsertBeforeAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.insert_after import InsertAfterAction
from calibre_plugins.editor_chains.actions.tag_actions.actions.search_replace import SearchReplaceAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/actions/__int__.py - exception when loading translations")



def get_user_tag_actions(plugin_action):
    user_tag_actions = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[ElementAction]):
        name = cls.name
        # must define a name attribute, must be set and not clash with builtin names
        # which can be imported into the module manager by custom actions
        if name in ['', 'ElementAction']:
            continue
        user_tag_actions[name] = cls            
    return user_tag_actions

def get_all_tag_actions(plugin_action):

    builtin_tag_actions = OrderedDict()
    
    _builtin_tag_actions = [
        ModifyAction,
        DeleteAction,
        WrapAction,
        UnwrapAction,
        InsertBeforeAction,
        InsertAfterAction,
        SearchReplaceAction,
        CodeAction
    ]

    for tag_action_cls in _builtin_tag_actions:
        builtin_tag_actions[tag_action_cls.name] = tag_action_cls

    _user_tag_actions = get_user_tag_actions(plugin_action)

    all_tag_actions = OrderedDict()
    user_tag_actions = OrderedDict()
    
    for tag_action_name, tag_action_cls in builtin_tag_actions.items():
        if tag_action_name in ['', 'Element Action']:
            continue
        tag_action = tag_action_cls(plugin_action)
        all_tag_actions[tag_action_name] = tag_action

    for tag_action_name, tag_action_cls in _user_tag_actions.items():
        # dont override builtin tag_actions
        if tag_action_name in list(builtin_tag_actions.keys()) + ['', 'Element Action']:
            continue
        try:
            tag_action = tag_action_cls(plugin_action)
            all_tag_actions[tag_action_name] = tag_action
            user_tag_actions[tag_action_name] = tag_action
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Editor Chains: Tag Actions: Error intializing user filter: {tag_action_name}\n{traceback.format_exc()}')

    return all_tag_actions, builtin_tag_actions, user_tag_actions    
