#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~
__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from collections import OrderedDict, defaultdict
import copy
import builtins

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import OEB_STYLES, OEB_DOCS, XHTML, css_text, XPNSMAP, barename, etree
from css_selectors.select import Select, get_parsed_selector

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.actions.tag_actions.actions import get_all_tag_actions
from calibre_plugins.editor_chains.actions.tag_actions.filters import get_all_filters
from calibre_plugins.editor_chains.actions.tag_actions.gui import ConfigWidget

from calibre_plugins.editor_chains.css import get_style_maps
from calibre_plugins.editor_chains.css import resolve_property as calibre_resolve_property
from calibre_plugins.editor_chains.scope import scope_names, validate_scope, scope_is_headless
from calibre_plugins.editor_chains.common_utils import validate_xpath, validate_css

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/__init__.py - exception when loading translations")

class Context:
    pass

class TagActions(EditorAction):

    name = 'Tag Actions'
    _is_builtin_ = True
    headless = True

    def on_modules_update(self, all_objects):
        if DEBUG:
            prints('Editor Chains: Tag Actions: running on_modules_update()')
        self.actions, self.builtin_actions, self.user_actions = get_all_tag_actions(self.plugin_action)
        self.filters, self.builtin_filters, self.user_filters = get_all_filters(self.plugin_action)

    def run(self, chain, settings, *args, **kwargs):

        container = chain.current_container
        style_map, pseudo_style_map = get_style_maps(container)

        def resolve_property(elem, name):
            ans = calibre_resolve_property(style_map, elem, name)
            return ans.cssText

        context = Context()
        # action vars
        action_vars = chain.action_vars
        # used by code filter, and code action
        action_vars['data'] = {}
        action_vars['context'] = context
        action_vars['resolve_property'] = resolve_property
        action_vars['style_map'] = style_map
        action_vars['pseudo_style_map'] = pseudo_style_map
        #

        tag_selection = settings['tag_selection']
        xpath_expression = settings.get('xpath', '')
        css_expression = settings.get('css', '')
        max_steps = settings.get('max_steps', 0)
        if max_steps == 0:
            max_steps = float('inf')
        filters_config = settings['filters_config']
        group_filter = self.filters['Group']
        dirty = set()
        all_ranges = defaultdict(list)
        # cache is used to prevent overlapping ranges
        cache = set()

        names = scope_names(chain, settings.get('where', 'text'))
        for name, media_type in container.mime_map.items():
            if name not in names: continue
            if media_type in OEB_DOCS:
                context.file_name = name
                roots = [ container.parsed(name) ]
                if xpath_expression:
                    roots = roots[0].xpath(xpath_expression, namespaces=XPNSMAP)
                elif css_expression:
                    roots = tuple(Select(roots[0])(css_expression))
                is_filters = filters_config.get('config', {}).get('filters_config', [])
                if (xpath_expression or css_expression) and not filters_config.get('config', {}).get('filters_config', {}):
                    for elem in roots:
                        all_ranges[name].append([elem])
                        dirty.add(name)
                    continue
                for root in roots:
                    elems = root.iter()
                    for elem in elems:
                        if not isinstance(elem.tag, str):
                            continue
                        is_match = group_filter.evaluate(chain, elem, filters_config, context)
                        #is_match = evluate_filters(elem, self.filters, filters_config)
                        if is_match is True:
                            if tag_selection == 'individual':
                                all_ranges[name].append([elem])
                                dirty.add(name)
                            elif tag_selection == 'range':
                                # make sure we have no overlapping ranges
                                if elem in cache:
                                    continue
                                r = []
                                r.append(elem)
                                steps = 0
                                for sibling in elem.itersiblings():
                                    if isinstance(sibling.tag, str):
                                        steps += 1
                                    else:
                                        # elements like entities and comments are appended to the
                                        # range without incrementing the counter for max_steps
                                        r.append(sibling)
                                        continue
                                    if steps > max_steps:
                                        # Failed to find the last tag in the range within the specified max_steps
                                        break
                                    is_match = group_filter.evaluate(chain, sibling, settings['last_filters_config'], context)
                                    #is_match = evluate_filters(sibling, self.filters, settings['last_filters_config'])
                                    if is_match:
                                        r.append(sibling)
                                        # Range is now complete, we can add it to final results
                                        all_ranges[name].append(r)
                                        cache |= set(r)
                                        dirty.add(name)
                                        break
                                    # Check for abort criteria. Must be done after the check for the last tag
                                    # criteria, in case it is a subset of the criteria for last tag.
                                    if settings.get('abort_filters_config'):
                                        is_match = group_filter.evaluate(chain, sibling, settings['abort_filters_config'], context)
                                        #is_match = evluate_filters(sibling, self.filters, settings['abort_filters_config'])
                                        if is_match:
                                            # Encountered a tag that should not be in range
                                            break
                                    # keep appending until we find a tag matching criteria for last_tag
                                    r.append(sibling)

        tag_action = self.actions[settings['action_name']]
        tag_action.pre_run(chain, settings['action_settings'], context)
        number = 0
        for name, ranges in all_ranges.items():
            context.file_name = name
            for elements in ranges:
                number += 1
                tag_action.run(chain, number, elements, settings['action_settings'], context)
        tag_action.post_run(chain, settings['action_settings'], context)
        
        for name in dirty:
            root = container.parsed(name)
            etree.cleanup_namespaces(root, top_nsmap=XPNSMAP)
            container.dirty(name)

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        scope_ok = validate_scope(settings.get('where', 'text'))
        if scope_ok is not True:
            return scope_ok
        group_filter = self.filters['Group']
        apply_filters_opt = settings.get('apply_filters_opt', 'all')
        xpath_ok = False
        if apply_filters_opt == 'xpath':
            xpath = settings.get('xpath', '')
            if not xpath:
                return _('No xpath'), _('You must enter xpath expression')
            res = validate_xpath(xpath)
            if res is not True:
                return res
            xpath_ok = True
        css_ok = False
        if apply_filters_opt == 'css':
            css = settings.get('css', '')
            if not css:
                return _('No CSS Selector'), _('You must enter CSS Selector')
            res = validate_css(css)
            if res is not True:
                return res
            css_ok = True
        filters_config = settings.get('filters_config', {})
        tag_selection_opt = settings['tag_selection']
        #if (apply_filters_opt == 'all') and not filters_config:
            #_('No criteria'), _('You have to add at least one filter')
        if filters_config:
            # make an exception for empty filters in case of xpath or css pattern
            if (xpath_ok or css_ok) and not filters_config.get('config', {}).get('filters_config', {}):
                return True
            res = group_filter.validate(filters_config)
            if res is not True:
                return res
        if tag_selection_opt == 'range':
            if not filters_config:
                return _('First in ragne error'), _('Fitlers for first element in range not configured.')
            last_filters_config = settings['last_filters_config']
            if not last_filters_config:
                return _('Last in ragne error'), _('Fitlers for last element in range not configured.')
            res = group_filter.validate(settings['last_filters_config'])
            if not res is True:
                return res
            if settings['abort_filters_config']:
                res = group_filter.validate(settings['abort_filters_config'])
                if not res is True:
                    return res
        action_name = settings['action_name']
        if not action_name:
            return _('Missing action'), _('You must choose an action')
        else:
            tag_action = self.actions.get(action_name)
            if tag_action:
                res = tag_action.validate(settings['action_settings'])
                if not res is True:
                    return res
            else:
                return _('Missing action'), _(f'Cannot find an action with the name: {action_name}')
        return True

    def is_headless(self, settings):
        where = settings.get('where', 'text')
        return scope_is_headless(where)

    def config_widget(self):
        return ConfigWidget
