#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QGroupBox, QToolButton, QPushButton, QComboBox, QFrame,
                     QDialog, QCheckBox, QIcon)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.widgets import Dialog

from calibre_plugins.editor_chains.common_utils import get_icon, get_pixmap, ViewLog, reverse_lookup
from calibre_plugins.editor_chains.actions.style_actions.filters.group import GroupWidget

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/gui/__init__.py - exception when loading translations")


class FiltersDialog(Dialog):
    def __init__(self, parent, plugin_action, filters_config, title='Filters Dialog'):
        self.plugin_action = plugin_action
        self.filters = self.plugin_action.actions['Style Actions'].filters
        self.filters_config = filters_config
        self.title = title
        Dialog.__init__(self, title, 'editor-chains-filters-dialog', parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)
        
        self.group = GroupWidget(self, self.plugin_action, self.filters, identifier=self.title)
        self.group.load_settings(self.filters_config)
        l.addWidget(self.group)
        
        l.addWidget(self.bb)
        self.setMinimumSize(500, 500)

    def reject(self):
        Dialog.reject(self)

    def accept(self):
        group_filter = self.plugin_action.actions['Style Actions'].filters['Group']
        settings = self.group.save_settings()
        #all_valid = group_filter.validate(settings)
        #if all_valid is True:
            #self.filters_config = settings
            #Dialog.accept(self)
        #else:
            #return error_dialog(self, *all_valid, show=True)
        self.filters_config = settings
        Dialog.accept(self)

class ActionSettingsDialog(Dialog):
    def __init__(self, parent, plugin_action, action, action_config={}):
        self.action_config = action_config
        self.plugin_action = plugin_action
        self.action = action
        name = f'editor-chains-style-action-settings-dialog_{action.name}'
        Dialog.__init__(self, 'Action Settings Dialog', name, parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)

        config_widget_cls = self.action.config_widget()
        config_widget = config_widget_cls(self.plugin_action, self.action)
        l.addWidget(config_widget)
        config_widget.load_settings(self.action_config)  
        self.config_widget = config_widget
        
        l.addWidget(self.bb)

    def reject(self):
        Dialog.reject(self)

    def accept(self):
        new_settings = self.config_widget.save_settings()
        is_valid = self.action.validate(new_settings)
        if is_valid is True:
            self.action_config = new_settings
            Dialog.accept(self)
        else:
            return error_dialog(self, *is_valid, show=True)

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.style_actions = plugin_action.actions['Style Actions']
        self.filters = self.style_actions.filters
        self.actions = self.style_actions.actions
        self.action_settings_map = {}
        self.filters_config = {}
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        main_label = QLabel(_('Using this action, you can choose css or inline styles based on several '
                              'criteria, and then apply an action to them.'
        ))
        main_label.setWordWrap(True)
        l.addWidget(main_label)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        limits_gb = QGroupBox(_('Style types'))
        limits_l = QVBoxLayout()
        limits_gb.setLayout(limits_l)
        l.addWidget(limits_gb)
        self.css_chk = QCheckBox(_('Include Styles from CSS Sheets'))
        self.css_chk.setChecked(True)
        limits_l.addWidget(self.css_chk)
        self.styletag_chk = QCheckBox(_('Include Styles from Style Tag'))
        self.styletag_chk.setChecked(True)
        limits_l.addWidget(self.styletag_chk)
        self.inline_chk = QCheckBox(_('Include Styles from Inline Styles'))
        self.inline_chk.setChecked(True)
        limits_l.addWidget(self.inline_chk)

        style_group_box = QGroupBox(_('Restrict styles based on criteria (Optional)'))
        l.addWidget(style_group_box)
        style_group_box_l = QVBoxLayout()
        style_group_box.setLayout(style_group_box_l)

        self.style_button = QPushButton(_('Style Criteria'))
        style_group_box_l.addWidget(self.style_button)
        self.style_button.clicked.connect(self._on_style_button_clicked)

        action_group_box = QGroupBox(_('Apply action on selected Styles'))
        l.addWidget(action_group_box)
        action_group_box_l = QHBoxLayout()
        action_group_box.setLayout(action_group_box_l)
        
        self.action_combo = QComboBox()
        self.action_combo.addItems(self.actions.keys())
        self.action_combo.setCurrentIndex(-1)
        self.action_combo.currentTextChanged.connect(self._on_action_combo_changed)
        action_group_box_l.addWidget(self.action_combo)
        
        self.action_settings_button = QToolButton()
        self.action_settings_button.setIcon(get_icon('gear.png'))
        self.action_settings_button.clicked.connect(self._on_action_settings_button_clicked)
        action_group_box_l.addWidget(self.action_settings_button)

        self.setMinimumSize(600,600)
        #l.addStretch(1)
        
        self._on_action_combo_changed(self.action_combo.currentText())

    def _on_style_button_clicked(self):
        d = FiltersDialog(self, self.plugin_action, self.filters_config, title=_('Style criteria'))
        if d.exec_() == d.Accepted:
            self.filters_config = d.filters_config

    def _on_action_settings_button_clicked(self):
        action_name = self.action_combo.currentText()
        if action_name:
            action = self.actions[action_name]
            action_config_widget = action.config_widget()
            if action_config_widget:
                action_settings = self.action_settings_map.get(action_name, {})
                if issubclass(action_config_widget, QDialog):
                    # config widget is a dialog
                    d = action_config_widget(self, self.plugin_action, action, action_settings)
                else:
                    # config_widget is a QWidget
                    d = ActionSettingsDialog(self, self.plugin_action, action, action_settings)
                if d.exec_() == d.Accepted:
                    self.action_settings_map[action_name] = d.action_config

    def _on_action_combo_changed(self, text):
        action_config_widget = False
        if text:
            action = self.actions[text]
            action_config_widget = action.config_widget()
        self.action_settings_button.setEnabled(bool(text) and bool(action_config_widget))

    def load_settings(self, settings):
        if settings:
            self.css_chk.setChecked(settings['include_css'])
            self.styletag_chk.setChecked(settings['include_styletag'])
            self.inline_chk.setChecked(settings['include_inline'])
            action_name = settings['action_name']
            self.action_combo.setCurrentText(action_name)
            self.action_settings_map[action_name] = settings['action_settings']
            self.filters_config = settings['filters_config']

    def save_settings(self):
        settings = {}
        settings['include_css'] = self.css_chk.isChecked()
        settings['include_styletag'] = self.styletag_chk.isChecked()
        settings['include_inline'] = self.inline_chk.isChecked()
        action_name = self.action_combo.currentText()
        settings['action_name'] = action_name
        settings['action_settings'] = self.action_settings_map.get(action_name, {})
        settings['filters_config'] = self.filters_config
        return settings
