#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.style_actions.filters.base import StyleFilter
from calibre_plugins.editor_chains.common_utils import reverse_lookup

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/selector_text.py - exception when loading translations")

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

#        self.label = QLabel(_('Selector Text'))
#        l.addWidget(self.label, 0, 0, 1, 1)
        
        self.combo = QComboBox()
        self.combo.addItems(['matches', 'does not match'])
        l.addWidget(self.combo, 0, 1, 1, 1)
        
        self.value_edit = QLineEdit()
        l.addWidget(self.value_edit, 0, 2, 1, 1)
        
        self.regex_chk = QCheckBox(_('Regex'))
        l.addWidget(self.regex_chk, 0, 3, 1, 1)

    def load_settings(self, settings):
        if settings:
            self.combo.setCurrentText(settings['operator'])
            self.value_edit.setText(settings['value'])
            self.regex_chk.setChecked(settings['regex'])

    def save_settings(self):
        settings = {}
        settings['operator'] = self.combo.currentText()
        settings['value'] = self.value_edit.text()
        settings['regex'] = self.regex_chk.isChecked()
        return settings


class SelectorTextFilter(StyleFilter):

    name = 'Selector Text'
    
    def evaluate(self, chain, rule, settings, context, *args, **kwargs):
        selector_text = rule.selectorText
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        
        if is_regex:
            is_match = bool(regex.search(value, selector_text))
            if operator == 'matches':
                return is_match
            elif operator == 'does not match':
                return not is_match
        else:
            is_match = value == selector_text
            if operator == 'matches':
                return is_match
            elif operator == 'does not match':
                return not is_match

    def validate(self, settings):
        if not settings:
            return (_('Selector Text Filter Error'), _('You must configure this filter'))
        if not settings.get('operator'):
            return (_('Selector Text Filter Error'), _('You must choose an operator'))
        if not settings.get('value'):
            return (_('Selector Text Filter Error'), _('You must specify a tag name'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget     
