#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.style_actions.filters.base import StyleFilter

from calibre_plugins.editor_chains.actions.tag_actions.filters.style_property import (
        NUMERICAL_OPERATORS, OPERATORS, SUPPORTED_DIMENSIONS, unit_convert,
        unit_convert_from_pt, normalize_fontsize_property, FilterWidget)

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/prop_value.py - exception when loading translations")

class PropertyValueFilter(StyleFilter):

    name = 'Property Value'
    
    def evaluate(self, chain, rule, settings, context, *args, **kwargs):
        name = settings['name']
        value = settings['value']
        operator = settings['operator']
        is_regex = settings['regex']
        convert_unit = settings['unit']
        try:
            prop = rule.style.getProperties(name)[0]
        except IndexError:
            return False
        if prop:
            property_value = prop.propertyValue
        else:
            property_value = ''

        if not property_value: property_value = ''

        if property_value:
            if operator == 'is null':
                return False
            elif operator == 'not null':
                return True
        else:
            if operator == 'is null':
                return True
            else:
                return False

        if operator in NUMERICAL_OPERATORS.keys():
            if len(property_value) == 1:
                property_value = property_value[0]
                if property_value.type == 'DIMENSION':
                    numerical_value = property_value.value
                    if convert_unit:
                        target_dimension = convert_unit
                        if target_dimension not in SUPPORTED_DIMENSIONS:
                            return False
                        value_in_pt = unit_convert(numerical_value, property_value.dimension)
                        numerical_value = unit_convert_from_pt(value_in_pt, target_dimension)
                    comp = NUMERICAL_OPERATORS[operator]
                    try:
                        return comp(numerical_value, float(value))
                    except TypeError:
                        return False
                else:
                    return False
            else:
                return False

        if operator == 'matches':
            if is_regex:
                return bool(regex.search(value, property_value.cssText))
            else:
                return value == property_value.cssText
        elif operator == 'does not match':
            if is_regex:
                return not bool(regex.search(value, property_value.cssText))
            else:
                return value != property_value.cssText

    def validate(self, settings):
        if not settings:
            return (_('Property Value Filter Error'), _('You must configure this filter'))
        operator = settings.get('operator')
        if not operator:
            return (_('Property Value Filter Error'), _('You must choose an operator'))
        if operator == 'equal default value':
            return True
        if not settings.get('value'):
            if operator not in [ 'is null', 'not null']:
                return (_('Property Value Filter Error'), _('You must specify an property value'))
        if not settings.get('name'):
            return (_('Property Value Filter Error'), _('You must specify an property name'))
        is_regex = settings['regex']
        if is_regex:
            try:
                pattern = settings['value']
                regex.search(pattern, 'random text')
            except regex._regex_core.error:
                return _('Invalid regex'), _(f'Pattern ({pattern}) is not a valid regex')
        return True

    def filter_widget(self):
        return FilterWidget  
