#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import inspect

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit, QPushButton)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.widgets import Dialog

from calibre_plugins.editor_chains.actions.style_actions.filters.base import StyleFilter
from calibre_plugins.editor_chains.code_editor import CodeEditor, compile_code

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/tag_actions/filters/code.py - exception when loading translations")

CODE_EDITOR_TEMPLATE = '''
def evaluate(rule, file_name, data, chain, context, *args, **kwargs):
    # Enter you code here, it should return True if the elements meets the desired
    # criteria
    pass
'''

class FilterCodeEditor(CodeEditor):
    def accept(self):
        source = self.source
        try:
            mod = compile_code(source, self.mod_name)
        except Exception as err:
            return error_dialog(self, _('Invalid Python code'), _(
                'The code you created is not valid Python code, with error: %s') % err, show=True)

        try:
            func = mod['evaluate']
        except Exception as err:
            return error_dialog(self, _('Function not defined'), _(
                'You must define a a filter functino called "evaluate"') % err, show=True)

        Dialog.accept(self)

class FilterWidget(QWidget):
    def __init__(self, parent, plugin_action, filter_, *args, **kwargs):
        QWidget.__init__(self)
        self.parent = parent
        self.plugin_action = plugin_action
        self.filter_ = filter_
        self.source = ''
        self.template = CODE_EDITOR_TEMPLATE
        self._init_controls()

    def _init_controls(self):

        l = self.l = QHBoxLayout()
        self.setLayout(l)

        self.label = QLabel(_('Use code editor to filter tag'))
        l.addWidget(self.label)

        l.addStretch(1)

        self.editor_button = QPushButton(_('Code Editor'))
        self.editor_button.clicked.connect(self._editor_button_clicked)
        l.addWidget(self.editor_button)

    def _editor_button_clicked(self):
        d = FilterCodeEditor(mod_name='_default', template=self.source or self.template,
                       show_name=False)
        if d.exec_() == d.Accepted:
            self.source = d.source

    def load_settings(self, settings):
        if settings:
            self.source = settings['source']

    def save_settings(self):
        settings = {}
        settings['source'] = self.source
        return settings

class CodeFilter(StyleFilter):

    name = 'Code'
    
    def evaluate(self, chain, rule, settings, context, *args, **kwargs):
        data = chain.action_vars['data']
        file_name = context.file_name
        source = settings['source']
        mod = compile_code(source, 'module')
        evaluate = mod.get('evaluate')
        return evaluate(rule, file_name, data, chain, context)

    def validate(self, settings):
        if not settings:
            return (_('Code Filter Error'), _('You must configure this filter'))
        return True

    def filter_widget(self):
        return FilterWidget     
