#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.style_actions.filters.base import StyleFilter
from calibre_plugins.editor_chains.actions.style_actions.filters.group import GroupFilter
from calibre_plugins.editor_chains.actions.style_actions.filters.selector_text import SelectorTextFilter
from calibre_plugins.editor_chains.actions.style_actions.filters.prop_value import PropertyValueFilter
from calibre_plugins.editor_chains.actions.style_actions.filters.code import CodeFilter

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/__int__.py - exception when loading translations")



def get_user_filters(plugin_action):
    user_filters = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[StyleFilter]):
        name = cls.name
        # must define a name attribute, must be set and not clash with builtin names
        # which can be imported into the module manager by custom actions
        if name in ['', 'Style Filter']:
            continue
        user_filters[name] = cls            
    return user_filters

def get_all_filters(plugin_action):

    builtin_filters = OrderedDict()
    
    _builtin_filters = [
        GroupFilter,
        SelectorTextFilter,
        PropertyValueFilter,
        CodeFilter
    ]

    for filter_cls in _builtin_filters:
        builtin_filters[filter_cls.name] = filter_cls

    _user_filters = get_user_filters(plugin_action)

    all_filters = OrderedDict()
    user_filters = OrderedDict()
    
    for filter_name, filter_cls in builtin_filters.items():
        if filter_name in ['', 'Style Filter']:
            continue
        filter_ = filter_cls(plugin_action)
        all_filters[filter_name] = filter_

    for filter_name, filter_cls in _user_filters.items():
        # dont override builtin filters
        if filter_name in list(builtin_filters.keys()) + ['', 'Style Filter']:
            continue
        try:
            filter_ = filter_cls(plugin_action)
            all_filters[filter_name] = filter_
            user_filters[filter_name] = filter_
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Editor Chains: Style Actions: Error intializing user filter: {filter_name}\n{traceback.format_exc()}')

    return all_filters, builtin_filters, user_filters    
