#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QLineEdit, QGroupBox, QFrame, QSpinBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.style_actions.actions.base import StyleAction
from calibre_plugins.editor_chains.modules import compile_code

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/style_arithmetic.py - exception when loading translations")

class ActionWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        name_lbl = QLabel(_('Property Name'))
        l.addWidget(name_lbl, 0, 0, 1, 1)

        self.name_edit = QLineEdit()
        self.name_edit.setToolTip(_('Property name e.g. margin-top, text-indent, .... etc, supports regex expressions e.g. margin.*'))
        l.addWidget(self.name_edit, 0, 1, 1, 1)

        func_lbl = QLabel(_('Arithmetic Function'))
        l.addWidget(func_lbl, 1, 0, 1, 1)

        self.func_edit = QLineEdit()
        self.func_edit.setToolTip(_('Arithmetic operations to perform on value denoted by x. e.g x / 2'))
        l.addWidget(self.func_edit, 1, 1, 1, 1)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line, 2, 0, 1, 2)

        self.precision_gb = QGroupBox(_('Control precision for resulting float nubmers'))
        precision_l = QVBoxLayout()
        self.precision_gb.setLayout(precision_l)
        self.precision_gb.setCheckable(True)
        self.precision_gb.setChecked(False)
        l.addWidget(self.precision_gb, 3, 0, 1, 2)

        self.precision_spin = QSpinBox()
        self.precision_spin.setMinimum(0), self.precision_spin.setMaximum(10), self.precision_spin.setValue(2)
        precision_l.addWidget(self.precision_spin)

        l.setRowStretch(l.rowCount(), 1)
        self.setMinimumSize(400,200)

    def load_settings(self, settings):
        if settings:
            self.name_edit.setText(settings['name'])
            self.func_edit.setText(settings['func_string'])
            precision = settings.get('precision')
            if precision:
                self.precision_gb.setChecked(True)
                self.precision_spin.setValue(precision)

    def save_settings(self):
        settings = {}
        settings['name'] = self.name_edit.text()
        settings['func_string'] = self.func_edit.text().strip()
        if self.precision_gb.isChecked():
            settings['precision'] = self.precision_spin.value()
        return settings

class ArithmeticAction(StyleAction):

    name = 'Style Arithmetic'

    def run_function(self, func_string, val):
        ns = compile_code('func = lambda x: ' + func_string.strip())
        func = ns['func']
        res = func(val)
        return res

    def run(self, chain, number, rule, settings, context, *args, **kwargs):
        changed = False
        name = settings['name']
        func_string = settings['func_string']
        precision = settings.get('precision')
        props = self.get_props(name, rule.style)
        for prop in props:
            for value in prop.propertyValue:
                if value.type in ('DIMENSION', 'NUMBER'):
                    dimension = value.dimension or ''
                    numerical_value = value.value
                    try:
                        res = self.run_function(func_string, numerical_value)
                        res = self.float_precision(res, precision)
                        value.cssText = f'{res}{dimension}'
                        changed = True
                    except Exception as e:
                        import traceback
                        print(traceback.format_exc())
        return changed

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this filter'))
        if not settings.get('name'):
            return (_('Settings Error'), _('You must specify an property name'))
        func_string = settings.get('func_string').strip()
        if not func_string:
            (_('Settings Error'), _('You must specify an function'))
        try:
            self.run_function(func_string, 1)
        except Exception as e:
            (_('Settings Error'), _(f'Function failed with the following error: {e}'))
        return True

    def config_widget(self):
        return ActionWidget
