#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QLineEdit, QComboBox, QGroupBox, QSpinBox, QFrame)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.editor_chains.actions.style_actions.actions.base import StyleAction
from calibre_plugins.editor_chains.actions.tag_actions.filters.style_property import (
        SUPPORTED_DIMENSIONS, unit_convert,
        unit_convert_from_pt, normalize_fontsize_property)

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/convert.py - exception when loading translations")

CONVERT_SUPPORTED_DIMENSIONS = list(set(SUPPORTED_DIMENSIONS).difference(['rem']))

class ActionWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        name_lbl = QLabel(_('Property Name'))
        l.addWidget(name_lbl, 0, 0, 1, 1)
        self.name_edit = QLineEdit()
        self.name_edit.setToolTip(_('Property name e.g. margin-top, text-indent, .... etc, supports regex expressions e.g. margin.*'))
        l.addWidget(self.name_edit, 0, 1, 1, 1)

        dim_lbl = QLabel(_('Target Dimension'))
        l.addWidget(dim_lbl, 1, 0, 1, 1)
        self.unit_combo = QComboBox()
        self.unit_combo.setToolTip(_(
            'Convert Property from one dimension to another.\n'
            'Supported Output Units are: {}'
            '\nSame units are supported for input, in addtion to em'.format("\n".join(["    " + x for x in CONVERT_SUPPORTED_DIMENSIONS]))
        ))
        self.unit_combo.addItems(CONVERT_SUPPORTED_DIMENSIONS)
        self.unit_combo.setCurrentIndex(-1)
        l.addWidget(self.unit_combo, 1, 1, 1, 1)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line, 2, 0, 1, 2)

        self.precision_gb = QGroupBox(_('Control precision for resulting float nubmers'))
        precision_l = QVBoxLayout()
        self.precision_gb.setLayout(precision_l)
        self.precision_gb.setCheckable(True)
        self.precision_gb.setChecked(False)
        l.addWidget(self.precision_gb, 3, 0, 1, 2)

        self.precision_spin = QSpinBox()
        self.precision_spin.setMinimum(0), self.precision_spin.setMaximum(10), self.precision_spin.setValue(2)
        precision_l.addWidget(self.precision_spin)

        l.setRowStretch(l.rowCount(), 1)
        self.setMinimumSize(400,200)

    def load_settings(self, settings):
        if settings:
            self.name_edit.setText(settings['name'])
            self.unit_combo.setCurrentText(settings.get('unit', ''))
            precision = settings.get('precision')
            if precision:
                self.precision_gb.setChecked(True)
                self.precision_spin.setValue(precision)

    def save_settings(self):
        settings = {}
        settings['name'] = self.name_edit.text()
        settings['unit'] = self.unit_combo.currentText()
        if self.precision_gb.isChecked():
            settings['precision'] = self.precision_spin.value()
        return settings


class ConvertUnitAction(StyleAction):

    name = 'Convert Unit'

    def run(self, chain, number, rule, settings, context, *args, **kwargs):
        changed = False
        name = settings['name']
        target_dimension = settings['unit']
        precision = settings.get('precision')
        props = self.get_props(name, rule.style)
        for prop in props:
            for value in prop.propertyValue:
                if value.type == 'DIMENSION':
                    if (target_dimension in CONVERT_SUPPORTED_DIMENSIONS) and \
                            (value.dimension in CONVERT_SUPPORTED_DIMENSIONS):
                        numerical_value = value.value
                        value_in_pt = unit_convert(numerical_value, value.dimension)
                        numerical_value = unit_convert_from_pt(value_in_pt, target_dimension)
                        numerical_value = self.float_precision(numerical_value, precision)
                        value.cssText = f'{numerical_value}{target_dimension}'
                        changed = True
        return changed

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this filter'))
        if not settings.get('name'):
            return (_('Settings Error'), _('You must specify an property name'))
        if not settings.get('unit'):
            return (_('Settings Error'), _('You must specify a unit'))
        try:
            self.run_function(func_string, 1)
        except Exception as e:
            (_('Settings Error'), _(f'Function failed with the following error: {e}'))
        return True

    def config_widget(self):
        return ActionWidget
