#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit, QPushButton, QDialog)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.tweak_book.widgets import Dialog

from calibre_plugins.editor_chains.actions.style_actions.actions.base import StyleAction
from calibre_plugins.editor_chains.code_editor import CodeEditor, compile_code

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/actions/code.py - exception when loading translations")

CODE_EDITOR_TEMPLATE = '''
def run(rule, number, file_name, data, chain, context, *args, **kwargs):
    # Enter you code here, you must return True for changes to be comitted
    pass
'''

class ActionCodeEditor(CodeEditor):

    def __init__(self, parent, plugin_action, action, action_settings):
        CodeEditor.__init__(self, '_default', template=action_settings.get('source','') or CODE_EDITOR_TEMPLATE,
                       show_name=False)

    def load_settings(self, settings):
        if settings:
            self.source = settings['source']
            self.source_code.setPlainText(settings['source'] or CODE_EDITOR_TEMPLATE)

    def save_settings(self):
        settings = {}
        settings['source'] = self.source
        return settings

    def accept(self):
        source = self.source
        try:
            mod = compile_code(source, self.mod_name)
        except Exception as err:
            return error_dialog(self, _('Invalid Python code'), _(
                'The code you created is not valid Python code, with error: %s') % err, show=True)

        try:
            func = mod['run']
        except Exception as err:
            return error_dialog(self, _('Function not defined'), _(
                'You must define a a filter functino called "run"') % err, show=True)
        self.action_config = self.save_settings()
        Dialog.accept(self)

class CodeAction(StyleAction):

    name = 'Code'
    
    def run(self, chain, number, rule, settings, context, *args, **kwargs):
        data = chain.action_vars['data']
        file_name = context.file_name
        source = settings['source']
        mod = compile_code(source, 'module')
        run = mod.get('run')
        return run(rule, number, file_name, data, chain, context)

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action'))
        return True

    def config_widget(self):
        return ActionCodeEditor
