#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
import regex

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.css_transform_rules import all_properties


class StyleFunctions(object):
    def float_precision(self, number, precision=None):
        if isinstance(number, float):
            if precision is not None:
                number = round(number, precision)
                if number.is_integer():
                    number = int(number)
        return number

    def get_props(self, name, style):
        props = []
        name_regex = not bool(regex.search(r'^[a-z\-]+$', name))
        if name_regex:
            props = all_properties(style)
            props = [x for x in props if regex.search(name, x.name)]
        else:
            try:
                prop = style.getProperties(name)[0]
                props.append(prop)
            except IndexError:
                pass
        return props

    def delete_rule_if_empty(self, rule):
        # delete rule if style has no properties
        if not all_properties(rule.style):
            sheet = rule.parentStyleSheet
            if sheet:
                sheet.deleteRule(rule)
            else:
                # Inline style
                rule.style.cssText = ''

class StyleAction(StyleFunctions):

    # Actions must have a unique name attribute.
    name = 'Style Action'

    '''
    This is the base class for all actions
    '''
    def __init__(self, plugin_action):
        '''
        All actions are intialized at startup
        The are re-initialized on adding or modifying custom modules
        '''
        self.plugin_action = plugin_action

    def run(self, chain, number, rule, settings, context, *args, **kwargs):
        raise NotImplementedError

    def config_widget(self):
        '''
        If you want your filter to have settings widget, implement this method
        This should return a Qwidget (not dialog) with the following methods:
        [*] __init__(self, plugin_action)
        [*] save_settings(settings)
                This method is used to save the settings from the widget
                it should return a dictionary containing all the settings
        [*] load_settings(self, settings)
                This method is used to load the saved settings into the
                widget
        '''
        return None

    def validate(self, settings):
        '''
        Validate settings dict. This is called when you press the OK button in config dialog.
        changes are applied only if the this methods returns True.
        It is called also when verifying the chain validity on before running.
        If the setting in the dictionary are not valid, return a tuple
        of two strings (message, details), these will be displayed as a warning dialog to
        the user and the process will be aborted.
        '''
        return True

    def default_settings(self):
        '''
        default settings to be used if no settings are configured
        '''
        return {}




