#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2023, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import regex

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.css_transform_rules import all_properties

from css_parser.css import CSSStyleDeclaration

from calibre_plugins.editor_chains.actions.style_actions.actions.base import StyleAction

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/filters/add_props.py - exception when loading translations")


class ActionWidget(QWidget):
    def __init__(self, plugin_action, action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.action = action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        props_lbl = QLabel(_('Properties'))
        l.addWidget(props_lbl, 0, 0, 1, 1)
        self.props_edit = QLineEdit()
        self.props_edit.setToolTip(_('Property to add. You can add multiple properties separated by a semicolon.'))
        l.addWidget(self.props_edit, 0, 1, 1, 1)

        l.setRowStretch(l.rowCount(), 1)
        self.setMinimumSize(400,100)

    def load_settings(self, settings):
        if settings:
            self.props_edit.setText(settings['props'])

    def save_settings(self):
        settings = {}
        settings['props'] = self.props_edit.text()
        return settings


class AddPropertiesAction(StyleAction):

    name = 'Add Properties'

    def run(self, chain, number, rule, settings, context, *args, **kwargs):
        changed = False
        props_text = settings['props']
        style = CSSStyleDeclaration(cssText=props_text)
        if style.valid:
            props = all_properties(style)
            for prop in props:
                rule.style.setProperty(prop.name, prop.value, priority=prop.priority)
                changed = True

        return changed

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this filter'))
        props_text = settings.get('props')
        if not props_text:
            return (_('Settings Error'), _('You must specify properties to add'))
        style = CSSStyleDeclaration(cssText=props_text)
        if not style.valid:
            return (_('Settings Error'), _('Some of the properties you added are invalid'))
        return True

    def config_widget(self):
        return ActionWidget
