#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~
__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from calibre import prints, force_unicode
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.base import OEB_STYLES, OEB_DOCS, serialize, XPNSMAP, etree
from css_parser.css import CSSStyleRule

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.actions.style_actions.actions.base import StyleFunctions
from calibre_plugins.editor_chains.actions.style_actions.actions import get_all_style_actions
from calibre_plugins.editor_chains.actions.style_actions.filters import get_all_filters
from calibre_plugins.editor_chains.actions.style_actions.gui import ConfigWidget

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/style_actions/__init__.py - exception when loading translations")

class Context(StyleFunctions):
    pass

class StyleActions(EditorAction):

    name = 'Style Actions'
    _is_builtin_ = True
    headless = True

    def on_modules_update(self, all_objects):
        if DEBUG:
            prints('Editor Chains: Style Actions: running on_modules_update()')
        self.actions, self.builtin_actions, self.user_actions = get_all_style_actions(self.plugin_action)
        self.filters, self.builtin_filters, self.user_filters = get_all_filters(self.plugin_action)

    def run(self, chain, settings, *args, **kwargs):
        filters_config = settings['filters_config']
        group_filter = self.filters['Group']
        context = Context()
        # action vars
        action_vars = chain.action_vars
        # used by code filter, and code action
        action_vars['data'] = {}
        action_vars['context'] = context

        #
        container = chain.current_container

        matched_rules = []
        dirty_names = set()
        dirty_sheets = set()
        name_sheet_map = {}
        inlinerule_elem_map = {}
        dirty_rules = set()
        # If not filters, add all styles
        is_filters = filters_config.get('config', {}).get('filters_config', [])

        for name, media_type in container.mime_map.items():
            context.file_name = name
            if media_type in OEB_STYLES:
                if settings['include_css']:
                    sheet = container.parsed(name)
                    for rule in sheet.cssRules:
                        if rule.type == rule.STYLE_RULE:
                            if is_filters:
                                is_match = group_filter.evaluate(chain, rule, filters_config, context)
                            else:
                                is_match = True
                            if is_match:
                                matched_rules.append((rule,name))
            elif media_type in OEB_DOCS:
                if settings['include_styletag']:
                    style_tags = container.parsed(name).xpath('//h:style', namespaces=XPNSMAP)
                    for style_tag in style_tags:
                        if style_tag.text and style_tag.get('type', None) in {None, 'text/css'}:
                            sheet = container.parse_css(style_tag.text)
                            name_sheet_map[name] = sheet
                            for rule in sheet.cssRules:
                                if rule.type == rule.STYLE_RULE:
                                    if is_filters:
                                        is_match = group_filter.evaluate(chain, rule, filters_config, context)
                                    else:
                                        is_match = True
                                    if is_match:
                                        matched_rules.append((rule,name))
                                        dirty_sheets.add((sheet,style_tag,name))
                if settings['include_inline']:
                    for elem in container.parsed(name).xpath('//*[@style]'):
                        block = container.parse_css(elem.get('style'), is_declaration=True)
                        rule = CSSStyleRule(selectorText='', style=block)
                        if is_filters:
                            is_match = group_filter.evaluate(chain, rule, filters_config, context)
                        else:
                            is_match = True
                        if is_match:
                            matched_rules.append((rule,name))
                            inlinerule_elem_map[rule] = elem

        style_action = self.actions[settings['action_name']]
        for number, match in enumerate(matched_rules, 1):
            rule = match[0]
            name = match[1]
            context.file_name = name
            res = style_action.run(chain, number, rule, settings['action_settings'], context)
            if res is True:
                dirty_rules.add(rule)
                dirty_names.add(name)

        for rule, elem in inlinerule_elem_map.items():
            # overwrite only if rule has been dirtied
            if rule in dirty_rules:
                if rule.style.cssText == '':
                    # deleted by DeleteRuleAction
                    del elem.attrib['style']
                else:
                    elem.set('style', force_unicode(rule.style.getCssText(separator=' '), 'utf-8'))

        for sheet, style_tag, name in dirty_sheets:
            style_tag.text = serialize(sheet, 'text/css', pretty_print=True)
            dirty_names.add(name)

        for name in dirty_names:
            root = container.parsed(name)
            if name in container.manifest_items_of_type(OEB_DOCS):
                etree.cleanup_namespaces(root, top_nsmap=XPNSMAP)
            container.dirty(name)

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        group_filter = self.filters['Group']
        apply_filters_opt = settings.get('apply_filters_opt', 'all')

        filters_config = settings.get('filters_config', {})
        if filters_config:
            res = group_filter.validate(filters_config)
            if res is not True:
                return res
        action_name = settings['action_name']
        if not action_name:
            return _('Missing action'), _('You must choose an action')
        else:
            style_action = self.actions.get(action_name)
            if style_action:
                res = style_action.validate(settings['action_settings'])
                if not res is True:
                    return res
            else:
                return _('Missing action'), _(f'Cannot find an action with the name: {action_name}')
        return True

    def is_headless(self, settings):
        return True

    def config_widget(self):
        return ConfigWidget
