#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import time
import copy
from functools import partial

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QStackedLayout, QGroupBox, QRadioButton, QComboBox, QBrush,
                     QAbstractTableModel, QModelIndex, QToolButton, QLabel, QFrame,
                     QPushButton, QCheckBox)

from calibre import prepare_string_for_xml
from calibre.constants import iswindows
from calibre.gui2.widgets2 import FlowLayout
from calibre.gui2.tweak_book import tprefs, editor_name
from calibre.gui2.tweak_book.function_replace import functions, Function, FunctionBox
from calibre.gui2.tweak_book.search import (reorder_files, get_search_regex, get_search_function,
                                            SearchWidget, InvalidRegex, NoSuchFunction, get_search_name,
                                            PushButton, HistoryBox, DirectionBox, ModeBox)
from calibre.ebooks.oeb.base import OEB_DOCS, OEB_STYLES
from polyglot.io import PolyglotStringIO
from polyglot.builtins import iteritems, error_message


from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.common_utils import get_icon
from calibre_plugins.editor_chains.gui.models import UP, DOWN
from calibre_plugins.editor_chains.gui.views import TableView
from calibre_plugins.editor_chains.scope import scope_names, ScopeWidget, validate_scope, scope_is_headless

try:
    load_translations()
except NameError:
    prints("EditorChains::actions/search_replace.py - exception when loading translations")


class SavedSearchesModel(QAbstractTableModel):

    def __init__(self, plugin_action, saved_searches, all_saved_searches):
        QAbstractTableModel.__init__(self)
        self.populate(saved_searches)
        self.all_saved_searches = all_saved_searches
        self.col_map = ['name']
        self.editable_columns = []
        self.optional_cols = []
        self.hidden_cols = []
        self.col_min_width = {
            'name': 300
        }
        all_headers = [_('Search')]
        self.headers = all_headers

    def populate(self, saved_searches):
        saved_searches = [{'name': s} for s in saved_searches]
        self.saved_searches = []
        for saved_search in saved_searches:
            self.insertRow(self.rowCount(QModelIndex()))
            self.saved_searches[-1] = saved_search
        self.layoutChanged.emit()

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.saved_searches)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return self.headers[section]
        elif role == Qt.DisplayRole and orientation == Qt.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.saved_searches):
            return None
        saved_search = self.saved_searches[row]
        col_name = self.col_map[col]
        value = saved_search.get(col_name, '')
        
        if role in [Qt.DisplayRole, Qt.UserRole, Qt.EditRole]:
            return value
        elif role == Qt.ForegroundRole:
            if col_name == 'name':
                if value not in self.all_saved_searches:
                    return QBrush(Qt.red)

        elif role == Qt.ToolTipRole:
            if col_name == 'name':
                if value not in self.all_saved_searches:
                    return _('Saved search is not currently available')                   

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        saved_search = self.saved_searches[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.EditRole:
            saved_search[col_name] = val[col_name]
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            saved_search = self.saved_searches[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemIsEditable
        return flags
        
    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            saved_search = {}
            saved_search['name'] = ''
            self.saved_searches.insert(row + i, saved_search)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.saved_searches.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.saved_searches.pop(row)
            self.saved_searches.insert(row+direction, pop)
        self.layoutChanged.emit()

class SavedSearchTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.setShowGrid(False)
        self.verticalHeader().setVisible(False)
        self.horizontalHeader().setVisible(False)

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

class SavedSearchesWidget(QWidget):

    def __init__(self, parent, plugin_action, all_saved_searches):
        self.plugin_action = plugin_action
        self.all_saved_searches = all_saved_searches
        QWidget.__init__(self)
        self.setup_ui()

    def setup_ui(self):
        l = QGridLayout()
        self.setLayout(l)

        avail_lbl = QLabel(_('Available Searches:'), self)
        avail_lbl.setStyleSheet('QLabel { font-weight: bold; }')
        l.addWidget(avail_lbl, 0, 0, 1, 1)

        chosen_lbl = QLabel(_('Chosen Searches:'), self)
        chosen_lbl.setStyleSheet('QLabel { font-weight: bold; }')
        l.addWidget(chosen_lbl, 0, 2, 1, 1)

        self.avail_tbl = SavedSearchTable(self)
        avail_model = SavedSearchesModel(self.plugin_action, [], self.all_saved_searches)
        self.avail_tbl.set_model(avail_model)
        l.addWidget(self.avail_tbl, 1, 0, 1, 1)

        move_button_layout = QVBoxLayout()
        l.addLayout(move_button_layout, 1, 1, 1, 1)

        self.add_btn = QToolButton(self)
        self.add_btn.setIcon(get_icon('plus.png'))
        self.add_btn.setToolTip(_('Add the selected saved_search'))
        self.remove_btn = QToolButton(self)
        self.remove_btn.setIcon(get_icon('minus.png'))
        self.remove_btn.setToolTip(_('Remove the selected saved_search'))

        move_button_layout.addStretch(1)
        move_button_layout.addWidget(self.add_btn)
        move_button_layout.addWidget(self.remove_btn)
        move_button_layout.addStretch(1)
        
        self.chosen_tbl = SavedSearchTable(self)
        chosen_model = SavedSearchesModel(self.plugin_action, [], self.all_saved_searches)
        self.chosen_tbl.set_model(chosen_model)
        l.addWidget(self.chosen_tbl, 1, 2, 1, 1)

        self.add_btn.clicked.connect(partial(self.move_rows, self.avail_tbl, self.chosen_tbl))
        self.remove_btn.clicked.connect(partial(self.move_rows, self.chosen_tbl, self.avail_tbl))

        sort_button_layout = QVBoxLayout()
        l.addLayout(sort_button_layout, 1, 3, 1, 1)

        self.up_btn = QToolButton(self)
        self.up_btn.setIcon(get_icon('arrow-up.png'))
        self.up_btn.setToolTip(_('Move the selected item up'))
        self.up_btn.clicked.connect(partial(self.chosen_tbl.move_rows, UP))
        self.down_btn = QToolButton(self)
        self.down_btn.setIcon(get_icon('arrow-down.png'))
        self.down_btn.setToolTip(_('Move the selected item down'))
        self.down_btn.clicked.connect(partial(self.chosen_tbl.move_rows, DOWN))
        sort_button_layout.addWidget(self.up_btn)
        sort_button_layout.addStretch(1)
        sort_button_layout.addWidget(self.down_btn)

        self.where_box = ScopeWidget(self, self.plugin_action, orientation='vertical',
                            hide=[],
                            headless=self.plugin_action.gui is None)
        l.addWidget(self.where_box)
        
        for tbl in [self.avail_tbl, self.chosen_tbl]:
            tbl.pressed.connect(partial(self.table_item_pressed, tbl))
        
        for btn in [self.add_btn, self.remove_btn, self.up_btn, self.down_btn]:
            btn.clicked.connect(self._on_table_selection_change)

        self._on_table_selection_change()

    def move_rows(self, from_tbl, to_tbl):
        moved_searches = []
        from_m = from_tbl.model()
        from_sm = from_tbl.selectionModel()
        to_m = to_tbl.model()
        selrows = from_sm.selectedRows()
        srows = sorted([selrow.row() for selrow in selrows], reverse=True)
        try:
            last_row = max(srows)
        except:
            last_row = 0
        for row in srows:
            pop = copy.deepcopy(from_m.saved_searches[row])
            moved_searches.insert(0, pop)
            from_m.removeRow(row)
        moved_count = 0
        for search in moved_searches:
            if from_tbl == self.chosen_tbl and (search['name'] not in self.all_saved_searches):
                continue
            moved_count += 1
            to_m.insertRow(to_m.rowCount(QModelIndex()))
            to_m.saved_searches[-1] = search
            to_m.layoutChanged.emit()
        select_row = last_row - moved_count + 1
        from_m.layoutChanged.emit()
        if select_row < len(from_m.saved_searches):
            from_tbl.select_rows([from_m.index(select_row, 0, QModelIndex())])
        from_m.layoutChanged.emit()
        to_m.layoutChanged.emit()

    def table_item_pressed(self, tbl):
        if tbl == self.avail_tbl:
            self.chosen_tbl.clearSelection()
        else:
            self.avail_tbl.clearSelection()
        self._on_table_selection_change()

    def _on_table_selection_change(self):
        avail_on = self.avail_tbl.hasFocus() and len(self.avail_tbl.selectionModel().selectedRows()) > 0
        chosen_on = self.chosen_tbl.hasFocus() and len(self.chosen_tbl.selectionModel().selectedRows()) > 0
        self.add_btn.setEnabled(avail_on)
        self.remove_btn.setEnabled(chosen_on)
        self.up_btn.setEnabled(chosen_on)
        self.down_btn.setEnabled(chosen_on)

    def populate(self, chosen_saved_searches):
        avail_saved_searches = [s for s in self.all_saved_searches if s not in chosen_saved_searches]  
        self.chosen_tbl.model().populate(chosen_saved_searches)
        self.avail_tbl.model().populate(avail_saved_searches)  

    def get_saved_searches(self):
        return [s['name'] for s in self.chosen_tbl.model().saved_searches]

#================

class FakeBoss(object):
    def __init__(self, container):
        self.current_metadata = container.mi

def replace_function_init_env(container, repl, name=''):
        repl.context_name = name or ''
        repl.match_index = 0
        repl.boss = FakeBoss(container)
        repl.data = {}
        repl.debug_buf = PolyglotStringIO()
        repl.functions = {name:func.mod for name, func in iteritems(functions()) if func.mod is not None}    

def search_from_saved_search(saved_search, where='text'):
    saved_search = copy.deepcopy(saved_search)
    saved_search['case_sensitive'] = saved_search.get('case_sensitive') or False
    saved_search['dot_all'] = saved_search.get('dot_all') or False
    saved_search['wrap'] = saved_search.get('wrap') or False
    saved_search['mode'] = saved_search.get('mode') or 'normal'
    saved_search['find'] = saved_search.get('find') or ''
    saved_search['replace'] = saved_search.get('replace') or ''
    saved_search['direction'] = saved_search.get('direction') or 'down'
    saved_search['where'] = where
    return saved_search

def process_searches(searches, validate=False):
    if isinstance(searches, dict):
        searches = [searches]
    wrap = searches[0]['wrap']
    errfind = searches[0]['find']
    if len(searches) > 1:
        errfind = _('the selected searches')

    search_names = [get_search_name(search) for search in searches]

    try:
        searches = [(get_search_regex(search), get_search_function(search)) for search in searches]
        if validate:
            return True
        return searches
    except InvalidRegex as e:
        return _('Invalid regex'), '<p>' + _(
            'The regular expression you entered is invalid: <pre>{0}</pre>With error: {1}').format(
                prepare_string_for_xml(e.regex), error_message(e))
    except NoSuchFunction as e:
        return _('No such function'), '<p>' + _(
            'No replace function with the name: %s exists') % prepare_string_for_xml(error_message(e))


def replace_all(filenames, container, searches, current_editor, raw_data={}):
    count = 0
    updates = set()

    if current_editor is not None:
        # Search replace in marked text
        name = editor_name(current_editor)
        for p, repl in searches:
            current_editor.all_in_marked(p, repl)
        updates.add(name)
        raw_data[name] = container.raw_data(name)
        return updates, raw_data

    if not filenames:
        return updates, raw_data
    lfiles = filenames

    for n in lfiles:
        if not raw_data.get(n):
            raw = container.raw_data(n)
            raw_data[n] = raw

    for p, repl in searches:
        repl_is_func = isinstance(repl, Function)
        file_iterator = lfiles
        if repl_is_func:
            replace_function_init_env(container, repl)
            if repl.file_order is not None and len(lfiles) > 1:
                file_iterator = reorder_files(file_iterator, repl.file_order)
        for n in file_iterator:
            raw = raw_data[n]
            if repl_is_func:
                repl.context_name = n
            raw, num = p.subn(repl, raw)
            if num > 0:
                updates.add(n)
                raw_data[n] = raw
            count += num
        if repl_is_func:
            repl.end()

    return updates, raw_data

def write_all(container, updates, raw_data):
    for n in updates:
        raw = raw_data[n]
        try:
            with container.open(n, 'wb') as f:
                f.write(raw.encode('utf-8'))
        except PermissionError:
            if not iswindows:
                raise
            time.sleep(2)
            with container.open(n, 'wb') as f:
                f.write(raw.encode('utf-8'))


#==========================
# Config Widget
#==========================

class ModifiedSearchWidget(SearchWidget):
    def __init__(self, parent, plugin_action):
        self.plugin_action = plugin_action

        QWidget.__init__(self, parent)
        self.l = l = QGridLayout(self)
        left, top, right, bottom = l.getContentsMargins()
        l.setContentsMargins(0, 0, right, 0)

        self.fl = fl = QLabel(_('&Find:'))
        fl.setAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignCenter)
        self.find_text = ft = HistoryBox(self, _('Clear search &history'))
        ft.save_search.connect(self.save_search)
        ft.show_saved_searches.connect(self.show_saved_searches)
        ft.initialize('tweak_book_find_edit')
        connect_lambda(ft.lineEdit().returnPressed, self, lambda self: self.search_triggered.emit('find'))
        fl.setBuddy(ft)
        l.addWidget(fl, 0, 0)
        l.addWidget(ft, 0, 1)
        l.setColumnStretch(1, 10)

        self.rl = rl = QLabel(_('&Replace:'))
        rl.setAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignCenter)
        self.replace_text = rt = HistoryBox(self, _('Clear replace &history'))
        rt.save_search.connect(self.save_search)
        rt.show_saved_searches.connect(self.show_saved_searches)
        rt.initialize('tweak_book_replace_edit')
        rl.setBuddy(rt)
        self.replace_stack1 = rs1 = QVBoxLayout()
        self.replace_stack2 = rs2 = QVBoxLayout()
        rs1.addWidget(rl), rs2.addWidget(rt)
        l.addLayout(rs1, 1, 0)
        l.addLayout(rs2, 1, 1)

        self.rl2 = rl2 = QLabel(_('F&unction:'))
        rl2.setAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignCenter)
        self.functions = fb = FunctionBox(self, show_saved_search_actions=True)
        fb.show_saved_searches.connect(self.show_saved_searches)
        fb.save_search.connect(self.save_search)
        rl2.setBuddy(fb)
        rs1.addWidget(rl2)
        self.functions_container = w = QWidget(self)
        rs2.addWidget(w)
        self.fhl = fhl = QHBoxLayout(w)
        fhl.setContentsMargins(0, 0, 0, 0)
        fhl.addWidget(fb, stretch=10, alignment=Qt.AlignmentFlag.AlignVCenter)
        self.ae_func = b = QPushButton(_('Create/&edit'), self)
        b.clicked.connect(self.edit_function)
        b.setToolTip(_('Create a new function, or edit an existing function'))
        fhl.addWidget(b)
        self.rm_func = b = QPushButton(_('Remo&ve'), self)
        b.setToolTip(_('Remove this function'))
        b.clicked.connect(self.remove_function)
        fhl.addWidget(b)
        self.fsep = f = QFrame(self)
        f.setFrameShape(QFrame.Shape.VLine)
        fhl.addWidget(f)

        self.fb = fb = PushButton(_('Fin&d'), 'find', self)
        self.rfb = rfb = PushButton(_('Replace a&nd Find'), 'replace-find', self)
        self.rb = rb = PushButton(_('Re&place'), 'replace', self)
        self.rab = rab = PushButton(_('Replace &all'), 'replace-all', self)
        l.addWidget(fb, 0, 2)
        l.addWidget(rfb, 0, 3)
        l.addWidget(rb, 1, 2)
        l.addWidget(rab, 1, 3)

        self.ml = ml = QLabel(_('&Mode:'))
        self.ol = ol = FlowLayout()
        ml.setAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
        l.addWidget(ml, 2, 0)
        l.addLayout(ol, 2, 1, 1, 3)
        self.mode_box = mb = ModeBox(self)
        ml.setBuddy(mb)
        ol.addWidget(mb)

        self.where_box = wb = ScopeWidget(self, self.plugin_action, hide=[],
                                orientation='vertical', headless=self.plugin_action.gui is None)
        ol.addWidget(wb)

        self.direction_box = db = DirectionBox(self)
        ol.addWidget(db)

        self.cs = cs = QCheckBox(_('&Case sensitive'))
        ol.addWidget(cs)

        self.wr = wr = QCheckBox(_('&Wrap'))
        wr.setToolTip('<p>'+_('When searching reaches the end, wrap around to the beginning and continue the search'))
        ol.addWidget(wr)

        self.da = da = QCheckBox(_('&Dot all'))
        da.setToolTip('<p>'+_("Make the '.' special character match any character at all, including a newline"))
        ol.addWidget(da)

        self.mode_box.currentIndexChanged.connect(self.mode_changed)
        self.mode_changed(self.mode_box.currentIndex())

        # Block signals to prevent initiating search when user clicks enter
        self.find_text.blockSignals(True)

        for w in [self.fb, self.rfb, self.rb, self.rab, self.wr, self.direction_box]:
            w.setVisible(False)

    @property
    def state(self):
        ans = {x:getattr(self, x) for x in self.DEFAULT_STATE}
        ans['find'] = self.find
        ans['replace'] = self.replace
        return ans

    @state.setter
    def state(self, val):
        for x in list(self.DEFAULT_STATE.keys()) + ['find','replace']:
            if x in val:
                setattr(self, x, val[x])
        if val['mode'] == 'function':
            self.functions.setText(val['replace'])

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        opt_l = QHBoxLayout()
        l.addLayout(opt_l)

        self.stacked_l = stacked_l = QStackedLayout()
        l.addLayout(stacked_l)

        self.search_replace_opt = QRadioButton(_('Search & Replace'))
        self.search_replace_opt.setChecked(True)
        opt_l.addWidget(self.search_replace_opt)
        
        self.saved_searches_opt = QRadioButton(_('Add saved searches'))
        opt_l.addWidget(self.saved_searches_opt)

        for opt in [self.search_replace_opt, self.saved_searches_opt]:
            opt.toggled.connect(self._on_opt_toggled)
                
        self.search_widget = ModifiedSearchWidget(self, self.plugin_action)
        stacked_l.addWidget(self.search_widget)

        all_saved_searches = [s['name'] for s in tprefs['saved_searches']]
        self.saved_searches_widget = SavedSearchesWidget(self, self.plugin_action, all_saved_searches)
        self.saved_searches_widget.populate([])
        stacked_l.addWidget(self.saved_searches_widget)
        
        self._on_opt_toggled()
        
        l.addStretch(1)

        self.setMinimumSize(500,300)

    def _on_opt_toggled(self, *args):
        if self.search_replace_opt.isChecked():
            self.stacked_l.setCurrentIndex(0)
        elif self.saved_searches_opt.isChecked():
            self.stacked_l.setCurrentIndex(1)

    def load_settings(self, settings):
        if settings:
            opt = settings.get('opt', 'search_replace')
            if opt == 'search_replace':
                self.search_replace_opt.setChecked(True)
                self.search_widget.state = settings['searches'][0]
            elif opt == 'saved_searches':
                self.saved_searches_opt.setChecked(True)
                self.saved_searches_widget.populate(settings['saved_searches'])   
                self.saved_searches_widget.where_box.where = settings['where']

    def save_settings(self):
        settings = {}
        if self.search_replace_opt.isChecked():
            settings['opt'] = 'search_replace'
            settings['searches'] = [self.search_widget.state]
        elif self.saved_searches_opt.isChecked():
            settings['opt'] = 'saved_searches'
            settings['saved_searches'] = self.saved_searches_widget.get_saved_searches()
            settings['where'] = self.saved_searches_widget.where_box.where
        return settings

#==================
# Editor Action
#==================

class SearchReplace(EditorAction):

    name = 'Search & Replace'
    _is_builtin_ = True
    headless = True

    def get_searches(self, settings):
        opt = settings.get('opt', 'search_replace')
        if opt == 'search_replace':
            searches = settings['searches']
        elif opt == 'saved_searches':
            where = settings['where']
            saved_searches_names = settings['saved_searches']
            searches = [s for s in tprefs['saved_searches'] if s['name'] in saved_searches_names]
            searches = [search_from_saved_search(s, where=where) for s in searches]
        return searches

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        searches = self.get_searches(settings)
        where = searches[0].get('where', 'text')
        # only used for marked text, otherwise set to None to enable batch processing
        # from action chains
        current_editor = None
        if chain.gui and (where == 'selected-text'):
            current_editor = chain.gui.central.current_editor
        log = []
        raw_data = {}
        all_updates = set()
        for search in searches:
            #where = search['where']
            search_name = get_search_name(search)
            filenames = scope_names(chain, where)
            updates, raw_data = replace_all(filenames, container, process_searches(search), current_editor, raw_data=raw_data)
            all_updates |= updates
        if current_editor is None:
            write_all(container, all_updates, raw_data)
        else:
            name = editor_name(current_editor)
            chain.boss.commit_editor_to_container(name, container)

    def validate(self, settings):
        if not settings:
            return _('Settings Error'), _('You must configure this action')
        searches = self.get_searches(settings)
        if settings.get('opt') == 'saved_searches':
            all_saved_searches = [s['name'] for s in tprefs['saved_searches']]
            saved_searches = settings['saved_searches']
            if not saved_searches:
                return _('No saved searches'), _('You must select at least one saved search')
            for s in saved_searches:
                if s not in all_saved_searches:
                    return _('Saved search unavailable'), _(f'Cannot find saved search: {s}')

        scope = searches[0].get('where', 'text')
        scope_ok = validate_scope(scope)
        if scope_ok is not True:
            return scope_ok
        for search in searches:
            s = process_searches(search, validate=True)
            if s is not True:
                return s
            if not search.get('find'):
                return _('Missing find'), _(f'You must specify find term. (Search name: {get_search_name(search)})')
            #if not search.get('replace'):
                #return _('Missing replace'), _('You must specify replace term/function. (Search name: {})'.format(get_search_name(search)))
        return True

    def config_widget(self):
        return ConfigWidget

    def is_headless(self, settings):
        searches = self.get_searches(settings)
        for search in searches:
            srh = scope_is_headless(search['where'])
            if srh is not True:
                return srh
        return True
