#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2023, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
import sys
import subprocess
import tempfile
import shutil
import shlex
import traceback

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout,
                     QGroupBox, QCheckBox, QLabel, QLineEdit, QToolButton)

from calibre import prints
from calibre.constants import iswindows, isosx, islinux, DEBUG
from calibre.gui2 import error_dialog, choose_files
from calibre.ebooks.oeb.polish.container import guess_type
from calibre.ebooks.oeb.polish.replace import rename_files, replace_file
from calibre.ptempfile import PersistentTemporaryDirectory, better_mktemp, remove_dir

from calibre_plugins.editor_chains.actions.base import EditorAction
from calibre_plugins.editor_chains.scope import scope_names, ScopeWidget, validate_scope, scope_is_headless
from calibre_plugins.editor_chains.common_utils import DragDropComboBox, get_icon, get_file_path


try:
    load_translations()
except NameError:
    prints("EditorChains::actions/run_command.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        scope_groupbox = QGroupBox(_('Files to run command on'))
        l.addWidget(scope_groupbox)
        scope_l = QVBoxLayout()
        scope_groupbox.setLayout(scope_l)
        self.where_box = ScopeWidget(self, self.plugin_action, orientation='vertical',
                                     headless=self.plugin_action.gui is None)
        scope_l.addWidget(self.where_box)


        self.binary_box = QGroupBox(_('&Choose binary:'))
        l.addWidget(self.binary_box)
        binary_layout = QVBoxLayout()
        self.binary_box.setLayout(binary_layout)
        self.binary_combo = DragDropComboBox(self, drop_mode='file')
        binary_layout.addWidget(self.binary_combo)
        hl1 = QHBoxLayout()
        binary_layout.addLayout(hl1)
        hl1.addWidget(self.binary_combo, 1)
        self.choose_binary_button = QToolButton(self)
        self.choose_binary_button.setToolTip(_('Choose binary'))
        self.choose_binary_button.setIcon(get_icon('document_open.png'))
        self.choose_binary_button.clicked.connect(self._choose_file)
        hl1.addWidget(self.choose_binary_button)

        args_box = QGroupBox(_('Args: '), self)
        binary_layout.addWidget(args_box)
        args_l = QVBoxLayout()
        args_box.setLayout(args_l)
        label_args = QLabel(_(
            '<p>Enter the arguments to the binary or the command. You can enter '
            'command line switches and options here.</p>'
            '<p>There are two placeholders you can use:</p>'
            '<p><b>{inputfile}:</b> This will pass the input filename to the command to act on. '
            '<u>This is a mandatory placeholder that must be placed correctly</u>.</p>'
            '<p><b>{outputfile}:</b> This is a placeholder to use if your command need an outputfile. '
            'The action will pass an output filename to the command, and it will use the resulting '
            'outputfile to replace the input file from which it was created. '
            '<u>This placeholder should not be supplied if the command/binary act '
            'on the inputfile inplace without creating a new outputfile.</u></p>'
        ))
        label_args.setWordWrap(True)
        args_l.addWidget(label_args)
        self.args_edit = QLineEdit(self)
        args_l.addWidget(self.args_edit)

        oext_groupbox = QGroupBox(_('Output Extension'))
        oext_groupbox.setToolTip(_(
            'If the {outputfile} in your command needs an extension that is\n'
            'different from the input file extension, enter that extension here\n'
            'and it will be appended to the output file. Also the corresponding\n'
            'file in you ebook will be renamed to the new extension.\n'
            'Note: Renaming files is not supported for azw3 format, so this\n'
            'will not work on azw3 books'))
        l.addWidget(oext_groupbox)
        oext_l = QVBoxLayout()
        oext_groupbox.setLayout(oext_l)
        self.output_ext_ledit = QLineEdit()
        oext_l.addWidget(self.output_ext_ledit)

        self.output_smaller_chk = QCheckBox(_(
            'Output file must be smaller than input, otherwise skip.'))
        self.output_smaller_chk.setToolTip(_(
            'This option can be useful if the purpose of the command is\n'
            'to reduce the size of the files. If it fails to produce a file\n'
            'with a smaller size, the original file will be kept.'))
        l.addWidget(self.output_smaller_chk)

        self.print_output_chk = QCheckBox(_('Print command ouptut to shell'))
        l.addWidget(self.print_output_chk)

        self.shell_chk = QCheckBox(_('Execute command through shell'))
        self.shell_chk.setChecked(False)
        l.addWidget(self.shell_chk)

        l.addStretch(1)

        self.setMinimumSize(400,700)

    def _choose_file(self):
        files = choose_files(None, _('Select binary dialog'), _('Select a binary'),
                             all_files=True, select_only_single_file=True)
        if not files:
            return
        binary_path = files[0]
        if iswindows:
            binary_path = os.path.normpath(binary_path)

        self.block_events = True
        existing_index = self.binary_combo.findText(binary_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.binary_combo.setCurrentIndex(existing_index)
        else:
            self.binary_combo.insertItem(0, binary_path)
            self.binary_combo.setCurrentIndex(0)
        self.block_events = False

    def load_settings(self, settings):
        if settings:
            self.binary_combo.setCurrentText(settings['path_to_binary'])
            self.args_edit.setText(settings['args'])
            self.output_smaller_chk.setChecked(settings.get('output_must_be_smaller', False))
            self.print_output_chk.setChecked(settings.get('print_output', False))
            self.output_ext_ledit.setText(settings['output_ext'])
            self.where_box.where = settings['where']
            self.shell_chk.setChecked(settings.get('shell', False))

    def save_settings(self):
        settings = {}
        settings['path_to_binary'] = self.binary_combo.currentText().strip()
        settings['args'] = self.args_edit.text()
        settings['output_must_be_smaller'] = self.output_smaller_chk.isChecked()
        settings['print_output'] = self.print_output_chk.isChecked()
        settings['output_ext'] = self.output_ext_ledit.text()
        settings['where'] = self.where_box.where
        settings['shell'] = self.shell_chk.isChecked()
        return settings

class RunCommand(EditorAction):

    name = 'Run Command'
    _is_builtin_ = True
    headless = True

    def run(self, chain, settings, *args, **kwargs):
        container = chain.current_container
        filenames = scope_names(chain, settings['where'], include_nonsearchable=True)
        tdir = PersistentTemporaryDirectory('_ec_run_command')
        path_to_binary = settings['path_to_binary']
        shell = settings.get('shell', False)
        clean_env = dict(os.environ)
        kw = {'env': clean_env, 'shell': shell, 'capture_output': True}
        if iswindows:
            del clean_env['PATH']
        elif isosx:
            if path_to_binary.lower().endswith(".app"):
                path_to_binary = 'open -a ' + path_to_binary
        else: #Linux
            clean_env['LD_LIBRARY_PATH'] = ''
        for name in filenames:
            data = container.raw_data(name, decode=False)
            dirname = os.path.dirname(name)
            basename = os.path.basename(name)
            base, sep, ext = basename.lower().rpartition('.')
            output_ext = settings.get('output_ext', '')
            suffix = '.' + output_ext if output_ext else '.' + ext
            inputfile = better_mktemp(dir=tdir, suffix='.' + ext)
            if settings['args'].find('{outputfile}') == -1:
                outputfile = inputfile
            else:
                outputfile = tempfile.mktemp(dir=tdir, suffix=suffix)
            if iswindows:
                inputfile = os.path.normpath(inputfile)
                outputfile = os.path.normpath(outputfile)
            args = settings['args']
            path_to_binary = os.path.expandvars(path_to_binary)
            if shell is True:
                if ' ' in path_to_binary:
                    path_to_binary = shlex.quote(path_to_binary)
                args = args.format(inputfile=inputfile, outputfile=outputfile)
                cmd = f'{path_to_binary} {args}'
            else:
                cmd = [path_to_binary]
                if args:
                    cmd += shlex.split(args)
                try:
                    index = cmd.index('{inputfile}')
                    cmd = cmd[0:index] + [inputfile] + cmd[index+1:]
                except ValueError:
                    traceback.print_exc()
                try:
                    index = cmd.index('{outputfile}')
                    cmd = cmd[0:index] + [outputfile] + cmd[index+1:]
                except ValueError:
                    pass
            with open(inputfile, 'wb') as f:
                f.write(data)
            # Run command
            result = subprocess.run(cmd, **kw)
            if result.returncode != 0:
                prints(f'Command error: Return code is {result.returncode} when running command for file: {name}\ncmd: {cmd}')
                sys.stdout.buffer.write(b'stderr: ' + result.stderr)
                prints()
                continue
            else:
                prints(f'Command ran successfully for file: {name}')
                if settings.get('print_output', False):
                    sys.stdout.buffer.write(result.stdout)
                    prints()
            if not os.access(outputfile, os.R_OK):
                prints(f'Editor Chains: Run Command: Error: no output files resulting from running command on inputfile: {inputfile}')
                continue
            output_filesize = os.stat(outputfile).st_size
            if output_filesize == 0:
                prints(f'Editor Chains: Run Command: converted filesize for file: {name} is zero. Skipping')
                continue
            if settings.get('output_must_be_smaller', False) and ( output_filesize > container.filesize(name) ):
                prints(f'Output filesize for file: {name} is larger than input filesize. Skipping.')
                continue
            with open(outputfile, 'rb') as f:
                new_data = f.read()
                container.open(name ,'wb').write(new_data)
                # Rename file in container if outputfile has a different extension
                if output_ext and ( guess_type(name) != guess_type(outputfile) ):
                    new_name = f'{base}.{output_ext}'
                    if dirname:
                        new_name = f'{dirname}/{base}.{output_ext}'
                    new_name = container.make_name_unique(new_name)
                    name_map = {name:new_name}
                    rename_files(container, name_map)
                    try:
                        container.mime_map[new_name] = container.guess_type(new_name)
                        for itemid, q in container.manifest_id_map.items():
                            if q == new_name:
                                for item in container.opf_xpath('//opf:manifest/opf:item[@href and @id="%s"]' % itemid):
                                    item.set('media-type', container.mime_map[new_name])
                    except:
                        print(f'Error changing media_type for {new_name}')
                        traceback.print_exc()
        remove_dir(tdir)

    def validate(self, settings):
        if not settings:
            return (
                _('Settings Error'),
                _('You must configure this action before running it')
            )
        path_to_binary = settings['path_to_binary']
        if not path_to_binary:
            return (_('No Command'), _('You must specify a command to run'))
        if os.path.sep in path_to_binary:
            if not shutil.which(path_to_binary):
                return (_('Command not found'), _(f'Command {path_to_binary} is found on your system'))
        if settings['args'].find('{inputfile}') == -1:
            return _('Setting error'), _('Args box must contain the {inputfile} placeholder')
        scope_ok = validate_scope(settings['where'])
        if scope_ok is not True:
            return scope_ok
        return True

    def config_widget(self):
        return ConfigWidget

    def is_headless(self, settings):
        return scope_is_headless(settings['where'])

    def supported_formats(self, settings):
        if settings.get('output_ext', ''):
            return ['epub']
        return ['epub','azw3']
